import contextlib
import gevent
import logging
import platform
import random
import time
import traceback

from .component import Component
from .lock import AdvisoryLock
from .utils import LogDbObj


class JobManager(Component):
    def __init__(self, db, dbt, parent=None):
        self.jobs = {}  # name => {meth, active_greenlet, run_ts}
        self.db = dbt  # !!! using transactional db here, we will not even forward db to self.db
        self.node = platform.node()
        super(JobManager, self).__init__(parent=parent)
        self.log = LogDbObj(db, logging.getLogger('jobmngr'), 'jobmngr')

    def start(self):
        with self.lock(wait=600) as locked:
            if locked:
                missing_jobs = set(self.jobs.keys())
                for name in self.db.query_col('SELECT name FROM job'):
                    missing_jobs.discard(name)

                if missing_jobs:
                    self.log.info('Jobs %r missing in db, creating them', missing_jobs)
                    for name in missing_jobs:
                        self.db.execute('INSERT INTO job (name, next_run_ts) VALUES (%s, %s)', (name, 0))

        return super(JobManager, self).start()

    def register_job(self, name, meth, fail_interval, success_interval):
        assert name not in self.jobs

        self.jobs[name] = {
            'meth': meth,
            'fail_interval': fail_interval,
            'success_interval': success_interval,
            'active': None,
            'active_ts': None
        }

    @contextlib.contextmanager
    def lock(self, wait=0):
        deadline = time.time() + wait

        while True:
            if wait > 0 and time.time() > deadline:
                # We are in waiting mode and deadline already passed
                yield False
                break

            with AdvisoryLock(self.db, 'job_mngr') as locked:
                if locked:
                    yield True
                    break
                elif wait > 0:
                    # Wait requested, continue to next loop
                    continue
                else:
                    # Non-waiting mode -- just yield False right now
                    yield False
                    break

    def _run_job(self, name):
        jobinfo = self.jobs[name]
        jobinfo['active_ts'] = int(time.time())

        try:
            meth = jobinfo['meth']
            ts = time.time()
            next_run_in = meth()
            te = time.time()
        except:
            failed = True
            self.log.warning('Job "%s" failed', name)
            self.log.warning(traceback.format_exc())
            next_run_in = jobinfo['fail_interval']
        else:
            failed = False
            self.log.info('Job "%s" finished success in %0.4fs', name, te - ts)
            if next_run_in is None:
                next_run_in = jobinfo['success_interval']

        with self.lock(wait=3600) as locked:
            if not locked:
                return

            now = int(time.time())

            if not failed:
                extra = 'cnt_success = cnt_success + 1'
            else:
                extra = 'cnt_failed = cnt_failed + 1'

            self.db.execute(
                'UPDATE job '
                'SET node = null, run_ts = null, last_node = %s, last_run_ts = %s, next_run_ts = %s, ' + extra + ' '
                'WHERE name = %s',
                (self.node, now, now + next_run_in, name)
            )
            self.db.commit()

        jobinfo['active_ts'] = jobinfo['active'] = None

        if self.scheduler.waiting:
            self.scheduler.wakeup()

    debug_force_run_evoq_planner = False
    debug_force_run_evoq_ticker = False
    debug_force_run_vmsync = False
    debug_force_run_hotbackup_scheduler = False

    @Component.green_loop
    def scheduler(self, log, wokeup, waited):
        log.info('Scheduler run, wokeup: %s, waited: %ds', wokeup, waited)

        with self.lock() as locked:
            if not locked:
                return random.randint(30, 60)

            if not wokeup:
                job_info = self.db.query_one(
                    'SELECT name, node, run_ts FROM job '
                    'WHERE ('
                    '   ('
                    '      node IS NULL AND ('
                    '         ('
                    '             last_node = %s AND next_run_ts <= %s'
                    '         ) OR ('
                    '             next_run_ts <= %s'
                    '         )'
                    '      )'
                    '   ) OR ('
                    '      node IS NOT NULL AND ('
                    '          run_ts < %s'
                    '      )'
                    '   )'
                    ') AND name IN (%S)'
                    'LIMIT 1',
                    (
                        self.node, int(time.time()),      # re-acquire job on same host
                        int(time.time() - 900),           # grab other host's job only after 15 min of next run+
                        int(time.time() - 3600),          # grab stalld jobs
                        list(self.jobs)
                    )
                )

                if job_info:
                    job, job_node, job_run_ts = job_info
                else:
                    job = job_node = job_run_ts = None

                if self.debug_force_run_evoq_planner:
                    job, job_node, job_run_ts = ('evoq_planner', None, None)
                    self.debug_force_run_evoq_planner = False

                elif self.debug_force_run_evoq_ticker:
                    job, job_node, job_run_ts = ('evoq_ticker', None, None)
                    self.debug_force_run_evoq_ticker = False

                elif self.debug_force_run_vmsync:
                    job, job_node, job_run_ts = ('vmsync', None, None)
                    self.debug_force_run_vmsync = False

                elif self.debug_force_run_hotbackup_scheduler:
                    job, job_node, job_run_ts = ('hotbackup_scheduler', None, None)
                    self.debug_force_run_hotbackup_scheduler = False

                if job:
                    log.debug('Will run job: %s', job)
                    if job_node and job_node != self.node:
                        log.debug('  reacquired from %s (we are %s)', job_node, self.node)
                        log.debug('  run ts is %d secs ago', time.time() - job_run_ts)

                    self.db.execute(
                        'UPDATE job SET node = %s, run_ts = %s WHERE name = %s',
                        (self.node, int(time.time()), job)
                    )

                    self.db.commit()
                    self.jobs[job]['active'] = gevent.spawn(self._run_job, job)
                else:
                    log.debug('No jobs to run was found')

            min_ts_our = self.db.query_one_col(
                'SELECT MIN(next_run_ts) FROM job WHERE node IS NULL AND last_node = %s', (self.node, )
            )
            min_ts_all = self.db.query_one_col(
                'SELECT MIN(next_run_ts) FROM job WHERE node IS NULL AND last_node != %s', (self.node, ),
            )

            if min_ts_all:
                min_ts_all += 900

            if min_ts_our is not None and (min_ts_all is None or min_ts_our < min_ts_all):
                min_ts = min_ts_our
            elif min_ts_all is not None:
                min_ts = min_ts_all
            else:
                min_ts = time.time() + 60  # no jobs to be scheduled right now

            interval = int(min_ts - time.time())

            # Cap interval 10s <= detected <= 60m
            interval = max(10, interval)
            interval = min(3600, interval)

            interval = interval + random.randint(0, 60)

        if self.debug_force_run_evoq_planner or self.debug_force_run_evoq_ticker:
            interval = 1

        log.debug('Next loop run in %ds', interval)
        return interval
