import hashlib
import logging
import msgpack
import platform
import time
import uuid

from . import tvm


CLUSTER_NAMES = {
    'vla', 'man', 'sas', 'myt', 'iva', 'test_sas', 'man_pre',
    'dev'  # used in development only, fake one
}


class Session(object):
    def __init__(self, db):
        self.db = db
        self.log = logging.getLogger('session')

        self.key = None
        self.type = None
        self.state = None
        self.state_ts = None
        self.modify_ts = None
        self.origin = None

        self.vm_id = None
        self.rev_id = None
        self.node_id = None

        self.stats = {
            'bytes_total': None,
            'bytes_done': None,
            'speed_bps': None
        }

        self.mds_tvm_ticket = None
        self.mds_tvm_ticket_ts = None

        self._new = False
        self._dirty = False

    def load(self, key):
        row = self.db.query_one(
            'SELECT '
            '   key, type, state, state_ts, modify_ts, origin, '
            '   vm_id, rev_id, run_vm_id, run_node_id, '
            '   bytes_total, bytes_done, speed_bps, '
            '   mds_tvm_ticket, mds_tvm_ticket_ts '
            'FROM session '
            'WHERE key = %s',
            (key, ),
        )

        if not row:
            return

        assert row[0] == key

        self.key = row[0]
        self.type = row[1]
        self.state = row[2]
        self.state_ts = row[3]
        self.modify_ts = row[4]
        self.origin = row[5]
        self.vm_id = row[6]
        self.rev_id = row[7]
        self.run_vm_id = row[8]
        self.run_node_id = row[9]
        self.stats['bytes_total'] = row[10]
        self.stats['bytes_done'] = row[11]
        self.stats['speed_bps'] = row[12]
        self.mds_tvm_ticket = row[13]
        self.mds_tvm_ticket_ts = row[14]

        self._new = False

        return True  # indicates session was loaded

    def generate(self, typ, origin, key=None):
        if key is None:
            assert self.key is None
            self.key = hashlib.blake2b(uuid.uuid4().bytes, digest_size=32).hexdigest()
        else:
            self.key = key

        self.type = typ  # download|upload
        self.state = 'new'
        self.state_ts = self.modify_ts = int(time.time())

        assert origin in ('qdm', 'evoq', 'hot')
        self.origin = origin

        self.vm_id = None
        self.rev_id = None

        self.run_vm_id = None
        self.run_node_id = None

        self._new = True
        self._dirty = True

    def save(self):
        if self._new:
            self.db.execute(
                'INSERT INTO session ('
                '   key, type, state, state_ts, modify_ts, origin, '
                '   vm_id, rev_id, run_vm_id, run_node_id, '
                '   bytes_total, bytes_done, speed_bps, '
                '   mds_tvm_ticket, mds_tvm_ticket_ts'
                ') VALUES (%S)',
                (
                    (
                        self.key, self.type, self.state, self.state_ts, self.modify_ts, self.origin,
                        self.vm_id, self.rev_id, self.run_vm_id, self.run_node_id,
                        self.stats['bytes_total'], self.stats['bytes_done'], self.stats['speed_bps'],
                        self.mds_tvm_ticket, self.mds_tvm_ticket_ts
                    ),
                )
            )
        else:
            if self._dirty:
                self.db.execute(
                    'UPDATE session SET ('
                    '   type, state, state_ts, modify_ts, origin, '
                    '   vm_id, rev_id, run_vm_id, run_node_id, '
                    '   bytes_total, bytes_done, speed_bps, '
                    '   mds_tvm_ticket, mds_tvm_ticket_ts'
                    ') = (%S) '
                    'WHERE key = %s',
                    (
                        (
                            self.type, self.state, self.state_ts, self.modify_ts, self.origin,
                            self.vm_id, self.rev_id, self.run_vm_id, self.run_node_id,
                            self.stats['bytes_total'], self.stats['bytes_done'], self.stats['speed_bps'],
                            self.mds_tvm_ticket, self.mds_tvm_ticket_ts
                        ),
                        self.key
                    )
                )

        self._new = False
        self._dirty = False

    def get_mds_tvm_ticket(self, secret):
        if self.mds_tvm_ticket is None or time.time() - self.mds_tvm_ticket_ts > 3600:
            self.mds_tvm_ticket = tvm.generate_mds_ticket(secret)
            self.mds_tvm_ticket_ts = int(time.time())
            self.touch()
            self.save()

        return self.mds_tvm_ticket

    def touch(self):
        self.modify_ts = int(time.time())
        self._dirty = True

    def add_audit(self, severity, message):
        audit = Audit(self.db, self.vm_id, self.key, int(time.time()), severity, message)
        audit.save()
        self.touch()

    def set_state(self, state):
        if self.state != state:
            self.state = state
            self.state_ts = int(time.time())
            self._dirty = True

            self.touch()

    def archive(self):
        self.set_state('archive')
        self.mds_tvm_ticket = None
        self.mds_tvm_ticket_ts = None
        self._dirty = True


class StorageBlock(object):
    def __init__(self, db):
        self.db = db
        self.log = logging.getLogger('storageblock')

        self.id = None
        self.hashtype = None
        self.hash = None
        self.size = None
        self.mds_storage = None
        self.mds_key = None
        self.mds_ttl = None

        self._new = False

    @classmethod
    def new(cls, db, size, hashtype, hash_, mds_storage, mds_key, mds_ttl):
        self = cls(db)
        self.hashtype = hashtype
        self.hash = hash_
        self.size = size
        self.mds_storage = mds_storage
        self.mds_key = mds_key
        self.mds_ttl = mds_ttl

        self._new = True

        return self

    def from_dbrow(self, row):
        assert len(row) == 7

        self.id = row[0]
        self.hashtype = row[1]
        self.hash = row[2]
        self.size = row[3]
        self.mds_storage = row[4]
        self.mds_key = row[5]
        self.mds_ttl = row[6]

        self._new = False

    def dbdict(self):
        dct = self.__dict__.copy()

        drop_keys = set()

        for key in dct.keys():
            if key in ('db', 'log') or key.startswith('_'):
                drop_keys.add(key)

        for key in drop_keys:
            dct.pop(key)

        assert len(dct) == 7, 'Invalid dbdict size %d' % (len(dct), )

        return dct

    def save(self):
        if self._new:
            self.id = self.db.query_one_col(
                'INSERT INTO storage_block ('
                '   size, hashtype, hash, '
                '   mds_storage, mds_key, mds_ttl'
                ') VALUES (%S) '
                'RETURNING id',
                (
                    (
                        self.size, self.hashtype, self.hash,
                        self.mds_storage, self.mds_key, self.mds_ttl
                    ),
                )
            )
        else:
            self.db.execute(
                'UPDATE storage_block SET ('
                '   size, hashtype, hash, '
                '   mds_storage, mds_key, mds_ttl'
                ') = (%S) '
                'WHERE id = %s',
                (
                    (
                        self.key, self.type, self.state, self.state_ts, self.modify_ts,
                        self.vm_id, self.rev_id, self.node_id,
                        self.stats['bytes_total'], self.stats['bytes_done'], self.stats['speed_bps'],
                        self.mds_tvm_ticket, self.mds_tvm_ticket_ts
                    ),
                    self.id
                )
            )

        self._new = False


class StorageData(object):
    def __init__(self, db):
        self.db = db
        self.log = logging.getLogger('storagedata')

        self.vm_id = None
        self.rev_id = None
        self.idx = None
        self.block_id = None

        self._new = False

    @classmethod
    def new(cls, db, vm_id, rev_id, idx, block_id):
        self = cls(db)

        self.vm_id = vm_id
        self.rev_id = rev_id
        self.idx = idx
        self.block_id = block_id

        self._new = True

        return self

    def from_dbrow(self, row):
        assert len(row) == 4

        self.vm_id = row[0]
        self.rev_id = row[1]
        self.idx = row[2]
        self.block_id = row[3]

        self._new = False

    def save(self):
        if self._new:
            self.db.execute(
                'INSERT INTO storage_data ('
                '   rev_id, vm_id, idx, block_id'
                ') VALUES (%S)',
                (
                    (
                        self.rev_id, self.vm_id, self.idx, self.block_id
                    ),
                )
            )
        else:
            raise NotImplementedError('We are unable to update storage data!')

        self._new = False


class StorageRevision(object):
    def __init__(self, db):
        self.db = db
        self.log = logging.getLogger('storagerev')

        self.vm_id = None
        self.rev_id = None
        self.key = None
        self.state = None
        self.origin = None
        self.create_ts = None
        self.access_ts = None
        self.access_cnt = None
        self.filemap = None
        self.vmspec = None

        self._new = False

        self._data = []

    @classmethod
    def new(cls, db, vm_id, rev_id, origin):
        self = cls(db)

        self.vm_id = vm_id
        self.rev_id = rev_id
        self.origin = origin

        self.state = 'draft'
        self.key = hashlib.sha256(uuid.uuid4().bytes).hexdigest()
        self.create_ts = int(time.time())
        self.access_cnt = 0

        self._new = True

        return self

    def load_dbrow(self, row):
        assert len(row) == 10

        self.vm_id = row[0]
        self.rev_id = row[1]
        self.key = row[2]
        self.state = row[3]
        self.origin = row[4]
        self.create_ts = row[5]
        self.access_ts = row[6]
        self.access_cnt = row[7]

        if row[8] is not None:
            self.filemap = msgpack.loads(row[8])
        else:
            self.filemap = None

        if row[9] is not None:
            self.vmspec = msgpack.loads(row[9])
        else:
            self.vmspec = None

    def dbdict(self):
        dct = self.__dict__.copy()

        drop_keys = set()

        for key in dct.keys():
            if key in ('db', 'log') or key.startswith('_'):
                drop_keys.add(key)

        for key in drop_keys:
            dct.pop(key)

        assert len(dct) == 10, 'Invalid dbdict size %d' % (len(dct), )

        return dct

    def load(self, key):
        row = self.db.query_one(
            'SELECT '
            '   vm_id, rev_id, key, state, origin, '
            '   create_ts, access_ts, access_cnt, '
            '   filemap, vmspec '
            'FROM storage_revision '
            'WHERE key = %s',
            (key, )
        )

        if not row:
            return

        assert row[2] == key

        self.load_dbrow(row)

        return True

    def search(self, vm_id, rev_id):
        row = self.db.query_one(
            'SELECT '
            '   vm_id, rev_id, key, state, origin, '
            '   create_ts, access_ts, access_cnt, '
            '   filemap, vmspec '
            'FROM storage_revision '
            'WHERE vm_id = %s AND rev_id = %s',
            (vm_id, rev_id)
        )

        if not row:
            return

        assert row[0] == vm_id
        assert row[1] == rev_id

        self.load_dbrow(row)

        return True

    @classmethod
    def search_state(cls, db, vm_id, state='active'):
        ret = []

        for row in db.query(
            'SELECT '
            '   vm_id, rev_id, key, state, origin, '
            '   create_ts, access_ts, access_cnt, '
            '   filemap, vmspec '
            'FROM storage_revision WHERE vm_id = %s and state = %s '
            'ORDER BY rev_id ASC',
            (
                (vm_id, state)
            )
        ):
            assert row[0] == vm_id

            rev = cls(db)
            rev.load_dbrow(row)

            ret.append(rev)

        return ret

    @classmethod
    def select_owners_revs(cls, db, user, groups):
        ret = []

        rows = db.query(
            'SELECT DISTINCT '
            '   sr.vm_id, sr.rev_id, sr.key, sr.state, sr.origin, '
            '   sr.create_ts, sr.access_ts, sr.access_cnt, sr.filemap, sr.vmspec '
            'FROM storage_revision sr '
            'LEFT JOIN revision_user ru USING(vm_id, rev_id) '
            'LEFT JOIN revision_group rg USING(vm_id, rev_id) '
            'WHERE '
            '   (ru.user_id = %s OR rg.group_id IN (%S)) AND sr.state IN (%s, %s) '
            'ORDER BY sr.vm_id, sr.create_ts DESC',
            (user, groups, 'draft', 'active')
        )

        for row in rows:
            rev = cls(db)
            rev.load_dbrow(row)
            ret.append(rev)
        return ret

    def load_data(self):
        pos = 0

        for row in self.db.query(
            'SELECT '
            '   sd.idx, sd.block_id, sb.hashtype, sb.hash, sb.size, '
            '   sb.mds_storage, sb.mds_key, sb.mds_ttl '
            'FROM storage_data sd '
            'JOIN storage_block sb ON sb.id = sd.block_id '
            'WHERE sd.vm_id = %s AND sd.rev_id = %s '
            'ORDER BY sd.idx ASC',
            (self.vm_id, self.rev_id)
        ):
            idx, block_id, hashtype, hash_, size, mds_storage, mds_key, mds_ttl = row
            assert idx == pos, 'Some blocks missing in db'

            pos += 1

            block = StorageBlock(self.db)
            block.from_dbrow(row[1:])

            self._data.append(block)

        return self._data

    def set_state(self, state):
        assert state in ('draft', 'active', 'archive')
        self.state = state

    def archive(self):
        self.set_state('archive')

    def activate(self):
        self.set_state('active')
        self.create_ts = int(time.time())

    def save(self):
        if self._new:
            self.db.execute(
                'INSERT INTO storage_revision ('
                '   vm_id, rev_id, key, state, origin, '
                '   create_ts, access_ts, access_cnt, '
                '   filemap, vmspec'
                ') VALUES (%S)',
                (
                    (
                        self.vm_id, self.rev_id, self.key, self.state, self.origin,
                        self.create_ts, self.access_ts, self.access_cnt,
                        msgpack.dumps(self.filemap, use_bin_type=True) if self.filemap is not None else self.filemap,
                        msgpack.dumps(self.vmspec, use_bin_type=True) if self.vmspec is not None else self.vmspec
                    ),
                )
            )
        else:
            self.db.execute(
                'UPDATE storage_revision SET ('
                '   key, state, origin, '
                '   create_ts, access_ts, access_cnt, '
                '   filemap, vmspec'
                ') = (%S) '
                'WHERE vm_id = %s AND rev_id = %s',
                (
                    (
                        self.key, self.state, self.origin,
                        self.create_ts, self.access_ts, self.access_cnt,
                        msgpack.dumps(self.filemap, use_bin_type=True) if self.filemap is not None else self.filemap,
                        msgpack.dumps(self.vmspec, use_bin_type=True) if self.vmspec is not None else self.vmspec

                    ),
                    self.vm_id,
                    self.rev_id
                )
            )

        self._new = False

    def store_block(self, hashtype, hash_, size, mds_key, ttl):
        block = StorageBlock.new(
            self.db, size, hashtype, hash_,
            'storage-int-mds', mds_key, int(time.time() + ttl)
        )

        block.save()
        return block

    def store_data(self, idx, block):
        data = StorageData.new(
            self.db, self.vm_id, self.rev_id, idx, block.id
        )
        data.save()
        return data

    def create_owners_relations(self):
        if not self.vmspec:
            return

        owners = self.vmspec['meta']['auth']['owners']
        for user_id in owners.get('logins', []):
            RevisionUser(self.db, self.vm_id, self.rev_id, user_id).save()
        for group_id in owners.get('group_ids', []):
            RevisionGroup(self.db, self.vm_id, self.rev_id, group_id).save()


class RevisionUser(object):
    def __init__(self, db, vm_id, rev_id, user_id):
        self.db = db
        self.vm_id = vm_id
        self.rev_id = rev_id
        self.user_id = user_id
        self._new = True

    def save(self):
        if self._new:
            self.db.execute(
                'INSERT INTO revision_user ('
                '   vm_id, rev_id, user_id'
                ') VALUES (%S)',
                (
                    (
                        self.vm_id, self.rev_id, self.user_id
                    ),
                )
            )
        else:
            raise NotImplementedError()
        self._new = False


class RevisionGroup(object):
    def __init__(self, db, vm_id, rev_id, group_id):
        self.db = db
        self.vm_id = vm_id
        self.rev_id = rev_id
        self.group_id = group_id
        self._new = True

    def save(self):
        if self._new:
            self.db.execute(
                'INSERT INTO revision_group ('
                '   vm_id, rev_id, group_id'
                ') VALUES (%S)',
                (
                    (
                        self.vm_id, self.rev_id, self.group_id
                    ),
                )
            )
        else:
            raise NotImplementedError()
        self._new = False


class StorageRevisionCollection(object):
    MAX_ACTIVE_REVS = 3

    def __init__(self, db, vm_id):
        self.log = logging.getLogger('storagerevcol')

        self.db = db
        self.vm_id = vm_id

        self.revs = {}
        self._blocks_by_hash_type_size = {}

    def load(self):
        for row in self.db.query(
            'SELECT '
            '   rev_id, key, state, origin, '
            '   create_ts, access_ts, access_cnt, '
            '   filemap, vmspec '
            'FROM storage_revision '
            'WHERE vm_id = %s AND state = %s',
            (
                self.vm_id, 'active'
            )
        ):
            rev = StorageRevision(self.db)
            rev.load_dbrow(
                [self.vm_id] + list(row)
            )
            self.revs[rev.rev_id] = rev

        if self.revs:
            datas = []

            for row in self.db.query(
                'SELECT '
                '   rev_id, idx, block_id '
                'FROM storage_data '
                'WHERE vm_id = %s AND rev_id IN (%S)',
                (
                    self.vm_id,
                    [_.rev_id for _ in self.revs.values()]
                )
            ):
                data = StorageData(self.db)
                data.from_dbrow([self.vm_id] + list(row))
                datas.append(data)

            if datas:
                blocks = {}

                for row in self.db.query(
                    'SELECT '
                    '   id, hashtype, hash, size, '
                    '   mds_storage, mds_key, mds_ttl '
                    'FROM storage_block '
                    'WHERE id in (%S)',
                    (
                        [_.block_id for _ in datas],
                    )
                ):
                    block = StorageBlock(self.db)
                    block.from_dbrow(row)
                    blocks[block.id] = block

                    if 'qdmblock:1' not in block.mds_key:
                        # Ignore blocks v0 and do not put into blacks_by_hash_type_size dict, which is used
                        # only to decide need we upload new block or not
                        continue

                    hashtypesize_idx = block.hashtype, block.hash, block.size

                    if hashtypesize_idx not in self._blocks_by_hash_type_size:
                        self._blocks_by_hash_type_size[hashtypesize_idx] = block
                    else:
                        self.log.warning('double block found in vm %r: %r', self.vm_id, hashtypesize_idx)

    def find_block(self, hashtype, hash_, size):
        return self._blocks_by_hash_type_size.get((hashtype, hash_, size), None)

    def get_client_metadata(self):
        metadata = {
            'qdm_metadata_version': 1,
            'data': [],
            'hashmap': {}
        }
        return metadata

    def get_next_rev_id(self):
        max_rev_id = self.db.query_one_col(
            'SELECT MAX(rev_id) FROM storage_revision WHERE vm_id = %s',
            (self.vm_id, )
        )
        if max_rev_id is not None:
            return max_rev_id + 1
        return 1

    def add_revision(self, rev):
        assert rev.rev_id not in self.revs
        assert rev.vm_id == self.vm_id
        self.revs[rev.rev_id] = rev

    def archive_old_revisions(self):
        archived_count = self.db.execute(
            'UPDATE storage_revision '
            'SET state = %s '
            'WHERE '
            '   vm_id = %s AND '
            '   state = %s AND '
            '   rev_id NOT IN ( '
            '       SELECT rev_id '
            '       FROM storage_revision '
            '       WHERE '
            '           vm_id = %s AND '
            '           state = %s '
            '       ORDER BY rev_id DESC LIMIT %s '
            '   ) ',
            (
                'archive', self.vm_id, 'active',
                self.vm_id, 'active', self.MAX_ACTIVE_REVS
            )
        ).rowcount

        return archived_count


class Audit(object):
    def __init__(self, db, vm_id, session, timestamp, severity, message):
        self.db = db

        self.vm_id = vm_id
        self.session = session
        self.timestamp = timestamp
        self.severity = severity
        self.message = message

    def save(self):
        self.db.execute(
            'INSERT INTO audit ('
            '   vm_id, session, timestamp, severity, message'
            ') VALUES (%S)',
            (
                (
                    self.vm_id, self.session, self.timestamp, self.severity, self.message
                ),
            )
        )


class Log(object):
    def __init__(self, db, name, timestamp, severity, message, args=None):
        if args:
            message = message % args

        assert severity in ('debug', 'info', 'warning', 'error')

        self.node = platform.node()
        self.name = name
        self.db = db
        self.timestamp_ms = int(timestamp * 1000)
        self.severity = severity
        self.message = message

    def save(self):
        self.db.execute(
            'INSERT INTO log ('
            '   node, name, timestamp_ms, severity, message'
            ') VALUES (%S)',
            (
                (
                    self.node, self.name, self.timestamp_ms, self.severity, self.message
                ),
            ),
            log=False
        )


class Evoq(object):
    def __init__(self, db, vm_id, node_id):
        self.db = db

        self.id = None
        self.vm_id = vm_id
        self.node_id = node_id
        self.session_key = None
        self.state = 'init'
        self.extra = {}

        self.active = None
        self.init_ts = None

        self.run_node = None
        self.run_cnt = None
        self.run_ts = None
        self.run_duration = None
        self.info = ''

        self.fail_cnt = None

        self._new = True

    def load_dbrow(self, row):
        assert len(row) == 14

        self.id = row[0]
        self.vm_id = row[1]
        self.node_id = row[2]
        self.session_key = row[3]
        self.state = row[4]

        if row[5] is not None:
            self.extra = msgpack.loads(row[5])
        else:
            self.extra = {}

        self.active = row[6]
        self.init_ts = row[7]

        self.run_node = row[8]
        self.run_cnt = row[9]
        self.run_ts = row[10]
        self.run_duration = row[11]
        self.info = row[12]

        self.fail_cnt = row[13]

        self._new = False

    def load(self, evoq_id):
        row = self.db.query_one(
            'SELECT '
            '   id, vm_id, node_id, session_key, state, extra, active, init_ts, '
            '   run_node, run_cnt, run_ts, run_duration, info, fail_cnt '
            'FROM evoq '
            'WHERE id = %s',
            (evoq_id, )
        )

        if not row:
            return

        assert row[0] == evoq_id

        self.load_dbrow(row)
        self._new = False

        return True

    @staticmethod
    def find(db, vm_id, node_id):
        evoq_id = db.query_one_col(
            'SELECT id FROM evoq WHERE vm_id = %s AND node_id = %s AND state NOT IN (%S)',
            (vm_id, node_id, ['done'])
        )
        return evoq_id

    @staticmethod
    def grab(db, node):
        now = int(time.time())

        records = {}

        for row in db.query(
            'SELECT '
            '   id, vm_id, node_id, session_key, state, extra, active, init_ts, '
            '   run_node, run_cnt, run_ts, run_duration, info, fail_cnt '
            'FROM evoq WHERE ('
            '   (run_node = %s) OR (run_node != %s AND run_ts < %s)'
            ') AND state NOT IN (%S) '
            'ORDER BY run_cnt ASC, init_ts ASC',
            (node, node, now - 600, ['done', 'stop', 'fail'])
        ):
            vm_id, node_id = row[1], row[2]
            evoq = Evoq(db, vm_id, node_id)
            evoq.load_dbrow(row)

            if vm_id in records:
                if records[vm_id].run_ts < evoq.run_ts:
                    records[vm_id] = evoq
            else:
                records[vm_id] = evoq

        return list(records.values())

    @staticmethod
    def grab_active(db, node, flag=True):
        records = []

        for row in db.query(
            'SELECT '
            '   id, vm_id, node_id, session_key, state, extra, active, init_ts, '
            '   run_node, run_cnt, run_ts, run_duration, info, fail_cnt '
            'FROM evoq WHERE active = %s',
            (flag, )
        ):
            vm_id, node_id = row[1], row[2]
            evoq = Evoq(db, vm_id, node_id)
            evoq.load_dbrow(row)

            records.append(evoq)

        return records

    def save(self):
        if self._new:
            self.id = self.db.query_one_col(
                'INSERT INTO evoq ('
                '   vm_id, node_id, session_key, state, extra, active, init_ts, '
                '   run_node, run_cnt, run_ts, run_duration, info, fail_cnt'
                ') VALUES (%S) '
                'RETURNING id',
                (
                    (
                        self.vm_id, self.node_id, self.session_key, self.state,
                        msgpack.dumps(self.extra, use_bin_type=True) if self.extra else None,
                        self.active, self.init_ts,
                        self.run_node, self.run_cnt, self.run_ts, self.run_duration, self.info,
                        self.fail_cnt
                    ),
                )
            )
        else:
            # We DO NOT update run_duration here!!!
            self.db.execute(
                'UPDATE evoq SET ('
                '   vm_id, node_id, session_key, state, extra, active, init_ts, '
                '   run_node, run_cnt, run_ts, info, fail_cnt'
                ') = (%S) '
                'WHERE id = %s',
                (
                    (
                        self.vm_id, self.node_id, self.session_key, self.state,
                        msgpack.dumps(self.extra, use_bin_type=True) if self.extra else None,
                        self.active, self.init_ts,
                        self.run_node, self.run_cnt, self.run_ts, self.info,
                        self.fail_cnt
                    ),
                    self.id
                )
            )
        self._new = False


class EvoqLog(object):
    def __init__(self, db, evoq_id, timestamp, severity, message, args=None):
        if args:
            message = message % args

        assert severity in ('debug', 'info', 'warning', 'error')

        self.db = db
        self.node = platform.node()
        self.evoq_id = evoq_id
        self.timestamp_ms = int(timestamp * 1000)
        self.severity = severity
        self.message = message

    def save(self):
        self.db.execute(
            'INSERT INTO evoq_log ('
            '   node, evoq_id, timestamp_ms, severity, message'
            ') VALUES (%S)',
            (
                (
                    self.node, self.evoq_id, self.timestamp_ms, self.severity, self.message
                ),
            ),
            log=False
        )


class StatWorker(object):
    def __init__(self, db):
        self.id = None
        self.start_ts = None
        self.run_ts = None
        self.run_node = None
        self.solomon_send_cnt = 0

        self._new = True

    def load_dbrow(self, row):
        self.id = row[0]
        self.start_ts = row[1]
        self.run_ts = row[2]
        self.run_node = row[3]
        self.solomon_send_cnt = row[4]

        self._new = False

    def load(self, id):
        row = self.db.query_one(
            'SELECT '
            '   id, start_ts, run_ts, run_node, solomon_send_cnt '
            'FROM stat_worker '
            'WHERE id = %s',
            (id, )
        )

        assert row[0] == id

        self.load_dbrow(row)
        self._new = False

        return True

    def save(self):
        if self._new:
            self.id = self.db.query_one_col(
                'INSERT INTO stat_worker ('
                '   start_ts, run_ts, run_node, solomon_send_cnt'
                ') VALUES (%S) '
                'RETURNING id',
                (
                    (
                        self.start_ts, self.run_ts, self.run_node, self.solomon_send_cnt
                    ),
                )
            )

        else:
            self.db.execute(
                'UPDATE stat_worker SET ('
                '   start_ts, run_ts, run_node, solomon_send_cnt'
                ') = (%S) '
                'WHERE id = %s',
                (
                    (
                        self.start_ts, self.run_ts, self.run_node, self.solomon_send_cnt
                    ),
                    self.id
                )
            )

        self._new = False


def convert_vm_id_and_cluster_to_db_vm_id(vm_id, cluster):
    assert cluster in CLUSTER_NAMES, 'Not allowed cluster name %r, use one from %r' % (cluster, CLUSTER_NAMES)
    return '%s.%s' % (vm_id, cluster)
