import logging

from .component import Component
from .model import Session, StorageRevision, StorageRevisionCollection, convert_vm_id_and_cluster_to_db_vm_id
from ..rpc.server import RPC as RPCObj, Server as RPCServer

import gevent


class RPC(Component):
    def __init__(self, parent, db, ready_ev, host, port, tvm_secret):
        super(RPC, self).__init__(parent, logname='rpc')

        self.db = db
        self.ready_ev = ready_ev
        self.tvm_secret = tvm_secret

        self.rpc = RPCObj(logging.getLogger('rpc'))
        self.server = RPCServer(
            logging.getLogger('rpc'), backlog=10, max_conns=1000,
            host=host, port=port
        )
        self.server.register_connection_handler(self.rpc.get_connection_handler())
        self._server_started = False

        self.rpc.mount(self.rpc_upload_session, typ='full', name='upload_session')
        self.rpc.mount(self.rpc_download_session, typ='full', name='download_session')

    def start(self):
        if not self._server_started:
            self.server.start()
            self._server_started = True

    def stop(self):
        super(RPC, self).stop()
        self.server.stop()
        return self

    def rpc_upload_session(self, job, key, rev_key):
        self.ready_ev.wait()

        session = Session(self.db)
        if session.load(key):
            session.set_state('active')
            session.save()
        else:
            job.log.warning('Unable to find session with key %r', key)
            raise Exception('Invalid session key')

        try:
            revs = StorageRevisionCollection(self.db, session.vm_id)
            revs.load()

            if rev_key:
                rev = StorageRevision(self.db)
                if not rev.load(rev_key):
                    raise Exception('Unable to find storage revision to continue')
            else:
                rev = StorageRevision.new(
                    self.db, session.vm_id,
                    revs.get_next_rev_id(),
                    origin=session.origin
                )
                rev.save()

            session.rev_id = rev.rev_id
            session.touch()
            session.save()

            while True:
                request = job.feedback

                if request[0] == 'init':
                    try:
                        vmspec = request[1][0]
                    except IndexError:
                        vmspec = None
                    session.add_audit('info', 'Upload initialize')
                    job.state(('init', rev.key))
                    if vmspec and rev.vmspec is None:
                        rev.vmspec = vmspec
                        rev.save()
                        rev.create_owners_relations()

                elif request[0] == 'get_tvm_ticket':
                    job.state(('tvm_ticket', session.get_mds_tvm_ticket(self.tvm_secret)))

                elif request[0] == 'get_metadata':
                    job.state(('metadata', revs.get_client_metadata()))

                elif request[0] == 'check_data_block':
                    # Check if specified block already exists and should not be uploaded
                    # to MDS anymore.
                    # False -- should be uploaded
                    # True -- it is okay, no more actions needed.
                    idx, size, hashtype, hash_ = request[1]

                    job.log.debug('checking block %s:%s (%d bytes)', hashtype, hash_, size)

                    need_to_upload = False
                    existing_block = revs.find_block(hashtype, hash_, size)

                    if not existing_block:
                        need_to_upload = True
                    else:
                        rev.store_data(idx, existing_block)
                        need_to_upload = False

                    job.state(('data_block_checked', not need_to_upload))

                elif request[0] == 'data_block_stored':
                    idx, size, hashtype, hash_, mds_key, ttl = request[1]

                    job.log.debug('storing block %s:%s (%d bytes) -- %s', hashtype, hash_, size, mds_key)

                    block = rev.store_block(hashtype, hash_, size, mds_key, ttl)
                    rev.store_data(idx, block)

                    job.state(('data_block_stored', mds_key))

                elif request[0] == 'finalize_revision':
                    filemap = request[1][0]
                    try:
                        vmspec = request[1][1]
                        job.log.warning('Using deprecated version of qdm-mds-cli')
                    except IndexError:
                        vmspec = rev.vmspec

                    job.log.debug('finalizing revision with:')
                    job.log.debug('  filemap:')
                    job.log.debug('    %r', filemap)
                    job.log.debug('  vmspec:')
                    job.log.debug('    %r', vmspec)

                    if isinstance(filemap, (list, tuple)):
                        # filemap version 0, without meta and without file sizes
                        # all files as fn + blocks count
                        # [(fn, 123), (fn2, 321)]
                        #
                        # we do not allow store such filemaps anymore, refusing backups
                        # all new qdm-mds-cli should produce new filemap v1
                        raise Exception('Please update qdm-mds-cli')

                    assert isinstance(filemap, dict), 'Unknown filemap format: %r' % (filemap, )

                    filemap_version = filemap.get('qdm_filemap_version', None)

                    assert filemap_version == 1, (
                        'Dont know how to interpret filemap version: %r' % (filemap_version, )
                    )

                    db_filemap = {'v': 1, 'f': []}

                    for info in filemap['files']:
                        db_filemap['f'].append({
                            'n': info['name'],
                            'b': info['blocks'],
                            's': info['size'],
                            'm': info['meta']
                        })

                    rev.filemap = db_filemap
                    rev.vmspec = vmspec
                    rev.activate()  # will change create_ts
                    rev.save()

                    revs.add_revision(rev)
                    archived_count = revs.archive_old_revisions()  # mark old revs as archived for easier removal

                    if archived_count > 0:
                        session.add_audit('info', 'Archived %d old revisions' % (archived_count, ))
                        job.log.info('Archived %d old revisions', archived_count)

                    job.state(('revision_done', rev.key))

                elif request[0] == 'finish':
                    session.archive()
                    session.add_audit('info', 'Upload success')
                    session.save()  # need to save here, bc of return
                    return True

                elif request[0] == 'set_progress':
                    job.log.info('progress %r', request[1])
                    session.add_audit('info', 'Upload progress %r' % (request[1], ))

                else:
                    raise Exception('Invalid request %r' % (request, ))

                session.save()

            return 42
        except (BaseException, gevent.GreenletExit) as ex:
            session.archive()
            session.save()

            if isinstance(ex, gevent.GreenletExit):
                session.add_audit('error', 'Upload not complete: connection closed (got green exit)')
            else:
                session.add_audit('error', 'Upload failed: %s: %s' % (type(ex).__name__, str(ex)))
            raise

    def rpc_download_session(self, job, rev, session_key, run_vm_id=None, run_cluster=None, run_node_id=None):
        self.ready_ev.wait()

        if run_cluster and run_vm_id:
            run_cluster = run_cluster.lower()
            run_vm_id = convert_vm_id_and_cluster_to_db_vm_id(run_vm_id, run_cluster)
        else:
            run_cluster = run_vm_id = None

        rev_key = rev

        # Strip qdm: prefix if any
        if rev_key.startswith('qdm:'):
            rev_key = rev_key[4:]

        rev = StorageRevision(self.db)
        rev.load(rev_key)

        if rev.state == 'archive':
            raise Exception('Storage revision already archived')
        if rev.state == 'draft':
            raise Exception('Storage revision is not ready yet (draft)')

        assert rev.state == 'active'

        session = Session(self.db)
        if session_key:
            session_loaded = session.load(session_key)
        else:
            session_loaded = False

        if not session_loaded:
            session.generate('download', 'qdm')

        session.set_state('active')
        session.vm_id = rev.vm_id
        session.rev_id = rev.rev_id
        session.run_vm_id = run_vm_id
        session.run_node_id = run_node_id
        session.touch()
        session.save()

        try:
            rev.access_cnt += 1
            rev.save()

            while True:
                # Request format: meth, (arg1, arg2, arg3)
                request = job.feedback

                if request[0] == 'init':
                    session.add_audit('info', 'Download %s' % ('initialize' if not session_loaded else 'continue'))
                    job.state(('init', session.key))

                elif request[0] == 'get_tvm_ticket':
                    job.state(('tvm_ticket', session.get_mds_tvm_ticket(self.tvm_secret)))

                elif request[0] == 'get_blockmap':
                    storage_datas = [storage_data.dbdict() for storage_data in rev.load_data()]

                    global_idx = 0

                    blockmap = {
                        'version': 1,
                        'files': {}
                    }

                    datas_by_fn = blockmap['files']

                    # filemap v0: [(name, blocks), (name, blocks), ...]
                    # filemap v1: {'v': 1, 'f': [{'n': 'filename', 'b': 1, 's': 497795652, 'm': {'a': 'v'}}]}

                    assert isinstance(rev.filemap, (list, tuple, dict))

                    if isinstance(rev.filemap, (list, tuple)):
                        fileinfo_blocks_cnt = sum(fileinfo[1] for fileinfo in rev.filemap)
                        filetuples = rev.filemap
                    else:
                        fileinfo_blocks_cnt = sum(info.get('b', 0) for info in rev.filemap.get('f', []))
                        filetuples = [(info.get('n', None), info.get('b')) for info in rev.filemap.get('f', [])]

                    assert fileinfo_blocks_cnt == len(storage_datas), (
                        'Block count mismatch fileinfo %d, storage datas %d' % (
                            fileinfo_blocks_cnt, len(storage_datas)
                        )
                    )

                    for fn, idx_count in filetuples:
                        prev_block = None

                        if isinstance(fn, bytes):
                            fn = fn.decode('utf-8')

                        for idx in range(idx_count):
                            block = storage_datas[global_idx]

                            if prev_block:
                                block['offset_byte'] = prev_block['offset_byte'] + prev_block['size']
                            else:
                                block['offset_byte'] = 0

                            prev_block = block
                            global_idx += 1

                            datas_by_fn.setdefault(fn, []).append((
                                block['hashtype'], block['hash'], block['size'],
                                block['mds_key'], block['mds_storage'], block['offset_byte']
                            ))

                    job.state(('blockmap', blockmap))

                elif request[0] == 'finish':
                    session.archive()
                    session.save()
                    session.add_audit('info', 'Download success')
                    return True

                elif request[0] == 'set_progress':
                    job.log.info('progress %r', request[1])
                    session.add_audit('info', 'Download progress %r' % (request[1], ))

                else:
                    raise Exception('Invalid request %r' % (request, ))

                session.save()

            return 42
        except (BaseException, gevent.GreenletExit) as ex:
            session.archive()
            session.save()

            if isinstance(ex, gevent.GreenletExit):
                session.add_audit('error', 'Download not complete: connection closed (got green exit)')
            else:
                session.add_audit('error', 'Download failed: %s: %s' % (type(ex).__name__, str(ex)))
            raise
