import logging
import time

from .model import Log


def human_secs(secs, months=False):
    if months and secs > 86400 * 60:
        months = int(secs // (86400 * 30))
        weeks = (secs % (86400 * 30)) // (86400 * 7)

        if months >= 12 or weeks == 0:
            return '%dM' % (months, )

        return '%dM%dw' % (months, weeks)

    elif secs > 86400 * 14:
        weeks = int(secs // (86400 * 7))
        days = (secs % (86400 * 7)) // 86400

        if weeks >= 4 or days == 0:
            return '%dw' % (weeks, )

        return '%dw%dd' % (weeks, days)

    elif secs > 86400:
        days = int(secs // 86400)
        hrs = (secs % 86400) // 3600

        if hrs == 0:
            return '%dd' % (days, )

        return '%dd%dh' % (days, hrs)

    elif secs > 3600:
        hrs = int(secs // 3600)
        mins = (secs % 3600) // 60

        return '%dh%dm' % (hrs, mins)

    elif secs > 60:
        return '%dm' % (secs // 60)

    else:
        return '%ds' % (secs, )


def human_size(b):
    kb = 1024
    mb = kb * 1024
    gb = mb * 1024
    tb = gb * 1024
    pb = tb * 1024

    if b >= pb:
        return '{:.2f}Pb'.format(b / pb)
    elif b >= tb:
        return '{:.2f}Tb'.format(b / tb)
    elif b >= gb:
        return '{:.2f}Gb'.format(b / gb)
    elif b >= mb:
        return '{:.2f}Mb'.format(b / mb)
    elif b >= kb:
        return '{:.2f}Kb'.format(b / kb)
    else:
        return '{:d}b'.format(b)


class BaseLogDbObj(object):
    def __init__(self, db, log):
        self._log = log
        self._db = db

    def _make_rec(self, ts_ms, severity, msg, args):
        raise NotImplementedError

    def _rec(self, logger, severity, msg, args):
        logger(msg, *args)
        self._make_rec(time.time(), severity, msg, args)

    def debug(self, msg, *args):
        return self._rec(self._log.debug, 'debug', msg, args)

    def info(self, msg, *args):
        return self._rec(self._log.info, 'info', msg, args)

    def warning(self, msg, *args):
        return self._rec(self._log.warning, 'warning', msg, args)

    def error(self, msg, *args):
        return self._rec(self._log.error, 'error', msg, args)

    def critical(self, msg, *args):
        # logging critical as errors in db
        return self._rec(self._log.critical, 'error', msg, args)

    def log(self, level, msg, *args, **kwargs):
        return {
            logging.DEBUG: self.debug,
            logging.INFO: self.info,
            logging.WARNING: self.warning,
            logging.ERROR: self.error,
            logging.CRITICAL: self.critical
        }[level](msg, *args, **kwargs)

    def isEnabledFor(self, level):
        return self._log.isEnabledFor(level)


class LogDbObj(BaseLogDbObj):
    def __init__(self, db, log, name):
        super(LogDbObj, self).__init__(db, log)
        self.name = name

    def _make_rec(self, ts_ms, severity, msg, args):
        rec = Log(self._db, self.name, ts_ms, severity, msg, args)
        rec.save()
