import datetime
import html
import json
import logging
import math
import msgpack
import os
import platform
import requests
import time
import urllib

from collections import OrderedDict

from .component import Component
from .deblock import Deblock
from .model import Session, StorageRevision, Evoq, CLUSTER_NAMES, convert_vm_id_and_cluster_to_db_vm_id
from .utils import human_secs, human_size
from .evoq.common import DC

import flask
import gevent
import yaml


class Web(Component):
    HTTP_PORT = 80
    HTTP_PORT_ALT = 8080
    HTTPS_PORT = 443
    HTTPS_PORT_ALT = 8443

    def __init__(self, parent, db, hostname, vmproxy_oauth_token):
        super(Web, self).__init__(parent=parent, logname='web')

        self.db = db
        self.http_app = flask.Flask(__name__)
        self.log = logging.getLogger('web')
        self.hostname = hostname
        self.vmproxy_oauth_token = vmproxy_oauth_token

        self._ssl = False
        self._ssl_key = self._ssl_cert = None

        # Setup basic http
        self.http_app.add_url_rule(
            '/', 'index', self.web_index
        )
        self.http_app.add_url_rule(
            '/ping', 'ping', self.web_ping
        )

        self.http_app.add_url_rule(
            '/evoq/', 'evoq', self.web_evoq
        )

        self.http_app.add_url_rule(
            '/evoq/<int:id>', 'evoq_info', self.web_evoq_info
        )

        self.http_app.add_url_rule(
            '/stats', 'stats', self.web_stats
        )

        # Used in 0.24..0.25 vmagent
        self.http_app.add_url_rule(
            '/cli_resource', 'cli_resource', self.web_cli_resource
        )
        self.http_app.add_url_rule(
            '/init_upload_session', 'init_upload_session', self.web_init_upload_session, methods=['POST']
        )

        # Used in 0.26+ vmagent
        self.http_app.add_url_rule(
            '/api/v1/hostname', 'api_v1_hostname', self.web_hostname
        )
        self.http_app.add_url_rule(
            '/api/v1/client_resource', 'api_v1_client_resource', self.web_cli_resource
        )
        self.http_app.add_url_rule(
            '/api/v1/init_upload_session', 'api_v1_init_upload_session',
            self.web_init_upload_session, methods=['POST']
        )
        self.http_app.add_url_rule(
            '/api/v1/revision_info/<string:revision_key>', 'api_v1_revision_info',
            self.web_revision_info, methods=['GET']
        )

        self.http_app.add_url_rule(
            '/api/v1/backup_create', 'api_v1_backup_create', self.web_backup_create, methods=['POST']
        )
        self.http_app.add_url_rule(
            '/api/v1/backup_status', 'api_v1_backup_status', self.web_backup_status
        )
        self.http_app.add_url_rule(
            '/api/v1/backup_list', 'api_v1_backup_list', self.web_backup_list
        )

        self.http_app.add_url_rule(
            '/api/v1/user_backup_list', 'api_v1_user_backup_list', self.web_user_backup_list
        )

        self.http_app.add_url_rule(
            '/metrics/solomon', 'metrics_solomon',
            self.web_metrics_solomon
        )

        self.http_app.after_request(self._web_after_request)

        yaml.add_representer(
            OrderedDict,
            lambda self, data: self.represent_mapping('tag:yaml.org,2002:map', data.items())
        )

        self._deblocker = Deblock(logger=self.log.getChild('deblock'), keepalive=None)

    def _web_after_request(self, response):
        response.headers['X-Forwarded-Host'] = self.hostname

        if flask.request.path in ('/ping', 'favicon.ico'):
            return response

        self.http_app.logger.debug(
            '%s %s %s %s',
            flask.request.remote_addr, flask.request.method,
            flask.request.full_path, response.status
        )
        return response

    def enable_ssl(self, ssl_key, ssl_cert):
        self._ssl = True
        self._ssl_key = ssl_key
        self._ssl_cert = ssl_cert

    def listen(self):
        if os.getuid() == 0:
            self.log.info('Listening HTTP on *:%d', self.HTTP_PORT)
            self.http_server = gevent.pywsgi.WSGIServer(('', self.HTTP_PORT), self.http_app, log=None)

            if self._ssl:
                self.log.info('Listening HTTPS on *:%s', self.HTTPS_PORT)
                self.http_server_ssl = gevent.pywsgi.WSGIServer(
                    ('', self.HTTPS_PORT), self.http_app,
                    keyfile=self._ssl_key, certfile=self._ssl_cert,
                    log=None
                )

        self.log.info('Listening HTTP on *:%s', self.HTTP_PORT_ALT)
        self.http_server_alt = gevent.pywsgi.WSGIServer(('', self.HTTP_PORT_ALT), self.http_app, log=None)

        if self._ssl:
            self.log.info('Listening HTTPS on *:%s', self.HTTPS_PORT_ALT)
            self.http_server_alt_ssl = gevent.pywsgi.WSGIServer(
                ('', self.HTTPS_PORT_ALT), self.http_app,
                keyfile=self._ssl_key, certfile=self._ssl_cert,
                log=None
            )

    def start(self):
        if os.getuid() == 0:
            gevent.spawn(self.http_server.serve_forever)

            if self._ssl:
                gevent.spawn(self.http_server_ssl.serve_forever)

        gevent.spawn(self.http_server_alt.serve_forever)

        if self._ssl:
            gevent.spawn(self.http_server_alt_ssl.serve_forever)

    def _header(self):
        response = ['<body>']
        response.append('<style type="text/css">')
        response.append('body { font-family: Verdana; font-size: 0.7em; background: #222; color: white }')
        response.append('a { color: orange; font-weight: bold }')
        response.append('a:hover { color: yellow }')
        response.append('hr { border-color: #735f39 }')
        response.append('</style>')
        response.append('<pre>')

        response.append('Hi, world! (qdm v0.4)')
        response.append('Current node: %s' % (platform.node(), ))

        response.append('')
        response.append(
            '<a href="/">uploads</a> | '
            '<a href="/evoq">evoq</a> | '
            '<a href="/stats">stats</a>'
            '<hr/>'
        )

        response.append('<hr/>')

        return '\n'.join(response)

    def web_index(self):
        response = [self._header()]

        active_sessions = self.db.query_one_col('SELECT COUNT(*) FROM session WHERE state = %s', ('active', ))
        response.append('Active sessions [%d]:' % (active_sessions, ))

        now = time.time()

        found_active_sessions = False

        for (
            key, type, state, state_ts, modify_ts,
            vm_id, rev_id, node_id, origin,
            last_state, last_state_ts
        ) in self.db.query(
            'SELECT '
            '   key, type, state, state_ts, modify_ts, vm_id, rev_id, node_id, origin, '
            '   (SELECT message FROM audit WHERE session = s.key ORDER BY timestamp DESC LIMIT 1), '
            '   (SELECT timestamp FROM audit WHERE session = s.key ORDER BY timestamp DESC LIMIT 1) '
            'FROM session s '
            'WHERE s.state = %s '
            'ORDER BY s.state_ts DESC LIMIT 100', ('active', )
        ):
            found_active_sessions = True

            if last_state is not None and last_state_ts is not None:
                last_state_str = '%s, %s ago' % (last_state, human_secs(time.time() - last_state_ts))
            else:
                last_state_str = None

            vmname = '%s rev %s' % (vm_id, rev_id or '?')
            typestr = 'ul' if type == 'upload' else 'dl'
            statestr = {
                'active': 'act',
                'archive': 'arh',
                'new': 'new'
            }[state]

            response.append(' - [%4s] [%-40s | %40s]  (%s|%s) state %s ago, modified %s ago' % (
                origin, vmname, node_id or '', statestr, typestr,
                human_secs(now - state_ts), human_secs(now - modify_ts)
            ))

            if last_state_str:
                response[-1] = '%-95s (%s)' % (response[-1], last_state_str)

        if not found_active_sessions:
            response.append('  none')

        response.append('')

        new_sessions = self.db.query_one_col('SELECT COUNT(*) FROM session WHERE state = %s', ('new', ))
        response.append('New sessions [%d]:' % (new_sessions, ))

        found_new_sessions = False

        for key, type, state_ts, cnt, vm_id in self.db.query(
            'SELECT max(key), max(type), max(state_ts) as state_ts, count(vm_id), vm_id '
            'FROM session WHERE state = %s '
            'GROUP BY vm_id '
            'ORDER BY state_ts DESC',
            ('new', )
        ):
            found_new_sessions = True

            typestr = 'ul' if type == 'upload' else 'dl'
            response.append(' - [%02d] [%-40s]  (new|%s) state %s ago' % (
                cnt, vm_id, typestr, human_secs(now - state_ts)
            ))

        if not found_new_sessions:
            response.append('  none')

        response.append('')

        response.append('Session state counts:')
        for state, count in self.db.query(
            'SELECT state, count(*) FROM session GROUP BY state'
        ):
            response.append(' - %-7s: %d' % (state, count))

        response.append('')
        response.append('<hr/>')

        response.append('Internal jobs:')

        for name, node, run_ts, last_node, last_run_ts, next_run_ts in self.db.query(
            'SELECT name, node, run_ts, last_node, last_run_ts, next_run_ts FROM job ORDER BY next_run_ts ASC'
        ):
            if node:
                response.append(' - [%-20s] running on %s for %ds' % (name, node, time.time() - run_ts))
            else:
                if last_run_ts:
                    response.append(' - [%-20s] last run %s ago on %s, next run in %s' % (
                        name, human_secs(time.time() - last_run_ts), last_node, human_secs(next_run_ts - time.time())
                    ))
                else:
                    response.append(' - [%-20s] not yet run, next run in %s' % (
                        name, human_secs(next_run_ts - time.time()))
                    )

        response.append('')
        response.append('<hr/>')

        response.append('Evoque:')

        for state, count in self.db.query('SELECT state, COUNT(*) FROM evoq GROUP BY state'):
            response.append(
                ' - {state}: <a href="/evoq/?states={state}">{count}</a>'.format(
                    state=state, count=count
                )
            )

        response.append('</body>')

        return '\n'.join(response)

    def web_evoq(self):
        response = [self._header()]

        state_links = ['<a href="/evoq/">all</a>']

        for state in ('init', 'run', 'fail', 'stop', 'done'):
            state_links.append('<a href=/evoq/?states={state}>{state}</a>'.format(state=state))

        response.append('state filter: %s' % (' | '.join(state_links)))

        response.append('')
        response.append('<hr/>')

        query = (
            'SELECT '
            '   id, vm_id, node_id, session_key, state, extra, '
            '   run_node, run_cnt, run_ts, info, run_duration '
            'FROM evoq'
        )
        query_args = []

        states = flask.request.args.get('states', '').split(',')

        if states and states != ['']:
            query += (
                ' WHERE state IN (%S)'
            )
            query_args.append(states)

        query += ' ORDER BY id DESC LIMIT 100'

        found_some = False
        for (
            evoq_id, vm_id, node_id, session_key, state, extra,
            run_node, run_cnt, run_ts, info, run_duration
        ) in self.db.query(query, query_args):
            id_link = '<a href="/evoq/{id}">{id}</a>'.format(id=evoq_id)

            if extra:
                extra = msgpack.loads(extra)
                extra_state = extra.get('state', '')
            else:
                extra_state = ''

            response.append(
                ' - [ {id_link} {state:>5} ] {vm:<40} {node:<30} '
                '{run_node:<30} {run_duration:>6} {extra_state:>10}'.format(
                    id_link=id_link, state=state, vm=vm_id, node=node_id, run_node=run_node,
                    run_duration=human_secs(run_duration), extra_state=extra_state
                )
            )
            found_some = True

        if not found_some:
            response.append('none')

        return '\n'.join(response)

    def web_evoq_info(self, id):
        response = [self._header()]

        evoq = Evoq(self.db, None, None)
        if not evoq.load(id):
            flask.abort(404)

        for key, value in evoq.__dict__.items():
            if key.startswith('_') or key == 'db':
                continue

            response.append('{key:>20}: {value}'.format(key=key, value=value))

        response.append('')
        response.append('<hr/>')

        for node, timestamp_ms, severity, message in self.db.query(
            'SELECT node, timestamp_ms, severity, message '
            'FROM evoq_log '
            'WHERE evoq_id = %s '
            'ORDER BY timestamp_ms desc',
            (evoq.id, )
        ):
            date = datetime.datetime.fromtimestamp(timestamp_ms / 1000)
            highlight = 'playing round' in message

            logline = []
            if highlight:
                logline.append('<span style="color: yellow; font-weight: bold">')

            logline.append('{node:<20}   {date}   {severity:>7}   {message}'.format(
                node=node,
                date=date.isoformat()[:-3],  # cut ns
                severity=severity.upper(),
                message=html.escape(message)
            ))

            if highlight:
                logline.append('</span>')

            response.append(''.join(logline))

            if 'playing round #1' in message:
                response.append('')

        return '\n'.join(response)

    _stats_cache = {}

    def web_stats(self):
        def _query_cached(key, query, meth='query_one_col'):
            now = time.time()
            if key not in self._stats_cache or now - self._stats_cache[key][0] > 3600:
                self._stats_cache[key] = (
                    now,
                    getattr(self.db, meth)(query)
                )
            return self._stats_cache[key][1]

        response = [self._header()]

        blocks_size = _query_cached('mds_blocks_size', 'SELECT SUM(size) FROM storage_block')
        orphan_blocks_cnt, orphan_blocks_size = _query_cached(
            'orphan_blocks',
            'SELECT COUNT(id), SUM(size) FROM storage_block sb '
            '   LEFT JOIN storage_data sd ON sd.block_id = sb.id '
            'WHERE sd.block_id IS NULL',
            'query_one'
        )

        orphan_blocks_size_str = '{:s} ({:.2%} of all)'.format(
            human_size(orphan_blocks_size),
            orphan_blocks_size / blocks_size
        )

        average_rev_blocks = _query_cached(
            'avg_rev_blocks',
            'SELECT AVG(t.bcount) FROM (SELECT COUNT(*) as bcount FROM storage_data GROUP BY vm_id) as t'
        )

        mds_stats = requests.get('http://storage-int.mds.yandex.net/statistics-qdm').json()

        mds_used_eff_space_str = '{:s} (effective, {:.2%} of limit)'.format(
            human_size(mds_stats['effective_used_space']),
            mds_stats['effective_used_space'] / mds_stats['space_limit']
        )

        mds_used_real_space_str = '{:s} (real)'.format(
            human_size(mds_stats['used_space']),
        )

        mds_total_space_left_str = '{:s} ({:.2%} of limit)'.format(
            human_size(mds_stats['total_space']),
            mds_stats['total_space'] / mds_stats['space_limit']
        )

        stats = OrderedDict({
            'Storage': OrderedDict({
                'total backup revs': self.db.query_one_col('SELECT COUNT(*) FROM storage_revision'),
                'backed up vm count': self.db.query_one_col('SELECT COUNT(DISTINCT vm_id) FROM storage_revision'),
                'mds blocks': _query_cached('mds_block_cnt', 'SELECT COUNT(*) FROM storage_block'),
                'mds size': human_size(blocks_size),
                'mds size (uniq)': human_size(_query_cached(
                    'mds_size_uniq',
                    'SELECT SUM(t.bsize) FROM ('
                    '   SELECT MAX(size) as bsize FROM storage_block GROUP BY mds_key'
                    ') as t'
                )),
                'orphan blocks cnt ': orphan_blocks_cnt,
                'orphan blocks size': orphan_blocks_size_str,
                'avg block count': math.ceil(average_rev_blocks),
                'avg backup size': human_size(average_rev_blocks * 512 * 1024 * 1024)
            }),
            'Sessions': OrderedDict({
                'total': self.db.query_one_col('SELECT COUNT(*) FROM session'),
                'dl': self.db.query_one_col('SELECT COUNT(*) FROM session WHERE type = %s', ('download', )),
                'ul': self.db.query_one_col('SELECT COUNT(*) FROM session WHERE type = %s', ('upload', ))
            }),
            'MDS': OrderedDict({
                'space limit': human_size(mds_stats['space_limit']),
                'used_space effc': mds_used_eff_space_str,
                'used space real': mds_used_real_space_str,
                'free_space': '{:s} (allocated)'.format(human_size(mds_stats['effective_free_space'])),
                'total keys': mds_stats['total_keys'],
                'total space left': mds_total_space_left_str,
                'exp full': '{:s} (in {:.0f} hrs)'.format(
                    mds_stats['can_be_full']['date'],
                    mds_stats['can_be_full']['remaining_hours']
                )
            })
        })

        for blockname, stats in stats.items():
            response.append('{blockname}:'.format(blockname=blockname))
            for key, value in stats.items():
                response.append('{key:>30}: {value}'.format(key=key, value=value))

        return '\n'.join(response)

    def web_ping(self):
        return ''

    def web_hostname(self):
        return platform.node()

    def web_cli_resource(self):
        # last update: migrate to production mds
        # last update: output only revision to stdout, grab key instead of id
        # last update: output qdm resource with qdm prefix
        # last update: add fallocate and fix rc during upload finish
        # last update: retry retryable errors (including EBADF)
        # last update: tune buffers count based on porto memory limit
        # last update: attempt to retry network and other failures
        # last update: add http timeouts
        # last update: multi storage support, tune http retries (10cnt => 2hrs)
        # last update: support download files with directory names
        # last update: support --background arg
        return 'rbtorrent:184242dc3585e38e34b8545eb1efe82a49d26896'

    def web_init_upload_session(self):
        tvm_ticket = flask.request.headers.get('X-Ya-Service-Ticket', None)
        if not tvm_ticket:
            return 'Forbidden: TVM Service Ticket required\n', 403

        assert flask.request.json['dc'].lower() in CLUSTER_NAMES

        session = Session(self.db)
        session.generate('upload', 'qdm')
        session.vm_id = convert_vm_id_and_cluster_to_db_vm_id(
            flask.request.json['vm_id'],
            flask.request.json['dc'].lower()
        )
        session.save()

        return flask.jsonify({'key': session.key})

    def web_revision_info(self, revision_key):
        fmt = flask.request.args.get('fmt', 'json')

        if fmt not in ('yaml', 'json'):
            flask.abort(400, 'Invalid fmt requested %r, only yaml and json are supported' % (fmt, ))

        if revision_key.startswith('qdm:'):
            revision_key = revision_key.split(':', 1)[1]

        rev = StorageRevision(self.db)
        if not rev.load(revision_key):
            flask.abort(404)

        if rev.state != 'active':
            flask.abort(404)

        if isinstance(rev.filemap, (list, tuple)):
            # Old filemap
            blocks = rev.load_data()

            filemap = []
            curblk = 0

            for (name, blocks_cnt) in rev.filemap:
                size = 0
                for i in range(blocks_cnt):
                    block = blocks[curblk]
                    size += block.size
                    curblk += 1

                filemap.append({
                    'path': name,
                    'size': size,
                    'meta': {}
                })
        else:
            assert rev.filemap['v'] == 1, 'Unknown filemap version'

            filemap = []

            for fileinfo in rev.filemap['f']:
                filemap.append({
                    'path': fileinfo['n'],
                    'size': fileinfo['s'],
                    'meta': fileinfo['m']
                })

        result = OrderedDict({
            'qdm_spec_version': 1,
            'rev_id': revision_key,
            'create_ts': rev.create_ts,
            'origin': rev.origin,
            'filemap': filemap,
            'vmspec': rev.vmspec,
        })

        if fmt == 'json':
            return flask.jsonify(result)

        elif fmt == 'yaml':
            response = yaml.dump(result, default_flow_style=False)
            return flask.Response(response, mimetype='text/vnd.yaml')

    def web_metrics_solomon(self):
        # Labels breakdown:
        # name=evoq_cnt, state=STATE
        # name=evoq_pending_time_(sum,avg,max)
        # name=evoq_pending_days_(sum,avg,max)
        #
        # name=job_duration, job=JOB_NAME
        # name=job_counts, type=(success, failed), job=JOB_NAME
        # name=job_lag, job=JOB_NAME
        #
        # name=session, type=(upload, download), origin=origin, state=(new,active,archive)
        # name=session_result, origin=origin, success=(yes|no)
        #
        # name=vm, segment=SEGMENT                                  # total vms
        # name=vm_hotbackup, segment=SEGMENT, state=(overdue,ok,disabled)
        # name=vm_backup_revision, origin=(evoq,user,hot)
        # name=vm_backup_status, segment=SEGMENT, has_backup=(true,false) # vms with/without backups, only existing
        # name vm_backup, segment=SEGMENT, hot_allow, age=(avg,90p,50p)   # backup age, only for whose who has backup

        fmt = flask.request.args.get('fmt', 'json')
        now = time.time()

        if fmt not in ('yaml', 'json'):
            flask.abort(400, 'Invalid fmt requested %r, only yaml and json are supported' % (fmt, ))

        evoq_counts = {
            'init': 0,
            'wait': 0,
            'run': 0,
            'fail': 0,
            'stop': 0,
            'done': 0
        }

        for state, count in self.db.query('SELECT state, COUNT(*) FROM evoq GROUP BY state'):
            evoq_counts[state] = count

        evoq_counts_sensors = []

        for state, count in evoq_counts.items():
            evoq_counts_sensors.append(OrderedDict({
                'labels': {'name': 'evoq_cnt', 'state': state},
                'value': int(count),
                'kind': 'COUNTER'
            }))

        pending_times = self.db.query_one(
            'SELECT '
            '   SUM(%s - init_ts), '
            '   AVG(%s - init_ts), '
            '   MAX(%s - init_ts) '
            'FROM evoq '
            'WHERE active and init_ts > 0',
            (
                int(now),
                int(now),
                int(now)
            )
        )

        if pending_times is not None:
            evoq_pending_time = {
                'sum': pending_times[0],
                'avg': pending_times[1],
                'max': pending_times[2]
            }
        else:
            evoq_pending_time = {'sum': 0, 'avg': 0, 'max': 0}

        evoq_pending_time_sensors = []

        for typ, pending_time in evoq_pending_time.items():
            if pending_time is None:
                pending_time = 0

            evoq_pending_time_sensors.append(OrderedDict({
                'labels': {'name': 'evoq_pending_time_%s' % (typ, )},
                'value': int(pending_time),
                'kind': 'IGAUGE'
            }))
            evoq_pending_time_sensors.append(OrderedDict({
                'labels': {'name': 'evoq_pending_days_%s' % (typ, )},
                'value': float(pending_time / 86400),
                'kind': 'DGAUGE'
            }))

        job_sensors = []

        for job, run_ts, next_run_ts, cnt_success, cnt_failed in (
            self.db.query('SELECT name, run_ts, next_run_ts, cnt_success, cnt_failed FROM job')
        ):
            if run_ts:
                duration = int(time.time() - run_ts)

                job_sensors.append(OrderedDict({
                    'labels': {'name': 'job_duration', 'job': job},
                    'value': duration,
                    'kind': 'COUNTER'
                }))

            job_sensors.append(OrderedDict({
                'labels': {'name': 'job_counts', 'job': job, 'type': 'success'},
                'value': cnt_success,
                'kind': 'COUNTER'
            }))
            job_sensors.append(OrderedDict({
                'labels': {'name': 'job_counts', 'job': job, 'type': 'failed'},
                'value': cnt_failed,
                'kind': 'COUNTER'
            }))
            job_sensors.append(OrderedDict({
                'labels': {'name': 'job_lag', 'job': job},
                'value': max(0, time.time() - next_run_ts)
            }))

        backup_sensors = []

        session_states = 'new', 'active', 'archive'
        session_types = 'upload', 'download'
        session_origins = 'qdm', 'evoq', 'hot'

        session_counts = {}

        for state, typ, origin, cnt in self.db.query(
            'SELECT state, type, origin, count(*) FROM session GROUP BY state, type, origin',
            ('active', )
        ):
            session_counts[(state, typ, origin)] = cnt

        for typ in session_types:
            for state in session_states:
                for origin in session_origins:
                    pair = state, typ, origin
                    if pair not in session_counts:
                        session_counts[pair] = 0

        for (state, typ, origin), cnt in session_counts.items():
            if state == 'active':
                kind = 'DGAUGE'
            else:
                kind = 'COUNTER'

            backup_sensors.append(OrderedDict({
                'labels': {'name': 'session', 'type': typ, 'state': state, 'origin': origin},
                'value': cnt,
                'kind': kind
            }))

        for origin, success, cnt in self.db.query(
            'SELECT origin, rev_id IS NOT NULL as success, COUNT(*) '
            'FROM session '
            'WHERE type = %s AND state = %s '
            'GROUP BY origin, success',
            ('upload', 'archive')
        ):
            backup_sensors.append(OrderedDict({
                'labels': {
                    'name': 'session_result', 'origin': origin,
                    'success': 'yes' if success else 'no'
                },
                'value': cnt,
                'kind': 'COUNTER'
            }))

        hotbackup_sensors = []

        all_segments = [s[0] for s in self.db.query('SELECT DISTINCT segment FROM vm')]

        seen_overdue = set()
        for segment, overdue_cnt in self.db.query(
            'SELECT subq.segment, COUNT(*) FROM ('
            '   SELECT vm_id, segment, hot_period, ('
            '       SELECT MAX(create_ts) as max_create_ts '
            '       FROM storage_revision '
            '       WHERE ('
            '           vm_id = vm.vm_id'
            '           AND state = %s'
            '       )'
            '   ) as mts'
            '   FROM vm'
            '   WHERE ('
            '       vm.hot_allow'
            '       AND vm.exists'
            '   )'
            ') as subq '
            'WHERE ('
            '   subq.mts IS NULL'
            '   OR subq.mts <= (%s - subq.hot_period) '
            ')'
            'GROUP BY subq.segment',
            (
                'active', now
            )
        ):
            hotbackup_sensors.append(OrderedDict({
                'labels': {'name': 'vm_hotbackup', 'segment': segment, 'state': 'overdue'},
                'value': overdue_cnt,
                'kind': 'COUNTER'
            }))
            seen_overdue.add(segment)

        for segment in all_segments:
            if segment not in seen_overdue:
                hotbackup_sensors.append(OrderedDict({
                    'labels': {'name': 'vm_hotbackup', 'segment': segment, 'state': 'overdue'},
                    'value': 0,
                    'kind': 'COUNTER'
                }))

        seen_ok = set()
        for segment, ok_cnt in self.db.query(
            'SELECT subq.segment, COUNT(*) FROM ('
            '   SELECT vm_id, segment, hot_period, ('
            '       SELECT MAX(create_ts) as max_create_ts '
            '       FROM storage_revision '
            '       WHERE ('
            '           vm_id = vm.vm_id'
            '           AND state = %s'
            '       )'
            '   ) as mts'
            '   FROM vm'
            '   WHERE ('
            '       vm.hot_allow'
            '       AND vm.exists'
            '   )'
            ') as subq '
            'WHERE ('
            '   subq.mts > (%s - subq.hot_period) '
            ')'
            'GROUP BY subq.segment',
            (
                'active', now
            )
        ):
            hotbackup_sensors.append(OrderedDict({
                'labels': {'name': 'vm_hotbackup', 'segment': segment, 'state': 'ok'},
                'value': ok_cnt,
                'kind': 'COUNTER'
            }))
            seen_ok.add(segment)

        for segment in all_segments:
            if segment not in seen_ok:
                hotbackup_sensors.append(OrderedDict({
                    'labels': {'name': 'vm_hotbackup', 'segment': segment, 'state': 'ok'},
                    'value': 0,
                    'kind': 'COUNTER'
                }))

        seen_disabled = set()
        for segment, not_allow_cnt in self.db.query(
            'SELECT segment, COUNT(*) '
            'FROM vm '
            'WHERE ('
            '   vm.exists'
            '   AND NOT vm.hot_allow'
            ')'
            'GROUP BY segment'
        ):
            hotbackup_sensors.append(OrderedDict({
                'labels': {'name': 'vm_hotbackup', 'segment': segment, 'state': 'disabled'},
                'value': not_allow_cnt,
                'kind': 'COUNTER'
            }))
            seen_disabled.add(segment)

        for segment in all_segments:
            if segment not in seen_disabled:
                hotbackup_sensors.append(OrderedDict({
                    'labels': {'name': 'vm_hotbackup', 'segment': segment, 'state': 'disabled'},
                    'value': 0,
                    'kind': 'COUNTER'
                }))

        for segment, total_vm in self.db.query('SELECT segment, COUNT(*) FROM vm WHERE exists GROUP BY segment'):
            hotbackup_sensors.append(OrderedDict({
                'labels': {'name': 'vm', 'segment': segment},
                'value': total_vm,
                'kind': 'COUNTER'
            }))

        revision_sensors = []

        all_origins = ('qdm', 'evoq', 'hot')
        all_states = ('active', 'draft', 'archive')

        revision_counts_by_origin = {}

        for origin in all_origins:
            for state in all_states:
                revision_counts_by_origin.setdefault(origin, {})[state] = 0

        for origin, state, count in self.db.query(
            'SELECT origin, state, COUNT(*) '
            'FROM storage_revision '
            'WHERE ('
            '   state = %s'
            ')'
            'GROUP BY origin, state',
            ('active', )
        ):
            revision_counts_by_origin[origin][state] = count

        for origin, state_counts in revision_counts_by_origin.items():
            for state, count in state_counts.items():
                revision_sensors.append(OrderedDict({
                    'labels': {'name': 'vm_backup_revision', 'origin': origin, 'state': state},
                    'value': count,
                    'kind': 'COUNTER'
                }))

        max_backup_ts_for_each_vm = {}
        for vm, segment, max_backup_ts in self.db.query(
            'SELECT vm.vm_id, vm.segment, MAX(sr.create_ts) '
            'FROM vm LEFT JOIN storage_revision sr ON sr.vm_id = vm.vm_id '
            'WHERE vm.exists '
            'GROUP BY vm.vm_id, vm.segment'
        ):
            max_backup_ts_for_each_vm.setdefault(segment, {})[vm] = max_backup_ts or 0

        vms_without_backup_by_segment = {}
        vms_with_backup_by_segment = {}

        for segment, vms in max_backup_ts_for_each_vm.items():
            vms_without_backup_by_segment[segment] = (
                len([_ for _ in max_backup_ts_for_each_vm[segment].values() if _ == 0])
            )
            vms_with_backup_by_segment[segment] = (
                len(max_backup_ts_for_each_vm[segment]) - vms_without_backup_by_segment[segment]
            )

        backup_age_by_segment_and_hot_allow = {}

        for vm_id, segment, hot_allow, backup_age in self.db.query(
            'SELECT'
            '   vm.vm_id, vm.segment, vm.hot_allow, (%s - MAX(sr.create_ts)) as age '
            'FROM vm '
            'JOIN storage_revision sr ON sr.vm_id = vm.vm_id '
            'WHERE ('
            '   vm.exists '
            ') '
            'GROUP BY '
            '   vm.vm_id, vm.segment, vm.hot_allow '
            'ORDER BY age',
            (now, )
        ):
            backup_age_by_segment_and_hot_allow \
                .setdefault(segment, {}) \
                .setdefault(hot_allow, OrderedDict())[vm_id] = backup_age

        backup_timings = []
        backup_timings_only_exists = []

        # List of backup age in seconds. First one includes vms without backup (current timestamp)

        backup_timings = sorted(backup_timings, reverse=True)
        backup_timings_only_exists = sorted(backup_timings_only_exists, reverse=True)

        for segment, data in backup_age_by_segment_and_hot_allow.items():
            for hot_allow, vm_ages in data.items():
                values_tuple = tuple(vm_ages.values())

                for percentile in 1.0, 0.99, 0.95, 0.9, 0.75, 0.5, 0.25, 0.1:
                    frame = values_tuple[:int(len(vm_ages) * percentile)]
                    if not frame:
                        if values_tuple:
                            frame = values_tuple[-1:]  # grab just one last vm with oldest backup age
                        else:
                            frame = [0]                # no vms in tuple - just use 0

                    value = max(frame)

                    agestr = '%dp' % (int(percentile * 100), )

                    backup_sensors.append(OrderedDict({
                        'labels': {
                            'name': 'vm_backup',
                            'segment': segment,
                            'hot_allow': hot_allow,
                            'has_backup': True,
                            'age': agestr,
                        },
                        'value': float(value / 86400),
                        'kind': 'DGAUGE'
                    }))

                for aggr in ('min', 'max', 'avg'):
                    if values_tuple:
                        if aggr == 'min':
                            value = min(values_tuple)
                        elif aggr == 'max':
                            value = max(values_tuple)
                        elif aggr == 'avg':
                            value = sum(values_tuple) / len(values_tuple)
                        else:
                            value = 0
                    else:
                        value = 0

                    backup_sensors.append(OrderedDict({
                        'labels': {
                            'name': 'vm_backup',
                            'segment': segment,
                            'hot_allow': hot_allow,
                            'has_backup': True,
                            'age': aggr
                        },
                        'value': float(value / 86400),
                        'kind': 'DGAUGE'
                    }))

        for segment, vms_with_backup in vms_with_backup_by_segment.items():
            backup_sensors.append(OrderedDict({
                'labels': {'name': 'vm_backup_status', 'segment': segment, 'has_backup': True},
                'value': vms_with_backup,
                'kind': 'COUNTER'
            }))

        for segment, vms_without_backup in vms_without_backup_by_segment.items():
            backup_sensors.append(OrderedDict({
                'labels': {'name': 'vm_backup_status', 'segment': segment, 'has_backup': False},
                'value': vms_without_backup,
                'kind': 'COUNTER'
            }))

        result = OrderedDict({
            'sensors': (
                evoq_counts_sensors
                + evoq_pending_time_sensors
                + job_sensors
                + backup_sensors
                + hotbackup_sensors
                + revision_sensors
            )
        })

        if fmt == 'json':
            return flask.jsonify(result)

        elif fmt == 'yaml':
            response = yaml.dump(result, default_flow_style=False)
            return flask.Response(response, mimetype='text/vnd.yaml')

    def web_backup_create(self):
        tvm_ticket = flask.request.headers.get('X-Ya-Service-Ticket', None)
        if not tvm_ticket:
            return 'Forbidden: TVM Service Ticket required\n', 403

        vmid = flask.request.json['vmid']
        dc = flask.request.json['dc']

        if dc not in CLUSTER_NAMES:
            flask.abort(400, 'Invalid cluster %r, only %r are supported' % (dc, CLUSTER_NAMES))

        if not isinstance(vmid, str):
            flask.abort(400, 'Invalid vm id specified %r' % (vmid, ))

        vm_id = convert_vm_id_and_cluster_to_db_vm_id(vmid, dc)

        pod_id = vmid
        del vmid

        session = Session(self.db)

        session.generate('upload', 'qdm')
        session.vm_id = vm_id
        session.save()

        try:
            url = urllib.parse.urljoin(DC[dc]['vmproxy'], '/api/MakeAction/')

            response = self._deblocker.apply(
                requests.post, url, data=json.dumps({
                    'vm_id': {'pod_id': pod_id},
                    'action': 'QDMUPLOAD',
                    'qdmreq': {'key': session.key}
                }),
                headers={
                    'Content-Type': 'application/json',
                    'Authorization': 'OAuth %s' % (self.vmproxy_oauth_token, )
                },
                timeout=30
            )

            if response.status_code != 200:
                raise Exception('vmproxy error: %d %s' % (response.status_code, response.text))

        except Exception as ex:
            session.archive()
            session.save()
            flask.abort(500, 'Error: %s' % (str(ex), ))

        return ''

    def web_backup_status(self):
        tvm_ticket = flask.request.headers.get('X-Ya-Service-Ticket', None)
        if not tvm_ticket:
            return 'Forbidden: TVM Service Ticket required\n', 403

        vmid = flask.request.args.get('vmid', None)
        dc = flask.request.args.get('dc', None)

        if dc not in CLUSTER_NAMES:
            flask.abort(400, 'Invalid cluster %r, only %r are supported' % (dc, CLUSTER_NAMES))

        if not isinstance(vmid, str):
            flask.abort(400, 'Invalid vm id specified %r' % (vmid, ))

        vm_id = convert_vm_id_and_cluster_to_db_vm_id(vmid, dc)
        del vmid, dc  # avoid usage

        existing_session_id = self.db.query_one_col(
            'SELECT key FROM session WHERE vm_id = %s AND type = %s AND state != %s',
            (vm_id, 'upload', 'archive')
        )

        session = Session(self.db)

        if not session.load(existing_session_id):
            flask.abort(404, 'No active backup session found')

        if time.time() - session.modify_ts > 600:
            flask.abort(404, 'No active backup session found')

        return flask.jsonify({
            'state': session.state,
            'revision': session.rev_id,
            'node_id': session.node_id,
            'origin': session.origin,
        })

        # self.db.query('SELECT

    def web_backup_list(self):
        tvm_ticket = flask.request.headers.get('X-Ya-Service-Ticket', None)
        if not tvm_ticket:
            return 'Forbidden: TVM Service Ticket required\n', 403

        vmid = flask.request.args.get('vmid', None)
        dc = flask.request.args.get('dc', None)

        if dc not in CLUSTER_NAMES:
            flask.abort(400, 'Invalid cluster %r, only %r are supported' % (dc, CLUSTER_NAMES))

        if not isinstance(vmid, str):
            flask.abort(400, 'Invalid vm id specified %r' % (vmid, ))

        vm_id = convert_vm_id_and_cluster_to_db_vm_id(vmid, dc)
        del vmid, dc  # avoid usage

        revs = StorageRevision.search_state(self.db, vm_id, 'active')
        revs = [r.dbdict() for r in revs]

        return flask.jsonify(revs)

    def web_user_backup_list(self):
        user = flask.request.args.get('user', None)
        groups = flask.request.args.getlist('groups', None)

        if not isinstance(user, str):
            flask.abort(400, 'Invalid user specified %r' % user)

        if groups and not isinstance(groups, list):
            flask.abort(400, 'Invalid groups specified %r' % groups)

        owners_revs = StorageRevision.select_owners_revs(self.db, user, groups)
        revs = [r.dbdict() for r in owners_revs]

        return flask.jsonify(revs)
