import datetime
import json
import logging
import math
import os
import time

import requests

from yql.api.v1.client import YqlClient

YA_MAKE_PREFIX = 'ya make'
YQL_QUERY_TMPL = """
SELECT
    `value`,
    `user`,
    `hostname`,
    `timestamp`
FROM hahn.`home/devtools-snowden/snowden_v1_report/{date}`
WHERE
    `namespace` = "yatool"
    AND `key` = "timeit"
    AND `timestamp` > {timestamp}
    AND `hostname` LIKE "%.yp-c.yandex.net"
ORDER BY `timestamp`;
"""


class YqlHelper(object):
    DEFAULT_YQL_DB = 'hahn'

    def __init__(self, token):
        """
        :type token: str
        """
        self.log = logging.getLogger('yql_helper')
        self.yql_client = YqlClient(db=self.DEFAULT_YQL_DB, token=token)

    def fetch_times(self, start_time):
        """
        :type start_time: int
        :rtype: list
        """
        dt = datetime.datetime.fromtimestamp(start_time)
        day = dt.strftime('%Y_%m_%d')
        day_before = (dt - datetime.timedelta(days=1)).strftime('%Y_%m_%d')
        if dt.hour < 2:
            query = YQL_QUERY_TMPL.format(date=day_before, timestamp=start_time)
        else:
            query = YQL_QUERY_TMPL.format(date=day, timestamp=start_time)
        self.log.info('run yql query: %s', query)
        request = self.yql_client.query(query)
        request.run()
        times = []
        for table in request.get_results():
            table.fetch_full_data()
            for row in table.rows:
                value = row[0]
                if not value:
                    # Sometimes it doesn't exist
                    continue
                if isinstance(row[0], str):
                    # Sometimes it's presented as string
                    try:
                        value = eval(row[0])
                    except Exception:
                        continue
                if value['prefix']:
                    prefix = ' '.join(value['prefix'])
                    if prefix == YA_MAKE_PREFIX:
                        duration = value['duration']
                        times.append(duration)
        return times


class Application(object):
    DEFAULT_RUN_INTERVAL = 3600

    def __init__(self, yql_token, run_interval, state_file, yasm_conf):
        self.log = logging.getLogger('app')
        self.yql_helper = YqlHelper(token=yql_token)
        self.run_interval = run_interval or self.DEFAULT_RUN_INTERVAL
        self.state_file = state_file
        self.yasm_conf = yasm_conf
        self.enable_yasm = self.yasm_conf and self.yasm_conf.get('enable')

    def run(self):
        last_state = self.get_last_state()
        if last_state is not None:
            last_run_time = last_state['last_run_time']
            time_to_wait = self.run_interval - (int(time.time()) - last_run_time)
            if time_to_wait > 0:
                self.log.info('Wait %s secs since last run', time_to_wait)
                time.sleep(time_to_wait)

        while True:
            self.log.info('Start new iteration')
            now = int(time.time())
            times = self.yql_helper.fetch_times(now - self.run_interval)
            self.log.info('New make timings: %s', len(times))
            if self.enable_yasm:
                self.push_to_yasm({'duration_ahhh': self.get_histogram(times)})
            self.write_state(now)
            self.log.info('Wait %s secs before new iteration', self.run_interval)
            time.sleep(self.run_interval)

    def write_state(self, now):
        """
        :type now: int
        """
        state = {'last_run_time': now}
        with open(self.state_file, 'w') as f:
            f.write(json.dumps(state))

    def get_last_state(self):
        """
        :rtype: dict
        """
        if not os.path.exists(self.state_file):
            return None
        with open(self.state_file) as f:
            content = f.read()
        return json.loads(content)

    def push_to_yasm(self, values):
        """
        :type values: dict
        """
        signals = [{'name': name, 'val': value} for name, value in values.items()]
        data = {
            "tags": self.yasm_conf.get('tags'),
            "values": signals,
            "ttl": self.yasm_conf.get('ttl'),
        }
        resp = requests.post(self.yasm_conf.get('url'), data=json.dumps([data]))
        resp.raise_for_status()

    @staticmethod
    def get_histogram(values, base=1.5, size=50):
        """
        :type values: list
        :type base: float
        :type size: int
        :rtype: list
        """
        values = sorted(values)
        max_idx = int(math.floor(math.log(values[-1], base)))
        min_idx = int(math.floor(math.log(values[0], base)))
        min_idx = max(min_idx, max_idx - size + 1)

        buckets = [[base**i, 0] for i in range(min_idx, max_idx+1)]
        buckets[0][0] = values[0]
        for i in values:
            idx = max(0, int(math.floor(math.log(i, base)) - min_idx)) if i else 0
            buckets[idx][1] += 1
        return buckets
