#!/usr/bin/python
import os
import re
import sys
import errno
import hashlib
import logging
import shutil
import socket
import subprocess
import traceback
import urlparse

from library.python import resource
from sandbox.common.proxy import OAuth
from sandbox.common.rest import Client as SandboxClient

SANDBOX_TIMEOUT = 60 * 5
DEFAULT_TASK_OWNER = 'QEMU_BACKUP'
DOWNLOAD_SPEED = 40
SEC_ADDR = 'nessus-3.sec.yandex.net'
IP6TABLES_COMMANDS = [
    'sudo ip6tables -A FORWARD -p tcp -m tcp --dport 22 --syn -j LOG',
    'sudo ip6tables -I FORWARD -p tcp -m tcp --dport 22 -s {} -j ACCEPT'.format(SEC_ADDR)  # QEMUKVM-1693
]
MEGA = 1024 ** 2

log = logging.getLogger('helpers')


def get_image_size(path):
    return os.stat(path).st_size


def get_storage_size(path):
    result = os.statvfs(path)
    block_size = result.f_frsize
    total_blocks = result.f_blocks
    total_size = total_blocks * block_size
    return total_size


def log_msg(msg):
    sys.stderr.write(msg + "\n")
    log.warning(msg)


def log_trace(*args, **kwargs):
    traceback.print_tb(*args, **kwargs)


def get_file_md5(path):
    if not os.access(path, os.R_OK):
        return None

    m = hashlib.md5()
    with open(path) as f:
        s = f.read(1048576)
        while len(s) > 0:
            m.update(s)
            s = f.read(1048576)

    return m.hexdigest()


def gen_vnc_password():
    with open("/dev/urandom", "r") as f:
        m = hashlib.sha256()
        m.update(f.read(64))
        return m.hexdigest()[0:8]


def purge_dir_contents(dirpath):
    try:
        subprocess.check_output(["sudo", "find", dirpath, "-mindepth", "1", "-delete"])

    except subprocess.CalledProcessError as e:
        log_msg("error purging {}: {} {}".format(dirpath, e.returncode, e.output))


def remove_if_exists(path, recursive=False):
    # Drop file/symlink/folder suppressing exception if
    # we catch ENOENT
    try:
        if not recursive:
            while os.path.islink(path):
                # read and drop all intermediate symlinks, leaving last path as realpath
                next_path = os.readlink(path)
                if not os.path.isabs(next_path):
                    next_path = os.path.join(os.path.dirname(path), next_path)
                os.unlink(path)
                path = next_path
            os.unlink(path)
        else:
            shutil.rmtree(path)
    except OSError as e:
        if e.errno != errno.ENOENT:
            raise


def ping_vm(vm_ip, timeout=1):
    try:
        subprocess.check_output(
            [
                "/bin/ping6", "-c", "1", "-W",
                "{}".format(int(timeout)), vm_ip
            ]
        )

        return True

    except subprocess.CalledProcessError:
        return False


def connect_vm(vm_ip, timeout=0.1):
    try:
        sk = socket.socket(socket.AF_INET6, socket.SOCK_STREAM)
        sk.settimeout(0.1)
        sk.connect((vm_ip, 22))
        sk.close()
        return True

    except Exception:
        return False


def create_ip6tables_ssh_syn_rule():
    for command in IP6TABLES_COMMANDS:
        try:
            check_command = command.replace('-I', '--check').replace('-A', '--check')
            subprocess.check_output(check_command.split())
        except subprocess.CalledProcessError as e:
            log_msg('Checking command: "{}", exit code: {}'.format(command, e.returncode))
            try:
                subprocess.check_output(command.split())
            except subprocess.CalledProcessError as e:
                log_msg("Can't create ip6tables ssh syn rule: exit code: {}, {}".format(e.returncode, e.output))


_count_packets_pattern = re.compile(r'\-\c\s([0-9]+)\s([0-9]+)')


def get_ip6tables_ssh_syn_rule_count():  # type: () -> int
    try:
        result = subprocess.check_output(['sudo', 'ip6tables', '--list-rules', '-v'])
    except subprocess.CalledProcessError as e:
        log_msg("Can't get ssh syn count: {}, {}".format(e.returncode, e.output))
        return None
    ssh_syn_rule_prefix = '-A FORWARD -p tcp -m tcp --dport 22 --tcp-flags FIN,SYN,RST,ACK SYN '
    for line in result.split('\n'):
        if line.startswith(ssh_syn_rule_prefix):
            res = _count_packets_pattern.findall(line)
            return int(res[0][0])
    log_msg("Can't find ssh syn count rule in {}".format(result))


def get_sandbox_task_url(client, task_id):
    return urlparse.SplitResult(
        scheme='https',
        netloc=client.host,
        path=str(client.task[task_id]),
        query='',
        fragment=''
    ).geturl()


def run_sandbox_merge_task(image_url, delta_url, size, dc):
    """
    :type image_url: str
    :type delta_url: str
    :type size: int
    :type dc: str
    """
    robot_token = resource.find('/secrets/sandbox_token').rstrip()
    client = SandboxClient(auth=OAuth(robot_token), total_wait=SANDBOX_TIMEOUT)
    kill_timeout = max(calculate_time(size, DOWNLOAD_SPEED) * 2, 3 * 60 * 60)
    task_id = client.task.create({
        'type': 'MERGE_QEMU_IMAGE',
        'owner': DEFAULT_TASK_OWNER,
        'description': 'Merge delta to original QEMU image',
        'priority': ('SERVICE', 'HIGH'),
        'custom_fields': [
            {
                'name': 'image_url',
                'value': image_url,
            },
            {
                'name': 'delta_url',
                'value': delta_url
            }
        ],
        'kill_timeout': kill_timeout,
        'requirements': {
            'disk_space': size * 2,
            'client_tags': 'LINUX & {}'.format(dc)
        }
    })['id']
    client.batch.tasks.start.update([task_id])
    return get_sandbox_task_url(client, task_id)


def run_sandbox_upload_task(image_url, size, dc):
    """
    :type image_url: str
    :type size: int
    :type dc: str
    """
    robot_token = resource.find('/secrets/sandbox_token').rstrip()
    client = SandboxClient(auth=OAuth(robot_token), total_wait=SANDBOX_TIMEOUT)
    kill_timeout = max(calculate_time(size, DOWNLOAD_SPEED), 3 * 60 * 60)
    task_id = client.task.create({
        'type': 'COPY_VM_IMAGE',
        'owner': DEFAULT_TASK_OWNER,
        'description': 'Share Qemu image with Sandbox',
        'priority': ('SERVICE', 'HIGH'),
        'custom_fields': [
            {
                'name': 'image_url',
                'value': image_url,
            },
        ],
        'kill_timeout': kill_timeout,
        'requirements': {
            'disk_space': size + 1024 ** 3,
            'client_tags': 'GENERIC & {}'.format(dc)
        }
    })['id']
    client.batch.tasks.start.update([task_id])
    return get_sandbox_task_url(client, task_id)


def calculate_time(size, download_speed):
    """
    :type size: int
    :type download_speed: int
    """
    mb_speed = download_speed / 8
    size_mb = size / 1024 ** 2
    return size_mb / mb_speed


def bytes_to_megabytes(b):
    """
    :type b: int
    rtype: int
    """
    return int(b / MEGA)


def megabytes_to_bytes(mega_bytes):
    """
    :type mega_bytes: int
    rtype: int
    """
    return mega_bytes * MEGA


def round_bytes(b):
    """
    QEMUKVM-1792
    :type b: int
    rtype: int
    """
    mb = bytes_to_megabytes(b)
    return megabytes_to_bytes(mb)
