# !/usr/bin/python

import os
import sys
import time
import Queue
import logging
from contextlib import contextmanager

from threading import Thread
from collections import deque
from infra.qyp.proto_lib import vmagent_pb2, vmagent_api_pb2, qdm_pb2

from infra.qyp.vmagent.src import helpers
from infra.qyp.vmagent.src import volume_manager
from infra.qyp.vmagent.src.helpers import log_trace, log_msg

SHUTDOWN_TIMEOUT_MS = 5000

EMPTY = vmagent_pb2.VMState.EMPTY
CONFIGURED = vmagent_pb2.VMState.CONFIGURED
STOPPED = vmagent_pb2.VMState.STOPPED
RUNNING = vmagent_pb2.VMState.RUNNING
BUSY = vmagent_pb2.VMState.BUSY
PREPARING = vmagent_pb2.VMState.PREPARING
CRASHED = vmagent_pb2.VMState.CRASHED
INVALID = vmagent_pb2.VMState.INVALID
READY_TO_START = vmagent_pb2.VMState.READY_TO_START


def state_str(state_type):
    return vmagent_pb2.VMState.VMStateType.Name(state_type)


def find_config(config_path, main_volume_name, main_volume_mount_path):
    """
    Parses configuration from last saved file
    """
    if os.path.exists(config_path):
        config_pb = vmagent_pb2.VMConfig()

        with open(config_path) as f:
            config_pb.ParseFromString(f.read())

        # backward compatibility with main volume in disk
        if config_pb.HasField('disk'):
            if config_pb.volumes:
                raise ValueError("WTF!! vm config with volumes can't have not empty .disk struct")
            main_volume = config_pb.volumes.add()  # type: vmagent_pb2.VMVolume
            main_volume.resource_url = config_pb.disk.resource.rb_torrent
            main_volume.is_main = True
            main_volume.vm_mount_path = '/'
            main_volume.name = main_volume_name
            main_volume.mount_path = main_volume_mount_path
            main_volume.image_type = config_pb.disk.type
            main_volume.order = 0
            config_pb.ClearField('disk')
        return config_pb


def find_last_state(last_state_file_path):
    if os.path.exists(last_state_file_path):
        state_pb = vmagent_pb2.VMState()
        with open(last_state_file_path) as f:
            state_pb.ParseFromString(f.read())
        return state_pb


class VMState(object):
    INVALIDATE_BACKUP_STATES = frozenset([RUNNING, PREPARING])
    INCREMENT_GENERATION_STATES = frozenset([RUNNING, PREPARING])
    SKIP_DUMP_TO_FILE_STATES = frozenset([BUSY])

    def __init__(self, current_state_file_path, log=None):
        self.log = log or logging.getLogger('vm-state')
        self._current_state_file_path = current_state_file_path
        self.actions = deque()
        self.target_state_types = deque()
        self._state_pb = vmagent_pb2.VMState()

    @property
    def actions_repr(self):
        return '->'.join(self.actions)

    def set(self, state_type=None, info=None, link_alive=None, net_alive=None, backup=None, rescue=None,
            target_state_type=None, generation=None, shared_image=None):
        """
        change state_type clear info message if no set
        if current state_type != BUSY, set trigger write in file (dump current state)
        """
        dump_to_file = False
        clear_backup = False
        increment_generation = False
        if state_type is not None and self._state_pb.type != state_type:
            prev_type_str = self.type_str
            self._state_pb.type = state_type
            info = info or ''  # clear info message if change state_type
            dump_to_file = state_type not in self.SKIP_DUMP_TO_FILE_STATES
            clear_backup = state_type in self.INVALIDATE_BACKUP_STATES and backup is None
            increment_generation = state_type in self.INCREMENT_GENERATION_STATES and generation is None
            self.log.info(
                'State Changed: {} -> {}, by {}'.format(prev_type_str, self.type_str, self.actions_repr))

        if rescue is not None:
            self._state_pb.rescue = rescue

        if target_state_type is not None:
            self._state_pb.target_type = target_state_type
            dump_to_file = True

        if info is not None:
            self._state_pb.info = info
            if info:
                self.log.info('Update Info by {}: {}'.format(self.actions_repr, info))

        if backup is not None:
            self._state_pb.backup.CopyFrom(backup)
            dump_to_file = True

        if shared_image is not None:
            self._state_pb.shared_image.CopyFrom(shared_image)

        if link_alive is not None:
            self._state_pb.link_alive = link_alive

        if net_alive is not None:
            self._state_pb.net_alive = net_alive

        if clear_backup:
            self._state_pb.backup.image_url = ''
            self._state_pb.backup.delta_url = ''
            self._state_pb.shared_image.image_url = ''
            self._state_pb.shared_image.delta_url = ''

        if increment_generation:
            generation = generation or self._state_pb.generation
            generation += 1

        if generation:
            self._state_pb.generation = generation

        if dump_to_file:
            with open(self._current_state_file_path, 'w') as f:
                f.write(self._state_pb.SerializeToString())

    def get(self):
        """

        :rtype: vmagent_pb2.VMState
        """
        res = vmagent_pb2.VMState()
        res.CopyFrom(self._state_pb)
        return res

    @property
    def type_str(self):
        return state_str(self._state_pb.type)

    @property
    def type(self):
        return self._state_pb.type

    @property
    def has_backup(self):
        return self._state_pb.HasField('backup')

    @property
    def is_empty(self):
        return self._state_pb.type == EMPTY

    @property
    def is_crashed(self):
        return self._state_pb.type == CRASHED

    @property
    def is_invalid(self):
        return self._state_pb.type == INVALID


class VMWorker(object):
    """
    Autorun Enabled:
        - check qemu start vm if CRASHED and READY_TO_START
        - config qemu start vm if RUNNING EMPTY READY_TO_START

    """
    DEFAULT_GRACEFUL_STOP_TIMEOUT = 10 * 60
    DEFAULT_RESTART_TIMEOUT = 5 * 60
    DEFAULT_ISS_HOOK_STOP_TIMEOUT = 5 * 60 - 20

    SLEEP_BEFORE_START_CRASHED_VM = 20

    def __init__(self, context, qemu_ctl, qemu_launcher, qemu_img_cmd, resource_manager, log=None):
        """

        :type context: infra.qyp.vmagent.src.config.VmagentContext
        :type qemu_ctl: infra.qyp.vmagent.src.qemu_ctl.QemuCtl
        :type qemu_launcher: infra.qyp.vmagent.src.qemu_launcher.QEMULauncher
        :type qemu_img_cmd: infra.qyp.vmagent.src.volume_manager.QEMUImgCmd
        :type resource_manager: infra.qyp.vmagent.src.resource_manager.ResourceManager
        """
        self.log = log or logging.getLogger('VMWorker')
        self._shutdown_needed = False

        self._context = context
        self._queue = Queue.Queue(maxsize=1)
        self._state = VMState(self._context.LAST_STATUS_FILE_PATH)
        self._config = vmagent_pb2.VMConfig()
        self._qemu_ctl = qemu_ctl
        self._qemu_img_cmd = qemu_img_cmd
        self._resource_manager = resource_manager
        self._qemu_launcher = qemu_launcher
        self._process = None

    @contextmanager
    def state(self, action, target_state_type=None):
        """

        :type action: str
        :type target_state_type: str
        """
        self._state.actions.append(action)
        self._state.target_state_types.append(target_state_type)
        if target_state_type is not None and len(self._state.target_state_types) == 1:
            self._state.set(target_state_type=target_state_type)
        yield self._state
        self._state.target_state_types.pop()
        self._state.actions.pop()

    def _store_config(self, config_pb):
        """

        :type config_pb: vmagent_pb2.VMConfig
        """
        self._config = config_pb
        with open(self._context.CURRENT_CONFIG_FILE_PATH, "w") as f:
            f.write(config_pb.SerializeToString())

    def get_config(self):
        """

        :rtype: vmagent_pb2.VMConfig
        """
        return self._config

    def get_state(self):
        """

        :rtype: vmagent_pb2.VMState
        """
        return self._state.get()

    def find_issues(self, vm_state):
        """

        :type vm_state: vmagent_pb2.VMState
        """
        if self._config:
            # try find volume available size issue
            for volume_pb in self._config.volumes:
                volume = volume_manager.VolumeWrapper(volume_pb)
                diff_virt, diff_file, = 0, 0
                delta_virt_size, image_virt_size = 0, 0
                delta_file_size, image_file_size = 0, 0

                if volume.image_type_is_delta and volume.image_file_exist and volume.delta_file_exist:
                    image_file_size = helpers.get_image_size(volume.image_file_path)
                    _, delta_virt_size = self._qemu_img_cmd.get_virtual_disk_size(volume.delta_file_path)
                    delta_file_size = helpers.get_image_size(volume.delta_file_path)
                    diff_virt = volume.available_size - delta_virt_size - image_file_size
                    diff_file = volume.available_size - delta_file_size - image_file_size

                elif volume.image_type_is_raw and volume.image_file_exist:
                    image_file_size = helpers.get_image_size(volume.image_file_path)
                    _, image_virt_size = self._qemu_img_cmd.get_virtual_disk_size(volume.image_file_path)
                    diff_virt = volume.available_size - image_virt_size
                    diff_file = volume.available_size - image_file_size

                if diff_virt != 0 or diff_file < 0:
                    issue = vm_state.issues.add()  # type: vmagent_pb2.VMIssue
                    issue.volume_available_size.volume_name = volume.name
                    issue.volume_available_size.delta_virt_size = delta_virt_size
                    issue.volume_available_size.delta_file_size = delta_file_size
                    issue.volume_available_size.image_file_size = image_file_size
                    issue.volume_available_size.image_virt_size = image_virt_size
                    issue.volume_available_size.diff_virt = diff_virt
                    issue.volume_available_size.diff_file = diff_file

    def init(self):
        """
        External Events:
          - Pod First Start
          - Pod Restart (static resource updated, host hard reboot, host graceful reboot)
          - Vmagent Unexpectedly Stopped (manual killed, oom kill)
          - Vmagent Graceful Restarted (Update Vmagent)

        last_state_target != EMPTY and last_state != last_state_target
            log this event

        Current State: STOPPED:
            last_state: ANY, last_state_target: ANY -> STOPPED

        Current State: CRASHED:
            last_state: ANY, last_state_target: ANY -> CRASHED (_handle_config and _check_qemu handle this state)

        Current State: EMPTY (first _check_qemu not found container)
            last_state: RUNNING, last_state_target: STOPPED  -> STOPPED
            last_state: STOPPED, last_state_target: STOPPED -> STOPPED
            last_state: STOPPED, last_state_target: EMPTY (prev vmagent) -> STOPPED (log this event)

            last_state: ANY_OTHER, last_state_target: ANY_OTHER -> EMPTY (_handle_config
                                                                          and _check_qemu handle this state)

        Current State: RUNNING:
            last_state: RUNNING, last_state_target: STOPPED  -> RUNNING (log this event)
            last_state: STOPPED, last_state_target: STOPPED -> RUNNING (log this event)
            last_state: STOPPED, last_state_target: EMPTY (prev vmagent) -> (log this event)

            last_state: ANY_OTHER, last_state_target: ANY_OTHER -> RUNNING (_handle_config and
                                                                         _check_qemu handle this state)
        """
        with self.state(action='init') as state:
            self.log.info('Starting worker...')

            # Check prev config exists
            last_config = find_config(
                config_path=self._context.CURRENT_CONFIG_FILE_PATH,
                main_volume_name=self._context.MAIN_QEMU_VOLUME_NAME,
                main_volume_mount_path=self._context.DEFAULT_MAIN_STORAGE_PATH
            )
            if last_config:
                self.log.info('Config found in {} file: {}'.format(self._context.CURRENT_CONFIG_FILE_PATH, last_config))
                self._store_config(last_config)
            else:
                self.log.info('Config not found in {} file'.format(self._context.CURRENT_CONFIG_FILE_PATH))

            # Check prev state exists
            last_state_pb = find_last_state(self._context.LAST_STATUS_FILE_PATH)
            if last_state_pb:
                self.log.info('Last State found in {} file : {}'.format(
                    self._context.LAST_STATUS_FILE_PATH, last_state_pb))
            else:
                self.log.info('Last State not found in {} file'.format(
                    self._context.LAST_STATUS_FILE_PATH))
                last_state_pb = vmagent_pb2.VMState()

            # Check Current State of Qemu Container
            info = None
            prev_state = last_state_pb.type
            prev_target = last_state_pb.target_type
            backup = last_state_pb.backup if last_state_pb.HasField('backup') else None

            if prev_state != EMPTY and prev_state != prev_target:
                self.log.warn('Vmagent unexpectedly crashed with state: {} and target_state: {}'.format(
                    state_str(last_state_pb.type),
                    state_str(last_state_pb.target_type)
                ))

            try:
                self._qemu_ctl.check()
            except self._qemu_ctl.QemuContainerDoesNotExists:
                current_state = EMPTY

                if prev_state == READY_TO_START:
                    current_state = READY_TO_START

                if prev_state == RUNNING:
                    if prev_target == STOPPED:
                        current_state = STOPPED
                    else:
                        self.log.warn('Found last state RUNNING, assuming unexpected stop')
                if prev_state == STOPPED and prev_target in (STOPPED, EMPTY):
                    current_state = STOPPED

            except self._qemu_ctl.QemuContainerStopped:
                current_state = STOPPED
            except self._qemu_ctl.QemuContainerCrashed as e:
                current_state = CRASHED
                info = e.message
            else:
                current_state = RUNNING

            state.set(
                state_type=current_state,
                info=info,
                backup=backup,
                generation=last_state_pb.generation,
                link_alive=helpers.ping_vm(self._context.VM_IP),
                net_alive=helpers.connect_vm(self._context.VM_IP)
            )

            task = vmagent_api_pb2.VMActionRequest()
            task.action = vmagent_api_pb2.VMActionRequest.PUSH_CONFIG
            task.config.CopyFrom(self._context.VM_CONFIG)

            error = self.push_task(task)
            if error:
                self.log.warn('Config error: {}'.format(error))
                state.set(info='Config error: {}'.format(error))
            if state.type == EMPTY and error:
                state.set(state_type=INVALID, info='Config error: {}'.format(error))

    def is_alive(self):
        return self._process and self._process.isAlive()

    def get_data_transfer_state(self):
        res = vmagent_pb2.DataTransferState()
        if not self.is_alive():
            return res
        if not self._state.actions:
            return res
        action = self._state.actions[-1]
        # for alive worker check state and last appended action:
        # BUSY(qdm_upload) - uploading backup with mds, PREPARING(config) - downloading
        # fill status and operation in case if process did not create progress file yet
        if self._state.type == BUSY and action == 'qdm_upload':
            res.operation = vmagent_pb2.DataTransferState.OperationType.UPLOAD
        elif self._state.type == PREPARING and action == 'config':
            res.operation = vmagent_pb2.DataTransferState.OperationType.DOWNLOAD
        else:
            return res

        self._fill_data_transfer_state(res)
        return res

    def _fill_data_transfer_state(self, bckp):
        bckp.is_in_progress = True
        resource_status = self._resource_manager.get_backup_status()
        if resource_status:
            bckp.progress = resource_status['progress']
            bckp.total_bytes = resource_status['total_bytes']
            bckp.done_bytes = resource_status['done_bytes']
            start_time, duration = resource_status.get('start_time', 0), resource_status.get('duration', 0)
            if start_time and duration:
                eta = duration - (time.time() - start_time)
                bckp.eta = max(int(eta), 1)  # set eta to 1 second if backup is taking longer than expected

    def start(self):
        if not self.is_alive():
            self._process = Thread(target=self.process)
            self._process.start()

    def stop(self):
        if self.is_alive():
            self._shutdown_needed = True
            self._process.join()
            self._process = None

    def on_iss_hook_stop(self):
        """
        Described in QEMUKVM-469
        1. Stop task processing
        2. Try shutdown VM until it stops
        3. Destroy container
        4. Dont change status, so worker will start vm during next run
        """
        prev_state = self._state.type
        target_state_type = READY_TO_START if prev_state == RUNNING else STOPPED
        self.stop()
        self._resource_manager.on_iss_hook_stop()
        self.log.info('Worker has ben stopped successfully')
        with self.state(action='iss_hook_stop', target_state_type=target_state_type) as state:
            self._shutdown_qemu(graceful_stop_timeout=self.DEFAULT_ISS_HOOK_STOP_TIMEOUT)
            self.log.info('QEMU has ben stopped successfully')
            state.set(state_type=target_state_type)

    def process(self):
        while not self._shutdown_needed:
            try:
                try:
                    task = self._queue.get(timeout=0.1)
                    self._handle_cmd(task)
                except Queue.Empty:
                    pass

                self._check_qemu()
            except Exception as e:
                log_msg('Unhandled exception in worker thread: {}'.format(e.message))
                log_trace(sys.exc_info()[2])
                time.sleep(5)

    def _check_qemu(self):
        with self.state(action='check') as state:
            try:
                self._qemu_ctl.check()
            except self._qemu_ctl.QemuContainerDoesNotExists:
                if state.type in (CRASHED, READY_TO_START) and self._config.autorun:
                    self.log.info('Autorun Enabled, Current state = {}, starting VM'.format(state.type_str))

                    if state.type == CRASHED:
                        wait_seconds = self.SLEEP_BEFORE_START_CRASHED_VM
                        self.log.info('Wait {} seconds before try start CRASHED vm...'.format(wait_seconds))
                        while not self._shutdown_needed and wait_seconds > 0:
                            time.sleep(1)
                            wait_seconds -= 1
                        if self._shutdown_needed:
                            return
                    self._start_qemu()
            except self._qemu_ctl.QemuContainerStopped:
                state.set(STOPPED)
            except self._qemu_ctl.QemuContainerCrashed as e:
                state.set(
                    state_type=CRASHED,
                    info=e.message
                )
            else:
                state.set(
                    state_type=RUNNING,
                    link_alive=helpers.ping_vm(self._context.VM_IP),
                    net_alive=helpers.connect_vm(self._context.VM_IP)
                )

    class RequestBadAction(Exception):
        pass

    def push_task(self, task):
        """

        :type task: vmagent_api_pb2.VMActionRequest
        :rtype: str | None
        """
        try:
            self.check_request(task)
        except self.RequestBadAction as e:
            return e.message

        try:
            self._queue.put(task, block=False)
        except Queue.Full:
            return "Request already pending"

        return None

    def check_request(self, task):
        if task.action == vmagent_api_pb2.VMActionRequest.PUSH_CONFIG:
            if not task.HasField('config'):
                raise self.RequestBadAction('Config empty')
            try:
                main_volume = next((v for v in task.config.volumes if v.is_main))
            except StopIteration:
                raise self.RequestBadAction('Invalid volumes, is empty')

            if not main_volume.resource_url:
                raise self.RequestBadAction('Invalid main image resource_url')

            if task.config.vcpu <= 0:
                raise self.RequestBadAction("Invalid vcpu count")

            if task.config.mem <= 0:
                raise self.RequestBadAction("Invalid mem demand")

            if task.config.type == vmagent_pb2.VMConfig.WINDOWS and not self._context.WINDOWS_READY:
                raise self.RequestBadAction("Cannot start Windows workload on this agent")

        return None

    def _handle_cmd(self, task):
        if task.action == vmagent_api_pb2.VMActionRequest.PUSH_CONFIG:
            self._handle_config(task.config, task.timeout)
            return

        try:
            # Abort any background backups if any
            self._resource_manager.bg_abort()

            if task.action == vmagent_api_pb2.VMActionRequest.QDMUPLOAD:
                self._handle_qdm_upload(task.qdmreq)
            elif task.action == vmagent_api_pb2.VMActionRequest.QDMUPLOAD_BG:
                self._handle_qdm_upload_bg(task.qdmreq)
            else:
                self._handle_command(task.action, task.timeout)
        except self.RequestBadAction as e:
            log_msg(e.message)
        except Exception as e:
            log_trace(sys.exc_info()[2])
            log_msg(e.message)
            # self._state.set(CRASHED, "last command execution error".format(e.message))

    def _handle_command(self, action, timeout):
        if action == vmagent_api_pb2.VMActionRequest.START:
            self._start_qemu()

        elif action == vmagent_api_pb2.VMActionRequest.SHUTDOWN:
            self._shutdown_qemu()

        elif action == vmagent_api_pb2.VMActionRequest.RESTART:
            self._restart_qemu()

        elif action == vmagent_api_pb2.VMActionRequest.RESET:
            self._reset_qemu()

        elif action == vmagent_api_pb2.VMActionRequest.POWEROFF:
            self._power_off_qemu()

        elif action == vmagent_api_pb2.VMActionRequest.HARD_RESET:
            self._revert_qemu()

        elif action == vmagent_api_pb2.VMActionRequest.RESCUE:
            self._start_qemu(rescue=True)

        elif action == vmagent_api_pb2.VMActionRequest.BACKUP:
            self._backup()

        elif action == vmagent_api_pb2.VMActionRequest.SHARE_IMAGE:
            self._share_image()

        else:
            raise self.RequestBadAction("Unknown action")

    def _handle_config(self, config_pb, timeout=None, force_configure_qemu=False, target_state_type=None):
        """

        :type config_pb: vmagent_pb2.VMConfig
        :type timeout: int | None
        :type force_configure_qemu: bool
        :type target_state_type: int
        :rtype:
        """
        prev_config = self._config
        need_configure_qemu = force_configure_qemu or (self._state.type in (EMPTY, INVALID))

        if config_pb.vcpu != prev_config.vcpu:
            self.log.info('VCPU has been changed: {} -> {}, reboot required!'.format(
                prev_config.vcpu, config_pb.vcpu,
            ))
            need_configure_qemu = True

        if config_pb.mem != prev_config.mem:
            self.log.info('Memory has been changed {} -> {}, reboot required!'.format(
                prev_config.mem, config_pb.mem
            ))
            need_configure_qemu = True

        if config_pb.audio != prev_config.audio:
            self.log.info('Audio has been changed {} -> {}, reboot required!'.format(
                vmagent_pb2.VMConfig.AudioType.Name(prev_config.audio), vmagent_pb2.VMConfig.AudioType.Name(config_pb.audio)
            ))
            need_configure_qemu = True

        if config_pb.autorun != prev_config.autorun:
            self.log.info('Autorun has been changed: {} -> {}, nothing side effects'.format(
                prev_config.autorun, config_pb.autorun
            ))

        current_volumes = {v.name: volume_manager.VolumeWrapper(v) for v in prev_config.volumes}
        new_volumes = {v.name: volume_manager.VolumeWrapper(v) for v in config_pb.volumes}

        if current_volumes != new_volumes:
            self.log.info('Volumes has been changed, reboot required!')
            need_configure_qemu = True

        config_pb.access_info.vnc_password = prev_config.access_info.vnc_password or helpers.gen_vnc_password()

        if not need_configure_qemu:
            self._store_config(config_pb)
            return

        if target_state_type is None:
            if not config_pb.autorun or self._state.type == STOPPED:
                target_state_type = CONFIGURED
            else:
                target_state_type = RUNNING

        with self.state('config', target_state_type=target_state_type) as state:
            self.log.info('Applying config with target state: {} \n{}'.format(
                state_str(target_state_type), config_pb))
            self._shutdown_qemu(graceful_stop_timeout=None)

            self._resource_manager.clear_data_transfer_progress()  # clear previous progress file if exists
            state.set(PREPARING)

            for volume_name, volume in new_volumes.items():
                prev_volume = current_volumes.get(volume_name)  # type: volume_manager.VolumeWrapper
                _volume_manager = volume_manager.VolumeManager(volume, self._resource_manager, self._qemu_img_cmd)
                state.set(info='Prepare volume: {}'.format(volume_name))
                if prev_volume:
                    self.log.info('Volume:{} already exists, try fix images path'.format(volume.name))
                    try:
                        fixed_locations = _volume_manager.fix_image_location()
                        if fixed_locations:
                            self.log.info('Volume:{} Fixed image locations {}'.format(
                                volume.name, fixed_locations))
                        else:
                            self.log.info('Volume:{} images path is valid'.format(volume.name))
                    except _volume_manager.BadAction as e:
                        state.set(INVALID, info="Volume:{}: {}".format(volume_name, e.message))
                        return
                else:
                    self.log.info('Volume:{} is new'.format(volume.name))

                should_init_image = not prev_volume
                should_remove_all_files = not prev_volume

                if prev_volume:
                    if prev_volume.resource_url != volume.resource_url:
                        # user change resource url
                        self.log.info('Volume:{} change resource_url \n {} -> {}'.format(
                            volume.name, prev_volume.resource_url, volume.resource_url))
                        should_init_image = True
                        should_remove_all_files = True
                    elif prev_volume.image_type != volume.image_type:
                        # user change image_type
                        self.log.info('Volume:{} change image_type \n {} -> {}'.format(
                            volume.name, prev_volume.image_type_str, volume.image_type_str))
                        should_init_image = True
                        should_remove_all_files = True
                    elif volume.image_type_is_raw and not volume.image_file_exist:
                        # user revert main volume with RAW or vm in state INVALID
                        self.log.info('Volume:{} image file not found: \n {}'.format(volume.name, volume))
                        should_init_image = True
                        should_remove_all_files = True
                    elif volume.image_type_is_delta and not volume.delta_file_exist:
                        # user revert main volume with DELTA or vm in state INVALID
                        self.log.info('Volume:{} delta file not found: \n {}'.format(volume.name, volume))
                        should_init_image = True
                        should_remove_all_files = not volume.image_file_exist

                if not should_init_image:
                    self.log.info('Volume:{} init process was skipped.'.format(volume.name))
                    continue

                if should_remove_all_files:
                    self.log.info('Volume:{} remove all files.'.format(volume.name))
                    _volume_manager.remove_all_files()

                try:
                    self.log.info('Volume:{} init process start'.format(volume.name))
                    _volume_manager.init_image()
                except _volume_manager.DownloadingImageError as e:
                    # We want to clean as much as possible -- all symlinks/files in direct paths
                    # and all data in image_folder/
                    # This is done to avoid running VM with incompletely downloaded images
                    if should_remove_all_files:
                        _volume_manager.remove_all_files()
                    state.set(INVALID, info="Volume:{}: {}".format(volume_name, e.message))
                    return
                except _volume_manager.ImageError as e:
                    state.set(INVALID, info="Volume:{}: {}".format(volume_name, e.message))
                    return
                except Exception as e:
                    self.log.exception(e)
                    self.log.error(e.message)
                    state.set(INVALID, info="Volume:{}: {}".format(volume_name, e.message))
                    return
                else:
                    self.log.info('Volume:{} init process finish'.format(volume.name))

            self._store_config(config_pb)
            state.set(CONFIGURED)

            if target_state_type == RUNNING:
                self._start_qemu()

    def _start_qemu(self, rescue=False):
        with self.state(action='start', target_state_type=RUNNING) as state:
            if state.is_empty or state.is_invalid:
                self.log.warn('Cannot start: vm in %s state', state.type_str)
                return
            state.set(BUSY)

            if self._context.REBUILD_QEMU_LAUNCHER:
                try:
                    self._qemu_launcher.build(self._context, self._config, rescue=rescue)
                except Exception as e:
                    self.log.exception(e)
                    self.log.error(e.message)
                    state.set(state_type=CRASHED, info=e.message)
            try:
                self._qemu_ctl.start(
                    qemu_launcher_path=self._context.QEMU_LAUNCHER_FILE_PATH,
                    cwd=self._context.WORKDIR,
                    vnc_password=self._config.access_info.vnc_password
                )
            except self._qemu_ctl.QemuContainerAlreadyExists:
                self.log.warn('VM already started')
                state.set(state_type=RUNNING)
            except self._qemu_ctl.QemuContainerCrashed as e:
                state.set(state_type=CRASHED, info=e.message)
            else:
                state.set(
                    state_type=RUNNING,
                    rescue=rescue,
                    link_alive=helpers.ping_vm(self._context.VM_IP),
                    net_alive=helpers.connect_vm(self._context.VM_IP)
                )

    def _power_off_qemu(self):
        with self.state(action='power_off', target_state_type=STOPPED) as state:
            state.set(BUSY)
            try:
                self._qemu_ctl.power_off()
            except self._qemu_ctl.QemuCtlRuntimeError as e:
                log_msg('Qemu Runtime Error:{}'.format(e.message))
            except self._qemu_ctl.QemuContainerDoesNotExists:
                log_msg('Qemu Container Does Not Exists')
            except self._qemu_ctl.QemuContainerCrashed as e:
                log_msg('Qemu crashed with error : {}'.format(e.message))

            state.set(
                state_type=STOPPED,
                link_alive=helpers.ping_vm(self._context.VM_IP),
                net_alive=helpers.connect_vm(self._context.VM_IP),
            )

    def _shutdown_qemu(self, graceful_stop_timeout=None):
        with self.state(action='shutdown', target_state_type=STOPPED) as state:
            graceful_stop_timeout = graceful_stop_timeout or self.DEFAULT_GRACEFUL_STOP_TIMEOUT

            state.set(state_type=BUSY,
                      info='Try shutdown qemu, power off after {} seconds'.format(graceful_stop_timeout))
            try:
                for left_seconds in self._qemu_ctl.try_graceful_stop(timeout=graceful_stop_timeout):
                    state.set(info='Try shutdown qemu, power off after {} seconds'.format(left_seconds))
            except self._qemu_ctl.QemuGracefulStopTimeout as e:
                log_msg('Qemu Graceful Stop Timeout Error: {}'.format(e))
                self._power_off_qemu()
            except self._qemu_ctl.QemuContainerDoesNotExists:
                log_msg('Qemu Container Does Not Exists')
                state.set(STOPPED)
            else:
                state.set(
                    state_type=STOPPED,
                    link_alive=helpers.ping_vm(self._context.VM_IP),
                    net_alive=helpers.connect_vm(self._context.VM_IP),
                )

    def _restart_qemu(self, timeout=None):
        with self.state(action='restart', target_state_type=RUNNING):
            self._shutdown_qemu(timeout)
            self._start_qemu()

    def _revert_qemu(self):
        with self.state(action='revert', target_state_type=RUNNING) as state:
            ret = None
            self._power_off_qemu()
            state.set(BUSY)
            main_volume = next((volume_manager.VolumeWrapper(v) for v in self._config.volumes if v.is_main))
            # TODO: what about extra volumes ?
            _vol_manager = volume_manager.VolumeManager(main_volume, self._resource_manager, self._qemu_img_cmd)

            if main_volume.image_type_is_raw:
                _vol_manager.remove_image()
            elif main_volume.image_type_is_delta:
                _vol_manager.remove_delta()

            self._handle_config(self._config,
                                force_configure_qemu=True,
                                target_state_type=RUNNING if self._config.autorun else None)

            return ret

    def _reset_qemu(self):
        with self.state(action='reset'):
            try:
                self._qemu_ctl.reset()
            except self._qemu_ctl.QemuContainerDoesNotExists:
                pass

    def _backup(self):
        with self.state('backup', target_state_type=STOPPED) as state:
            self._power_off_qemu()
            state.set(BUSY)
            # Stop virtual machine, you've been warned
            main_volume = next((volume_manager.VolumeWrapper(v) for v in self._config.volumes if v.is_main))
            backup = vmagent_pb2.Backup()
            if main_volume.image_type_is_delta:
                backup.image_url = self._resource_manager.share_resource(main_volume.image_file_path)
                backup.delta_url = self._resource_manager.share_resource(main_volume.delta_file_path)
            elif main_volume.image_type_is_raw:
                backup.image_url = self._resource_manager.share_resource(main_volume.image_file_path)
            backup.creation_time.GetCurrentTime()
            state.set(STOPPED, backup=backup)
            # wait until backup task get image urls from get_status
            time.sleep(3 * 60)

    def _prepare_qdm_backup_spec(self):
        qdm_backup_spec = qdm_pb2.QDMBackupSpec()
        qdm_backup_spec.qdm_spec_version = 1
        qdm_backup_spec.vmspec.CopyFrom(self._context.VM)
        volumes = [volume_manager.VolumeWrapper(v) for v in self._config.volumes]
        for volume_index, volume in enumerate(volumes):
            _vol_man = volume_manager.VolumeManager(volume, self._resource_manager, self._qemu_img_cmd)

            try:
                fixed_locations = _vol_man.fix_image_location()
                if fixed_locations:
                    self.log.info('Volume:{} Fixed image locations {}'.format(volume.name, fixed_locations))
            except _vol_man.BadAction as e:
                return False, 'Invalid volume {}: {}'.format(volume.name, e.message)

            file_result_dir = volume.name.replace('/', '')
            file_spec = qdm_backup_spec.filemap.add()  # type: qdm_pb2.QDMBackupFileSpec
            file_spec.path = '{}/layer.img'.format(file_result_dir)
            file_spec.source = volume.image_file_path
            file_spec.meta.volume_index = volume_index
            file_spec.meta.volume_name = volume.name

            if volume.image_type_is_delta:
                file_spec = qdm_backup_spec.filemap.add()  # type: qdm_pb2.QDMBackupFileSpec
                file_spec.path = '{}/current.qcow2'.format(file_result_dir)
                file_spec.source = volume.delta_file_path
                file_spec.meta.volume_index = volume_index
                file_spec.meta.volume_name = volume.name

        return True, qdm_backup_spec

    def _handle_qdm_upload(self, qdmreq):
        """

        :type qdmreq: qdm_pb2.QDMCliUploadRequest
        """

        with self.state('qdm_upload', target_state_type=STOPPED) as state:
            self._shutdown_qemu()

            success, result = self._prepare_qdm_backup_spec()
            if not success:
                state.set(INVALID, info=str(result))
                return
            else:
                qdm_backup_spec = result

            self._resource_manager.clear_data_transfer_progress()  # clear previous progress file if exists
            state.set(BUSY)

            try:
                start_at = time.time()
                resource_url = self._resource_manager.upload_resource(qdmreq.key, qdm_backup_spec)
                backup = vmagent_pb2.Backup()
                backup.image_url = resource_url
                backup.creation_time.GetCurrentTime()
                info = "Backup Success by {} seconds".format(int(time.time() - start_at))

            except self._resource_manager.ResourceUploadError as e:
                log_trace(sys.exc_info()[2])
                log_msg(e.message)
                info = "Backup Failed with error: {}".format(e.message)
                self.log.error(info)
                backup = None

            state.set(STOPPED, backup=backup, info=info)

    def _handle_qdm_upload_bg(self, qdmreq):
        success, result = self._prepare_qdm_backup_spec()
        if success:
            qdm_backup_spec = result
            self._resource_manager.upload_resource_bg(qdmreq.key, qdm_backup_spec)
        else:
            log_msg('Failed to execute qdm upload in background: {}'.format(result))

    def _share_image(self):
        with self.state('share_image', target_state_type=STOPPED) as state:
            self._shutdown_qemu(300)
            state.set(BUSY)
            main_volume = next((volume_manager.VolumeWrapper(v) for v in self._config.volumes if v.is_main))
            image = vmagent_pb2.SharedImage()
            if main_volume.image_type_is_delta:
                image.delta_url = self._resource_manager.share_resource(main_volume.delta_file_path)
                image.image_url = self._resource_manager.share_resource(main_volume.image_file_path)
            elif main_volume.image_type_is_raw:
                image.image_url = self._resource_manager.share_resource(main_volume.image_file_path)
            image.creation_time.GetCurrentTime()
            state.set(STOPPED, shared_image=image)

    def stop_backup(self):
        try:
            self._resource_manager.stop_qdmupload()
        except Exception as e:
            return e.message
