import sys
import time
import socket
import logging

from porto.exceptions import (SocketError as PortoSocketError,
                              ContainerDoesNotExist as PortoContainerNotExists)

from infra.qyp.vmagent.src.helpers import log_msg, log_trace


class QemuMonError(Exception):
    pass


class QemuMon(object):
    DEFAULT_TIMEOUT = 300

    def __init__(self, socket_file_path):
        """

        :type socket_file_path: str
        """
        self._socket_file_path = socket_file_path

    def __call__(self, cmd, timeout=None):
        timeout = timeout or self.DEFAULT_TIMEOUT
        deadline = time.time() + timeout
        now = time.time()
        exc = None

        while now < deadline:
            try:
                s = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
                s.settimeout(deadline - now)
                s.connect(self._socket_file_path)
                s.recv(1024)
                s.sendall(cmd + '\n')
                time.sleep(1)
                s.recv(1024)
                s.close()
                return

            except Exception as e:
                exc = e
                time.sleep(0.5)
                pass

            now = time.time()

        raise QemuMonError("Socket timeout: {}".format(exc))

    def system_powerdown(self, timeout=None):
        """
        Graceful stop Qemu Vm

        :type timeout: int
        """
        return self('system_powerdown', timeout=timeout)

    def system_reset(self, timeout=None):
        return self('system_reset', timeout=timeout)

    def power_off(self, timeout=None):
        return self('quit', timeout=timeout)

    def set_vnc_password(self, password, timeout=None):
        return self('set_password vnc {}'.format(password), timeout=timeout)


class QemuCtl(object):
    QEMU_CONTAINER_NAME = 'self/qemu'
    SHUTDOWN_TIMEOUT_MS = 5 * 1000

    def __init__(self, porto_connection, mon_socket_file_path):
        """

        :type porto_connection: porto.Connection
        :type mon_socket_file_path: str
        """
        self.log = logging.getLogger('qemu-ctl')
        self._porto_connection = porto_connection
        self._qemu_mon = QemuMon(mon_socket_file_path)

    def _get_qemu_container(self, new_one=False):
        """

        :type new_one: bool
        :rtype: porto.api.Container | None
        """
        try:
            r = self._porto_connection.Find(self.QEMU_CONTAINER_NAME)

            if not new_one:
                return r

            r.Destroy()

        except PortoContainerNotExists:
            if not new_one:
                return None

        r = self._porto_connection.Create(self.QEMU_CONTAINER_NAME)

        return r

    def check(self):
        qemu_container = self._get_qemu_container(new_one=False)
        if not qemu_container:
            raise self.QemuContainerDoesNotExists()

        try:
            wait_result = qemu_container.Wait(timeout=1000)  # ms here
            if "qemu" in wait_result:
                ret = qemu_container.GetProperty("exit_code")
                try:
                    qemu_container.Destroy()
                except PortoContainerNotExists:
                    pass

                if ret == "0":
                    raise self.QemuContainerStopped()
                else:
                    stderr = ''
                    try:
                        stderr = qemu_container.GetProperty('stderr')
                        qemu_container.Destroy()
                    except Exception:
                        pass
                    stderr = " ; ".join(stderr.split('\n')[-20:])
                    raise self.QemuContainerCrashed("qemu exit code: {}, stderr: {}".format(ret, stderr))
        except PortoContainerNotExists:
            raise self.QemuContainerCrashed("Qemu container dropped unexpectedly")
        except PortoSocketError as e:
            log_msg("Got porto socket error: {}".format(e))
            raise self.QemuContainerCrashed("Porto socket error")

    def start(self, qemu_launcher_path, cwd, vnc_password):
        qemu_container = self._get_qemu_container(new_one=False)
        if qemu_container:
            raise self.QemuContainerAlreadyExists()
        try:
            qemu_container = self._get_qemu_container(new_one=True)
            qemu_container.SetProperty("stdin_path", "/dev/null")
            qemu_container.SetProperty("command", " ".join(["/bin/bash", "-x", qemu_launcher_path]))
            qemu_container.SetProperty("cwd", cwd)

            qemu_container.Start()
            self._qemu_mon.set_vnc_password(vnc_password)

        except Exception as e:
            log_msg("Exception while starting: {}".format(e))
            log_trace(sys.exc_info()[2])

            # info = e.message
            stderr = ""

            if qemu_container:
                try:
                    stderr = qemu_container.GetProperty('stderr')
                    qemu_container.Destroy()
                except Exception:
                    pass

            raise self.QemuContainerCrashed("Failed to start qemu: {}, stderr: {}".format(
                e.message, " ; ".join(stderr.split('\n')[-20:])
            ))

    def reset(self):
        qemu_container = self._get_qemu_container(new_one=False)
        if not qemu_container:
            raise self.QemuContainerDoesNotExists()

        self._qemu_mon.system_reset()

    def try_graceful_stop(self, timeout=None):
        timeout = timeout or 5 * 60
        qemu_container = self._get_qemu_container(new_one=False)
        if not qemu_container:
            raise self.QemuContainerDoesNotExists()

        now = time.time()
        deadline = now + timeout
        qemu_stopped = False
        while now < deadline:
            try:
                self._qemu_mon.system_powerdown(timeout=10)
            except Exception as e:
                log_msg("Cannot send shutdown command: {}".format(e))
            try:
                wait_result = qemu_container.Wait(timeout=self.SHUTDOWN_TIMEOUT_MS)
                if 'qemu' in wait_result:
                    qemu_container.Destroy()
                    qemu_stopped = True
                    break
            except PortoContainerNotExists:
                self.log.warn('Qemu container dropped unexpectedly')
                qemu_stopped = True
                break
            now = time.time()
            yield round(deadline - now)

        if not qemu_stopped:
            raise self.QemuGracefulStopTimeout()

    def power_off(self, timeout=10):
        qemu_container = self._get_qemu_container(new_one=False)
        if not qemu_container:
            raise self.QemuContainerDoesNotExists()

        ret = None

        try:
            self._qemu_mon.power_off()

            wait_result = qemu_container.Wait(timeout=timeout * 1000)
            if 'qemu' not in wait_result:
                log_msg("Qemu mon quit timeout exceeded")
            else:
                ret = qemu_container.GetProperty('exit_code')

        except Exception as e:
            log_msg('Cannot stop porto with monitor: {}'.format(e))

        try:
            qemu_container.Destroy()
        except PortoContainerNotExists:
            pass

        if ret is None:
            try:
                qemu_container.Stop(timeout=timeout)
                ret = '9'
            except Exception as e:
                raise self.QemuCtlRuntimeError("Cannot stop qemu: {}".format(e))

        if ret != '0':
            raise self.QemuContainerCrashed(
                "Qemu exited with code: {}".format(ret)
            )

    class QemuCtlError(Exception):
        pass

    class QemuCtlRuntimeError(QemuCtlError):
        pass

    class QemuContainerDoesNotExists(QemuCtlError):
        pass

    class QemuContainerAlreadyExists(QemuCtlError):
        pass

    class QemuContainerStopped(QemuCtlError):
        pass

    class QemuGracefulStopTimeout(QemuCtlError):
        pass

    class QemuContainerCrashed(QemuCtlError):
        pass
