import os
import uuid
import shutil
import logging
import hashlib
from os import access, R_OK, W_OK
import library.python.resource

from jinja2 import Environment, PackageLoader, BaseLoader, TemplateNotFound, StrictUndefined

from infra.qyp.proto_lib import vmagent_pb2
from infra.qyp.vmagent.src import volume_manager
from infra.qyp.vmagent.src import helpers


class ArcadiaLoader(BaseLoader):
    def __init__(self, prefix):
        self._prefix = prefix

    def get_source(self, environment, template):
        for key, item in library.python.resource.iteritems(self._prefix, strip_prefix=True):
            if key == template:
                return item.decode('utf8'), key, None
        raise TemplateNotFound(template)

    def list_templates(self):
        results = []
        for template_name in library.python.resource.iterkeys(self._prefix, strip_prefix=True):
            results.append(template_name)
        results.sort()
        return results


Y_PYTHON_SOURCE_ROOT = os.environ.get('Y_PYTHON_SOURCE_ROOT')

if Y_PYTHON_SOURCE_ROOT:
    loader = PackageLoader('infra.qyp.vmagent.src', 'cfg_templates')
else:
    loader = ArcadiaLoader('resfs/file/cfg_templates/')

TEMPLATES_ENV = Environment(loader=loader, undefined=StrictUndefined)

log = logging.getLogger('qemu-launcher')


class QEMUSystemCmdBuilder(object):
    DEFAULT_BIN_PATH = '/usr/local/bin/qemu-system-x86_64'

    @staticmethod
    def check_dev_kvm():
        return access('/dev/kvm', R_OK) and access('/dev/kvm', W_OK)

    def __init__(self, bin_path=None, kvm_available=None):
        self._bin_path = bin_path or self.DEFAULT_BIN_PATH
        self._kvm_available = kvm_available if kvm_available is not None else self.check_dev_kvm()
        self._args_lines = [
            '-nographic',
            '-nodefaults',
        ]
        if self._kvm_available:
            self._args_lines.append('--enable-kvm')

    @property
    def bin_path(self):
        return self._bin_path

    def setup_graphics_card(self, card_type='std'):
        """

        :param card_type: std|cirrus|vmware|qxl|xenfb|tcx|cg3|virtio|none
        :return:
        """
        self._args_lines.append('-vga {}'.format(card_type))

    def setup_device_vga(self, device_id, **kwargs):
        """
        use 'qemu-system-x86_64 -device VGA,help' to print all possible properties

        :param device_id:
        :param rombar: uint32
        :param x-pcie-lnksta-dllla: str (on/off)
        :param mmio: str (on/off)
        :param multifunction: str (on/off)
        :param qemu-extended-regs: str (on/off)
        :param big-endian-framebuffer: str (on/off)
        :param romfile: str
        :param vgamem_mb: uint32
        :param x-pcie-extcap-init: str (on/off)
        :param command_serr_enable: str (on/off)
        :param addr: int32 (Slot and optional function number, example: 06.0 or 06)
        """
        options = ",".join(["{}={}".format(k, v) for k, v in kwargs.items()])
        self._args_lines.append('-device VGA,id={},{}'.format(device_id, options))

    def setup_device_virtio_rng_pci(self):
        """
        # https://wiki.qemu.org/Features/VirtIORNG
        https://st.yandex-team.ru/QEMUKVM-1590
        """
        self._args_lines.append('-object rng-random,id=rng0,filename=/dev/urandom -device virtio-rng-pci,rng=rng0')

    def setup_device_virtio_blk_pci(self, device_id, **kwargs):
        """
        use 'qemu-system-x86_64 -device virtio-blk-pci,help' to print all possible properties
        :param device_id:
        :param kwargs:
        :return:
        """
        options = ",".join(["{}={}".format(k, v) for k, v in kwargs.items()])
        self._args_lines.append('-device virtio-blk-pci,id={},{}'.format(device_id, options))

    def setup_rtc(self):
        self._args_lines.append('-rtc base=utc,driftfix=slew -global kvm-pit.lost_tick_policy=discard -no-hpet')

    def setup_net_over_tap_interface(self, ifname, mac):
        self._args_lines.append('-netdev tap,id=qemu_net,ifname={},script=no'.format(ifname))
        self._args_lines.append('-device virtio-net-pci,netdev=qemu_net,mac={}'.format(mac))

    def setup_memory(self, bytes_count):
        megabytes_count = helpers.bytes_to_megabytes(bytes_count)
        self._args_lines.append('-m "{}M"'.format(megabytes_count))

    def setup_cpu(self, cpus=1, model='host', flags=None, **params):
        """
        'qemu-system-x86_64 -cpu help' for list available cpu models and flags

        :param cpus: set the number of CPUs to 'n' [default=1]
        :param model: choose from 'qemu-system-x86_64 -cpu help' (default: host)
        :param flags: Recognized CPUID flags
        :param maxcpus: maximum number of total cpus, including offline CPUs for hotplug, etc
        :param cores: number of CPU cores on one socket
        :param threads: number of threads on one CPU core
        :param sockets: number of discrete sockets in the system
        """
        if not self._kvm_available and model == 'host':
            log.warning('cpu model "host" only available in KVM mode')

        self._args_lines.append('-cpu {}{}'.format(model, '' if not flags else ",".join([''] + flags)))

        smp_options = ",".join([''] + ["{}={}".format(k, v) for k, v in params.items()])
        self._args_lines.append('-smp "{}{}"'.format(cpus, smp_options if params else ''))

    def setup_vnc_socket(self, socket_path='./vnc.sock'):
        self._args_lines.append('-device usb-tablet -vnc unix:{},password'.format(socket_path))

    def setup_mon_socket(self, socket_path='./mon.sock'):
        self._args_lines.append('-chardev socket,id=monsk,path={socket_path},server,nowait '
                                '-mon monsk '
                                '-monitor none'.format(socket_path=socket_path))

    def setup_audio(self, audio):
        if audio != vmagent_pb2.VMConfig.NONE:
            audio_type = vmagent_pb2.VMConfig.AudioType.Name(audio).lower()
            self._args_lines.append('-soundhw {}'.format(audio_type))

    def setup_machine_type(self, machine_type, **params):
        """

        :param machine_type: 'qemu-system-x86_64 -machine help' for list supported machines
        :return:
        """
        self._args_lines.append('-machine {}{}'.format(machine_type, '' if not params else ",".join(
            [''] + ["{}={}".format(k, v) for k, v in params.items()])))

    def setup_serial_to_log_file(self, path):
        self._args_lines.append('-chardev file,id=serial_log,path={} -serial chardev:serial_log'.format(path))

    def setup_rescue(self):
        self._args_lines.append('-kernel /boot/vmlinuz -initrd /boot/initrd')
        self._args_lines.append("-append 'root=/dev/ram0 rdinit=/bin/sh init=/bin/sh ro console=tty1 loglevel=7'")

    def setup_numa_nodes(self, cpus, memory, numa_nodes):
        """
        :type cpus: int
        :type memory: int
        :type numa_nodes: list[int]
        """
        cpu_set_size = cpus / len(numa_nodes)
        cpus_left = cpus
        memory_numa_size = memory / len(numa_nodes)
        memory_left = memory
        for numa_node_index in numa_nodes:
            start = numa_node_index * cpu_set_size
            end = start + cpu_set_size - 1
            cpus_left -= cpu_set_size
            if cpus_left < cpu_set_size:
                end += cpus_left
            self._args_lines.append('-numa node,nodeid={numa},cpus={start}-{end},memdev=ram-node{numa}'.format(
                numa=numa_node_index,
                start=start,
                end=end,
            ))

            memory_left -= memory_numa_size
            memory_size = memory_numa_size
            if memory_left < memory_numa_size:
                memory_size += memory_left
            self._args_lines.append('-object memory-backend-ram,size={size},policy=bind,host-nodes={numa},id=ram-node{numa}'.format(
                size=memory_size,
                numa=numa_node_index,
            ))

    def setup_vfio_devices(self, vfio_numa_mapping, use_numa):
        """
        :type vfio_numa_mapping: dict[int, list[str]]
        :type use_numa: bool
        """
        if use_numa:
            slot_index = 0
            bus_nr = 128
            for numa_index in vfio_numa_mapping.iterkeys():
                pxb_id = 'pcie.{}'.format(numa_index + 1)
                self.add_pcie_expander_bus(numa=numa_index, pxb_id=pxb_id, bus_nr=bus_nr)
                for dev_id in vfio_numa_mapping[numa_index]:
                    self.add_numa_vfio_pci_device(slot=slot_index, pxb_id=pxb_id, bus_id=dev_id)
                    slot_index += 1
                bus_nr += len(vfio_numa_mapping[numa_index]) * 4
        else:
            for devices in vfio_numa_mapping.itervalues():
                for dev_id in devices:
                    self.add_vfio_pci_device(dev_id)

    def add_drive(self, image_file_path, interface, drive_id=None):
        """

        :param image_file_path:
        :param drive_id:
        :param interface: none or virtio
        """
        line = '-drive file="{image_file_path}",if={interface},cache=none'
        line_result = line.format(
            image_file_path=image_file_path,
            interface=interface,
        )
        if drive_id is not None:
            line_result += ',id={}'.format(drive_id)
        self._args_lines.append(line_result)

    def add_virtio_drive(self, file_path, drive_id):
        """
        :type file_path: str
        :type drive_id: str
        """
        self._args_lines.append('-drive file="{file_path}",id={drive_id},if=none,cache=none'.format(
            file_path=file_path,
            drive_id=drive_id,
        ))
        self.setup_device_virtio_blk_pci(
            device_id=drive_id,
            scsi='off',
            bus='pcie.0',
            drive=drive_id
        )

    def setup_drive_with_cloud_init_config(self, config_dir_path, file_label, drive_id):
        """

        :type config_dir_path: str
        :type file_label: str
        :type drive_id: str
        """
        self._args_lines.append(
            '-drive file=fat:"{config_dir_path}",'
            'id={drive_id},if=none,file.label={file_label},readonly=on,cache=none'.format(
                drive_id=drive_id,
                config_dir_path=config_dir_path,
                file_label=file_label
            ))
        self.setup_device_virtio_blk_pci(
            device_id=drive_id,
            scsi='off',
            bus='pcie.0',
            drive=drive_id
        )

    def add_vfio_pci_device(self, bus_id):
        """

        :param bus_id: str
        """
        self._args_lines.append('-device vfio-pci,host={bus_id}'.format(bus_id=bus_id))

    def add_numa_vfio_pci_device(self, slot, pxb_id, bus_id):
        """
        :type slot: int
        :type pxb_id: str
        :type bus_id: str
        """
        root_port_id = 's{}'.format(slot)
        self._args_lines.append('-device pcie-root-port,id={root_port_id},slot={slot},bus={pxb_id},mem-reserve=16M,pref64-reserve=128G'.format(
            root_port_id=root_port_id,
            pxb_id=pxb_id,
            slot=slot,
        ))
        self._args_lines.append('-device vfio-pci,host={bus_id},bus={root_port_id}'.format(
            root_port_id=root_port_id,
            bus_id=bus_id,
        ))

    def add_pcie_expander_bus(self, numa, bus_nr, pxb_id):
        """
        :type numa: int
        :type bus_nr: int
        :type pxb_id: str
        """
        self._args_lines.append('-device pxb-pcie,bus_nr={bus_nr},bus=pcie.0,id={pxb_id},numa_node={numa}'.format(
            bus_nr=bus_nr,
            pxb_id=pxb_id,
            numa=numa,
        ))

    @property
    def args(self):
        for arg in self._args_lines:
            yield arg + " " if arg != self._args_lines[-1] else arg


class QEMULauncher(object):
    WINDOWS_READY_FILE_NAMES = [
        'CloudbaseInitSetup_0_9_11_x64.msi',
        'CloudbaseInitSetup_0_9_11_x86.msi',
        'cloudbase-init.conf',
        'cloudbase-init-unattend.conf'
    ]
    WINDOWS_READY_SOURCE_DIR = '/opt'
    ROOT_DISK_ID = 'root_disk'
    CI_CONFIG_DISK_ID = 'ciconfig_disk'

    def __init__(self, qemu_system_bin_path=None, windows_source_dir=None):
        """

        :type qemu_system_bin_path: str
        """
        self._qemu_system_bin_path = qemu_system_bin_path or QEMUSystemCmdBuilder.DEFAULT_BIN_PATH
        self._windows_source_dir = windows_source_dir or self.WINDOWS_READY_SOURCE_DIR
        # todo: check exists and accessable

    def _generate_instance_id(self, context, volumes):
        """

        :type context: infra.qyp.vmagent.src.config.VmagentContext
        :type volumes: list[volume_manager.VolumeWrapper]
        """
        m = hashlib.md5()
        for volume in volumes:
            m.update(volume.name)
            m.update(volume.image_type_str)
            m.update(volume.resource_url)
            m.update(str(volume.available_size))
            m.update(volume.vm_mount_path)
            m.update(volume.req_id)  # case when add->remove->add volume
        m.update(context.VM_IP)
        m.update(context.VM_AUX_IP)
        prefix = m.hexdigest()
        return "{}-{}".format(prefix, context.VM_HOSTNAME)

    def _get_tpl_names(self, tpl_prefix):
        return TEMPLATES_ENV.list_templates(filter_func=lambda f: f.startswith(tpl_prefix))

    def _generate_cloud_init_config_files(self, tpl_prefix, context, main_volume, volumes):
        """

        :type tpl_prefix: str
        :type context: infra.qyp.vmagent.src.config.VmagentContext
        :type main_volume: volume_manager.VolumeWrapper
        :type volumes: list[volume_manager.VolumeWrapper]
        :rtype: list[tuple[list[str], str, str]]
        """
        tpl_names = self._get_tpl_names(tpl_prefix)
        instance_id = self._generate_instance_id(context, volumes)
        for tpl_name in tpl_names:
            tpl = TEMPLATES_ENV.get_template(tpl_name)
            content = tpl.render(VM_HOSTNAME=context.VM_HOSTNAME,
                                 VM_AUX_IP=context.VM_AUX_IP,
                                 VM_MAC=context.VM_MAC,
                                 SSH_AUTHORIZED_KEYS=context.SSH_AUTHORIZED_KEYS,
                                 MAIN_VOLUME=main_volume,
                                 VOLUMES=volumes,
                                 VM_IP=context.VM_IP,
                                 TAP_LL=context.TAP_LL,
                                 uuid=uuid.uuid4(),
                                 instance_id=instance_id
                                 )
            target_file_path_parts = tpl_name.replace(tpl_prefix, "").replace(".jinja2", "")
            target_file_path_parts = target_file_path_parts.split('__')
            yield target_file_path_parts[:-1], target_file_path_parts[-1], content

    def _generate_linux_bash_launcher(self, context, vm_config, volumes, qemu_system_builder, rescue=False):
        """

        :type context: infra.qyp.vmagent.src.config.VmagentContext
        :param vm_config:
        :type volumes: list[volume_manager.VolumeWrapper]
        :type qemu_system_builder: QEMUSystemCmdBuilder
        :type rescue: bool
        """
        tpl = TEMPLATES_ENV.get_template('launcher.sh.jinja2')
        qemu_system_builder.setup_machine_type('q35', usb='on')
        cpu_flags = ['host-phys-bits']
        if context.VFIO_DEVICES:
            # QEMUKVM-1380
            cpu_flags.append('kvm=off')
        qemu_system_builder.setup_cpu(model='host', cpus=vm_config.vcpu, flags=cpu_flags)
        qemu_system_builder.setup_memory(vm_config.mem)
        if context.USE_NUMA:
            rounded_bytes = helpers.round_bytes(vm_config.mem)
            qemu_system_builder.setup_numa_nodes(
                cpus=vm_config.vcpu,
                memory=rounded_bytes,
                numa_nodes=context.NUMA_NODES
            )
        qemu_system_builder.setup_mon_socket(context.MONITOR_PATH)
        qemu_system_builder.setup_audio(vm_config.audio)
        qemu_system_builder.setup_vnc_socket(context.VNC_SOCKET_FILE_PATH)
        qemu_system_builder.setup_graphics_card(card_type='std')
        qemu_system_builder.setup_serial_to_log_file(context.SERIAL_LOG_FILE_PATH)
        for volume in volumes:
            if volume.is_main:
                qemu_system_builder.add_virtio_drive(file_path=volume.drive_path, drive_id=self.ROOT_DISK_ID)
        if not rescue:
            qemu_system_builder.setup_device_virtio_rng_pci()
            qemu_system_builder.setup_net_over_tap_interface(context.TAP_DEV, context.VM_MAC)
            qemu_system_builder.setup_vfio_devices(
                vfio_numa_mapping=context.VFIO_NUMA_MAPPING,
                use_numa=context.USE_NUMA,
            )
            qemu_system_builder.setup_drive_with_cloud_init_config(
                config_dir_path=context.CLOUD_INIT_CONFIGS_FOLDER_PATH,
                file_label='cidata',
                drive_id=self.CI_CONFIG_DISK_ID,
            )
        else:
            qemu_system_builder.setup_rescue()
        for volume in volumes:
            if volume.is_main:
                continue
            qemu_system_builder.add_virtio_drive(file_path=volume.drive_path, drive_id=volume.name)

        result = tpl.render(qemu_system_command=qemu_system_builder,
                            VNC_PORT=context.VNC_PORT,
                            VNC_SOCKET_FILE_PATH=context.VNC_SOCKET_FILE_PATH)
        return result.replace('\\\\', "\\")

    def _copy_windows_files(self, target_dir):
        shutil.copy(os.path.join(self._windows_source_dir, 'CloudbaseInitSetup_0_9_11_x64.msi'), target_dir)
        shutil.copy(os.path.join(self._windows_source_dir, 'CloudbaseInitSetup_0_9_11_x86.msi'), target_dir)
        shutil.copy(os.path.join(self._windows_source_dir, 'cloudbase-init.conf'), target_dir)
        shutil.copy(os.path.join(self._windows_source_dir, 'cloudbase-init-unattend.conf'), target_dir)

    def _generate_windows_bash_launcher(self, context, vm_config, volumes, qemu_system_builder, rescue=False):
        """

        :type context: infra.qyp.vmagent.src.config.VmagentContext
        :type vm_config: infra.qyp.proto_lib.vmagent_pb2.VMConfig
        :type volumes: list[volume_manager.VolumeWrapper]
        :type qemu_system_builder: QEMUSystemCmdBuilder
        :type rescue: bool
        """
        if rescue:
            raise NotImplementedError('rescue mode not implemented for windows platform')

        tpl = TEMPLATES_ENV.get_template('launcher.sh.jinja2')

        qemu_system_builder.setup_machine_type('q35', usb='on')
        cpu_flags = ['host-phys-bits', '+vmx']
        if context.VFIO_DEVICES:
            # QEMUKVM-1380
            cpu_flags.append('kvm=off')
        qemu_system_builder.setup_cpu(model='host', cpus=vm_config.vcpu, flags=cpu_flags, sockets=1,
                                      cores=vm_config.vcpu, threads=1)
        qemu_system_builder.setup_memory(vm_config.mem)
        if context.USE_NUMA:
            rounded_bytes = helpers.round_bytes(vm_config.mem)
            qemu_system_builder.setup_numa_nodes(
                cpus=vm_config.vcpu,
                memory=rounded_bytes,
                numa_nodes=context.NUMA_NODES
            )
        qemu_system_builder.setup_mon_socket(context.MONITOR_PATH)
        qemu_system_builder.setup_audio(vm_config.audio)
        qemu_system_builder.setup_vnc_socket(context.VNC_SOCKET_FILE_PATH)
        qemu_system_builder.setup_device_vga(device_id='video0', vgamem_mb='64')
        qemu_system_builder.setup_serial_to_log_file(context.SERIAL_LOG_FILE_PATH)
        qemu_system_builder.setup_rtc()

        for volume in volumes:
            if volume.is_main:
                qemu_system_builder.add_virtio_drive(file_path=volume.drive_path, drive_id=self.ROOT_DISK_ID)

        qemu_system_builder.setup_net_over_tap_interface(context.TAP_DEV, context.VM_MAC)
        qemu_system_builder.setup_vfio_devices(
            vfio_numa_mapping=context.VFIO_NUMA_MAPPING,
            use_numa=context.USE_NUMA,
        )
        qemu_system_builder.setup_drive_with_cloud_init_config(
            config_dir_path=context.CLOUD_INIT_CONFIGS_FOLDER_PATH,
            file_label='config-2',
            drive_id=self.CI_CONFIG_DISK_ID,
        )

        result = tpl.render(qemu_system_command=qemu_system_builder,
                            VNC_PORT=context.VNC_PORT,
                            VNC_SOCKET_FILE_PATH=context.VNC_SOCKET_FILE_PATH)
        return result.replace('\\\\', "\\")

    def build(self, context, vm_config, rescue=False):
        """

        :type context: infra.qyp.vmagent.src.config.VmagentContext
        :type vm_config: vmagent_pb2.VMConfig
        :type rescue: bool
        """
        if vm_config.type == vmagent_pb2.VMConfig.LINUX:
            generate_bash_launcher = self._generate_linux_bash_launcher
            tpl_prefix = 'linux_'
        elif vm_config.type == vmagent_pb2.VMConfig.WINDOWS:
            generate_bash_launcher = self._generate_windows_bash_launcher
            tpl_prefix = 'win_'
        else:
            raise ValueError('Unsupported vm type: {}'.format(vm_config.type))

        volumes = [volume_manager.VolumeWrapper(v) for v in vm_config.volumes]
        main_volume_exists = [v for v in volumes if v.is_main]
        if not volumes or not main_volume_exists:
            raise ValueError('VMConfig.volumes should contain main volume')
        main_volume = main_volume_exists[0]

        helpers.remove_if_exists(context.CLOUD_INIT_CONFIGS_FOLDER_PATH, recursive=True)

        if not rescue:
            config_files_generator = self._generate_cloud_init_config_files(tpl_prefix, context, main_volume, volumes)
            for dirs, file_name, content in config_files_generator:
                result_dir = os.path.join(context.CLOUD_INIT_CONFIGS_FOLDER_PATH, *dirs)
                if not os.path.exists(result_dir):
                    os.makedirs(result_dir)
                with open(os.path.join(result_dir, file_name), 'w') as fp:
                    fp.write(content)

        if vm_config.type == vmagent_pb2.VMConfig.WINDOWS:
            self._copy_windows_files(context.CLOUD_INIT_CONFIGS_FOLDER_PATH)

        # build qemu_launcher.sh
        helpers.remove_if_exists(context.QEMU_LAUNCHER_FILE_PATH)

        with open(context.QEMU_LAUNCHER_FILE_PATH, 'w') as fp:
            bash_launcher_content = generate_bash_launcher(
                context=context,
                vm_config=vm_config,
                volumes=volumes,
                qemu_system_builder=QEMUSystemCmdBuilder(self._qemu_system_bin_path),
                rescue=rescue
            )
            fp.write(bash_launcher_content)
