import json
import logging
import os
import requests
import subprocess
import threading
import urllib

from google.protobuf.json_format import MessageToJson as ProtoMessageToJson


class SkynetCli(object):
    RESOURCE_PREFIX = ""
    DEFAULT_BIN_PATH = 'sky'

    def __init__(self, bin_path=None):
        self._bin_path = bin_path or self.DEFAULT_BIN_PATH

    def share(self, path):
        path = os.path.realpath(path)  # always use realpath (skybone will do that anyway, but we want real file name)
        try:
            cwd, file_path = os.path.split(path)
            result = subprocess.check_output([self._bin_path, 'share', '-d', cwd, file_path])
            return result.strip()
        except subprocess.CalledProcessError as e:
            raise self.SkynetError('sky share error: {}, {}'.format(e.returncode, e.output))

    def get(self, resource_url, path):
        try:
            subprocess.check_output([self._bin_path, "get", "-u", "-d", path, resource_url],
                                    stderr=subprocess.STDOUT)

        except subprocess.CalledProcessError as e:
            raise self.SkynetError("sky get error: {}, {}".format(e.returncode, e.output))

    class SkynetError(Exception):
        pass


class QDMCli(object):
    QDM_CLI_RESOURCE_URL_TPL = 'http://qdm.yandex-team.ru/api/v1/client_resource?%s'

    def __init__(self, extras_folder_path, skynet_cli, vmagent_version=None):  # type: (str, SkynetCli, str) -> None
        self.log = logging.getLogger('res_mngr.qdmcli')
        self._extras_folder_path = extras_folder_path
        self._skynet_cli = skynet_cli
        self._vmagent_version = vmagent_version
        self._upload_proc = None

    @property
    def vmagent_version(self):
        vmagent_version = self._vmagent_version
        if not isinstance(vmagent_version, basestring):
            vmagent_version = 'unknown'

        if vmagent_version == '___VMAGENT_VERSION___':
            vmagent_version = 'dev'
        return vmagent_version

    def get_qdm_cli(self, op, key):  # type: (str, str) -> str
        self.log.debug('Getting fresh qdm-mds-cli')
        try:
            query = urllib.urlencode((
                ('op', op),
                ('key', key),
                ('vmagent', self.vmagent_version),
            ))

            qdm_cli_resource = requests.get(self.QDM_CLI_RESOURCE_URL_TPL % (query,)).text
        except Exception as e:
            raise self.QdmError('Unable to get qdm cli rbtorrent id: {}'.format(e))

        self._skynet_cli.get(qdm_cli_resource, self._extras_folder_path)
        qdm_mds_cli = os.path.join(self._extras_folder_path, 'qdm-mds-cli')

        if not os.path.exists(qdm_mds_cli):
            raise self.QdmError('Unable to find downloaded qdm-mds-cli')

        self.log.debug('  done, got %d bytes', os.path.getsize(qdm_mds_cli))

        return qdm_mds_cli

    def _build_qdm_mds_cli_upload_args(self, key, logfile, backup_spec, progress_file='-', background=False):
        qdm_mds_cli = self.get_qdm_cli('up', key)

        args = [
            qdm_mds_cli,
            '--logfile', logfile,
            '--progress-file', progress_file,
            'upload',
            '--qdm-key', key,
            '--spec', ProtoMessageToJson(backup_spec, indent=0, preserving_proto_field_name=True)
        ]

        if background:
            args += ['--background']

        return args

    def upload(self, key, backup_spec, logfile, progress_file):
        try:
            args = self._build_qdm_mds_cli_upload_args(key, logfile, backup_spec, progress_file)
            self.log.info('Executing %r', args)
            self._upload_proc = subprocess.Popen(
                args, stdin=subprocess.PIPE, stderr=subprocess.STDOUT, stdout=subprocess.PIPE
            )
            qdm_key = self._upload_proc.communicate()[0].strip()
            # upload process can be killed by user
            if self._upload_proc.returncode == -9:
                raise self.UploadQdmError('qdm upload was terminated forcefully')
            return qdm_key
        except subprocess.CalledProcessError as e:
            raise self.UploadQdmError('qdm-mds-cli error: {}, {}'.format(e.returncode, e.output))

    def stop_upload(self):
        # Popen().poll() returns None while subprocess is running
        if self._upload_proc is None or self._upload_proc.poll() is not None:
            raise Exception('No active backup in progress')

        self._upload_proc.kill()

    def upload_bg(self, key, backup_spec, logfile):
        try:
            args = self._build_qdm_mds_cli_upload_args(key, logfile, backup_spec, background=True)
            self.log.info('Executing %r (in background thread)', args)
            proc = subprocess.Popen(args)
            self.log.debug('  proc pid: %d', proc.pid)
            return proc
        except OSError as ex:
            raise self.UploadQdmError('qdm-mds-cli unable to launch: {}, {}'.format(type(ex).__name__, ex))

    def get(self, resid, path, logfile, progress_file, vm_id, cluster, node_id):
        qdm_mds_cli = self.get_qdm_cli('dl', resid)

        for part_res_id in resid.split(','):
            try:
                subprocess.check_output([qdm_mds_cli,
                                         '--logfile', logfile,
                                         '--progress-file', progress_file,
                                         'download',
                                         '--rev', part_res_id,
                                         '--vm-id', vm_id,
                                         '--cluster', cluster,
                                         '--node-id', node_id,
                                         path], stderr=subprocess.STDOUT)
            except subprocess.CalledProcessError as e:
                raise self.DownloadQdmError('qdm-mds-cli error: {}, {}'.format(e.returncode, e.output))

    class QdmError(Exception):
        pass

    class UploadQdmError(QdmError):
        pass

    class DownloadQdmError(QdmError):
        pass


class ResourceManager(object):
    def __init__(self, vm_id, cluster, node_id, extras_folder_path, logs_folder_path, vmagent_version=None):
        # type: (str, str, str, str, str, str) -> None
        self.log = logging.getLogger('res_mngr')

        self._vm_id, self._cluster, self._node_id = vm_id, cluster, node_id
        self._extras_folder_path = extras_folder_path
        self._logs_folder_path = logs_folder_path
        self._data_transfer_state_file_path = os.path.join(extras_folder_path, 'data_transfer_state.json')
        self._skynet_cli = SkynetCli()
        self._qdm_cli = QDMCli(extras_folder_path=extras_folder_path,
                               skynet_cli=self._skynet_cli,
                               vmagent_version=vmagent_version)

        self._bg_proc = None
        self._bg_thr = None

    def get_resource(self, resid, path):
        if ':' not in resid:
            raise self.ResourceDownloadError('Invalid resource "{}"(does not looks like URI)'.format(resid))

        restype = resid.split(':', 1)[0]

        if restype == 'qdm':
            log_file = os.path.join(self._logs_folder_path, 'qdm.log')
            try:
                return self._qdm_cli.get(
                    resid, path, log_file, self._data_transfer_state_file_path, self._vm_id,
                    self._cluster, self._node_id,
                )
            except self._qdm_cli.QdmError as e:
                raise self.ResourceDownloadError(e.message)
        else:
            # Catch everything else - rbtorrent, http, https, etc.
            try:
                return self._skynet_cli.get(resid, path)
            except self._skynet_cli.SkynetError as e:
                raise self.ResourceDownloadError(e.message)

    def upload_resource(self, key, backup_spec):
        log_file = os.path.join(self._logs_folder_path, 'qdm.log')
        try:
            return self._qdm_cli.upload(key, backup_spec, log_file, self._data_transfer_state_file_path)
        except self._qdm_cli.QdmError as e:
            raise self.ResourceUploadError(e.message)

    def get_backup_status(self):
        if not os.path.exists(self._data_transfer_state_file_path):
            return
        try:
            with open(self._data_transfer_state_file_path) as f:
                return json.load(f)
        except json.decoder.JSONDecodeError:
            self.log.warning("can't decode progress file to json")
        except Exception as e:
            self.log.warning("Error reading backup progress file {}".format(e))

    def _bg_worker(self):
        if self._bg_proc:
            pid = self._bg_proc.pid
            self.log.debug('waiting bg proc pid %d to finish in background', pid)
            self._bg_proc.wait()
            returncode = self._bg_proc.returncode
            self._bg_proc = None
            self.log.debug('bg upload process pid %d exited with code %d', pid, returncode)

        self._bg_thr = None

    def upload_resource_bg(self, key, backup_spec):
        assert self._bg_proc is None
        log_file = os.path.join(self._logs_folder_path, 'qdm.log')
        self._bg_proc = self._qdm_cli.upload_bg(key, backup_spec, log_file)
        self._bg_thr = threading.Thread(target=self._bg_worker)
        self._bg_thr.start()

    def bg_abort(self):
        if self._bg_proc:
            self._bg_proc.kill()
        if self._bg_thr:
            self._bg_thr.join()

    def on_iss_hook_stop(self):
        # Abort any bg activity we may have
        self.bg_abort()

    def share_resource(self, path):
        try:
            return self._skynet_cli.share(path)
        except self._skynet_cli.SkynetError as e:
            raise self.ResourceShareError(e.message)

    def clear_data_transfer_progress(self):
        try:
            os.remove(self._data_transfer_state_file_path)
        except OSError:
            pass

    def stop_qdmupload(self):
        self.bg_abort()
        self._qdm_cli.stop_upload()

    class ResourceError(Exception):
        pass

    class ResourceDownloadError(ResourceError):
        pass

    class ResourceUploadError(ResourceError):
        pass

    class ResourceShareError(ResourceError):
        pass

    class ResourceUploadStopException(ResourceError):
        pass
