import os
import errno
import json
import string
from infra.qyp.proto_lib import vmagent_pb2
from infra.qyp.vmagent.src import helpers
from subprocess import check_output, CalledProcessError, STDOUT


class QEMUImgCmd(object):
    DEFAULT_BIN_PATH = '/usr/local/bin/qemu-img'

    def __init__(self, bin_path=None, cwd=None, sudo=False):
        bin_path = bin_path or self.DEFAULT_BIN_PATH
        self._cwd = cwd
        self._sudo = sudo
        self._bin_path = ['sudo', bin_path] if sudo else [bin_path]
        self._check_output = check_output

    def __call__(self, *args, **kwargs):
        """

        :rtype: tuple[bool, CalledProcessError | str]
        """
        if 'cwd' not in kwargs and self._cwd:
            kwargs['cwd'] = self._cwd
        command = self._bin_path + [str(a) for a in args]
        try:
            return True, self._check_output(command, stderr=STDOUT, **kwargs)
        except CalledProcessError as exc:
            return False, "FAILED: '{}', returncode: {}, output: {}".format(
                " ".join(command), exc.returncode, exc.output
            )

    def info(self, img_path):
        """

        :type img_path: str
        :rtype: (bool, dict | Exception)
        """
        success, result = self('info', img_path, '--output', 'json')
        if not success:
            return False, result
        return True, json.loads(result)

    def resize(self, img_path, size_for_raw):
        """

        :type img_path: str
        :param img_path: disk image path
        :type size_for_raw: int
        :param size_for_raw: the disk image size in bytes. Optional suffixes
            'k' or 'K' (kilobyte, 1024), 'M' (megabyte, 1024k), 'G' (gigabyte, 1024M),
            'T' (terabyte, 1024G), 'P' (petabyte, 1024T) and 'E' (exabyte, 1024P)  are
            supported. 'b' is ignored.
        :rtype: (bool, str | Exception)
        """
        return self('resize', img_path, size_for_raw)

    def rebase(self, path, backing_file):
        return self('rebase', '-u', '-b', backing_file, path)

    def create(self, path, disk_size, backing_file=None):
        cmd = ['create', '-f', 'qcow2', path, disk_size]
        if backing_file is not None:
            cmd += ['-b', backing_file]
        return self(*cmd)

    def get_virtual_disk_size(self, img_path):
        success, result = self.info(img_path)
        if not success:
            return False, result
        return True, result.get('virtual-size')


class VolumeWrapper(object):
    """
    Files organized as follows:

    {mount_path}/{IMAGE_FOLDER_NAME}/{DELTA_FILE_NAME}         -- image delta
    {mount_path}/{IMAGE_FOLDER_NAME}/<anyname>                 -- image itself (usually layer.img)
    {mount_path}/image => /image_folder/<anyname>              -- main image symlink
    {mount_path}/current.qcow2 => /image_folder/current.qcow2  -- main delta symlink
    """
    IMAGE_FOLDER_NAME = 'image_folder'
    DELTA_FILE_NAME = 'current.qcow2'
    IMAGE_FILE_NAME = 'image'

    def __init__(self, volume_pb):
        """

        :type volume_pb: infra.qyp.proto_lib.vmagent_pb2.VMVolume
        """
        self._volume_pb = volume_pb

    @property
    def mount_path(self):
        return self._volume_pb.mount_path

    @property
    def delta_file_path(self):
        return os.path.join(self._volume_pb.mount_path, self.DELTA_FILE_NAME)

    @property
    def image_file_path(self):
        return os.path.join(self._volume_pb.mount_path, self.IMAGE_FILE_NAME)

    @property
    def drive_path(self):
        if self._volume_pb.image_type == vmagent_pb2.VMVolume.RAW:
            return self.image_file_path
        elif self._volume_pb.image_type == vmagent_pb2.VMVolume.DELTA:
            return self.delta_file_path

    @property
    def image_folder_path(self):
        return os.path.join(self._volume_pb.mount_path, self.IMAGE_FOLDER_NAME)

    @property
    def image_folder_path_relative_to_storage(self):
        return self.IMAGE_FOLDER_NAME

    @property
    def available_size(self):
        return self._volume_pb.available_size

    @property
    def vm_device_name(self):
        order = 0 if self.is_main else self._volume_pb.order + 1
        return '/dev/vd{}'.format(string.lowercase[order])

    @property
    def name(self):
        return self._volume_pb.name

    @property
    def req_id(self):
        return self._volume_pb.req_id

    @property
    def resource_url(self):
        return self._volume_pb.resource_url

    @property
    def order(self):
        return self._volume_pb.order

    @property
    def vm_mount_path(self):
        return self._volume_pb.vm_mount_path or '/extra_{}'.format(self.name.lower())

    @property
    def is_main(self):
        """

        :rtype: bool
        """
        return self._volume_pb.is_main

    @property
    def is_empty(self):
        return not self._volume_pb.resource_url

    @property
    def image_type(self):
        return self._volume_pb.image_type

    @property
    def image_type_str(self):
        return vmagent_pb2.VMVolume.ImageType.Name(self._volume_pb.image_type)

    @property
    def image_type_is_raw(self):
        return self._volume_pb.image_type == vmagent_pb2.VMVolume.RAW

    @property
    def image_type_is_delta(self):
        return self._volume_pb.image_type == vmagent_pb2.VMVolume.DELTA

    @property
    def image_file_exist(self):
        return os.path.exists(self.image_file_path)

    @property
    def delta_file_exist(self):
        return os.path.exists(self.delta_file_path)

    @property
    def drive_path_exists(self):
        return os.path.exists(self.drive_path)

    def __eq__(self, other):
        if not isinstance(other, self.__class__):
            return False
        return (self._volume_pb.name == other.name and
                self._volume_pb.resource_url == other._volume_pb.resource_url and
                self._volume_pb.image_type == other._volume_pb.image_type and
                # self._volume_pb.available_size == other.available_size and
                # todo: return this check after edit volume capacity
                self._volume_pb.vm_mount_path == other._volume_pb.vm_mount_path)

    def __str__(self):
        return str(self._volume_pb)


class VolumeManager(object):

    def __init__(self, vm_volume, resource_manager, qemu_img_cmd):
        """

        :type vm_volume: VolumeWrapper
        :type resource_manager: infra.qyp.vmagent.src.resource_manager.ResourceManager
        :type qemu_img_cmd: QEMUImgCmd
        """
        self._volume = vm_volume
        self._qemu_img_cmd = qemu_img_cmd
        self._resource_manager = resource_manager

    def remove_delta(self):
        helpers.remove_if_exists(self._volume.delta_file_path)

    def remove_image(self):
        helpers.remove_if_exists(self._volume.image_file_path)

    def clear_image_folder(self):
        helpers.remove_if_exists(self._volume.image_folder_path, recursive=True)

    def remove_all_files(self):
        self.remove_image()
        self.remove_delta()
        self.clear_image_folder()

    def ensure_image_folder(self):
        # Just to be sure it exists, we will need to find realpath
        try:
            os.makedirs(self._volume.image_folder_path)
        except OSError as e:
            if e.errno != errno.EEXIST:
                raise

    def fix_image_location(self):
        self.ensure_image_folder()
        fixed_image_locations = []
        for path in (self._volume.image_file_path, self._volume.delta_file_path):
            if not os.path.exists(path):
                continue
            rpath = os.path.realpath(path)
            if os.path.realpath(self._volume.image_folder_path) not in os.path.dirname(rpath):
                basename = os.path.basename(rpath)
                target = os.path.join(self._volume.image_folder_path, basename)

                if basename != self._volume.DELTA_FILE_NAME:
                    # If filename already exists in image_folder somehow -- choose random name
                    # But do NOT do this for delta file, since we rely on delta filename
                    # during restoring from backup

                    if os.path.lexists(target) or os.path.exists(target):
                        for i in range(1, 100):
                            possible_target = target + '.%2d' % (i,)
                            if not os.path.lexists(possible_target) and not os.path.exists(possible_target):
                                target = possible_target
                                break
                        else:
                            raise self.BadAction('Unable to move image to image_folder/ -- already exists')
                else:
                    # We cant do same logic for delta file, so just check we dont have another one
                    # already
                    if os.path.lexists(target) or os.path.exists(target):
                        raise self.BadAction('We have another delta file in image_folder/, unable to move current')

                fixed_image_locations.append(rpath)
                os.rename(rpath, target)  # move to image_folder/
                os.symlink(target, rpath)  # make symlink back

        return fixed_image_locations

    def init_image(self):
        image_file, delta_file, unused_files = self.find_images()

        if self._volume.is_main and self._volume.is_empty:
            raise self.ImageError("Main volume can't be empty (resource_url required)")

        if self._volume.image_type_is_raw:
            if image_file:
                raise self.ImageError("Can't init RAW image, some file exists: {}".format(image_file))

            if self._volume.resource_url:
                image_file, delta_file = self.download_image(self._volume.resource_url)

                if delta_file:
                    raise self.DownloadingImageError(
                        'Downloading resource_url: {} contain more then one image file for RAW image'.format(
                            self._volume.resource_url
                        ))

            if not image_file:
                image_file = self.create_image()
            else:
                self.validate_image_size(image_file)

            success, result = self._qemu_img_cmd.resize(image_file, self._volume.available_size)
            if not success:
                raise self.ImageError("Can't resize RAW image with error: {}".format(result))

        elif self._volume.image_type_is_delta:
            if delta_file:
                raise self.ImageError("Can't init DELTA image, delta file exists: {}".format(delta_file))

            if not image_file and self._volume.resource_url:
                image_file, delta_file = self.download_image(self._volume.resource_url)

            if not image_file:
                raise self.ImageError('image file required for image_type == DELTA')

            if not delta_file:
                delta_file = self.create_image(backing_file=image_file)
            else:
                self.validate_image_size(image_file, delta_file)
                success, result = self._qemu_img_cmd.rebase(delta_file, image_file)
                if not success:
                    raise self.ImageError("Can't rebase DELTA image with error: {}".format(result))

                image_file_size = helpers.get_image_size(image_file)

                success, result = self._qemu_img_cmd.resize(delta_file, self._volume.available_size - image_file_size)
                if not success:
                    raise self.ImageError("Can't resize DELTA image with error: {}".format(result))

        self.create_image_symlinks(image_file, delta_file=delta_file)

    def validate_image_size(self, image_file, delta_file=None):
        available_size = self._volume.available_size
        success, virtual_size = self._qemu_img_cmd.get_virtual_disk_size(delta_file or image_file)
        if not success:
            raise self.ImageError("Can't get image virtual size with error: {}".format(virtual_size))
        if delta_file:
            image_size = helpers.get_image_size(image_file)
            available_size -= image_size

        if virtual_size > available_size:
            msg = "Available size for disk '{}' does not enough for this image: should be at least ({}), got({})".format(
                self._volume.name,
                virtual_size - available_size + self._volume.available_size,
                self._volume.available_size
            )
            raise self.VirtualSizeNotValidError(msg)

    def find_images(self):
        self.ensure_image_folder()
        found_image = found_delta = None
        all_files = []
        for root, _, filenames in os.walk(self._volume.image_folder_path):
            for fn in filenames:
                full_fn = os.path.join(root, fn)
                all_files.append(full_fn)
                if fn == self._volume.DELTA_FILE_NAME:
                    found_delta = full_fn
                elif not found_image:
                    found_image = full_fn

        if found_delta and not found_image:
            # If we have found just one file - it always should be treated as image
            # even if named "current.qcow2" for some reason.
            found_image = found_delta
            found_delta = None

        unused_files = set(all_files) - set([fn for fn in [found_image, found_delta] if fn])

        return found_image, found_delta, list([f.replace(self._volume.image_folder_path + '/', '')
                                               for f in unused_files])

    def download_image(self, resource_url):
        """
        After downloading data we may find these:
        - image file with any filename
        - image file with any filename + current.qcow2 (delta)

        :type resource_url: basestring
        :rtype: tuple[str, str | None]
        :return: list of downloaded files path (relative to mount path)
        """
        self.ensure_image_folder()

        try:
            self._resource_manager.get_resource(resource_url, self._volume.image_folder_path)
        except self._resource_manager.ResourceDownloadError as e:
            raise self.DownloadingImageError("Resource download error: {}".format(e.message))

        found_image, found_delta, unused_files = self.find_images()

        if unused_files:
            raise self.DownloadingImageError('To many files has been downloaded: {}'.format(unused_files))

        if not found_image:
            raise self.DownloadingImageError('Unable to find image file (no files?)')

        if not os.access(found_image, os.R_OK):
            raise self.DownloadingImageError('Disk resource not found')

        return found_image, found_delta

    def create_image(self, backing_file=None):
        self.ensure_image_folder()

        if self._volume.image_type_is_delta:
            if not self._volume.is_main:
                raise self.CreateImageError('Extra volume does not support create delta image')
            if not backing_file:
                raise self.CreateImageError('backing_file required for image_type == DELTA')
            if not os.path.exists(backing_file):
                raise self.CreateImageError("backing_file: '{}' does not exists".format(backing_file))

        if self._volume.image_type_is_raw:
            layer_file_path = os.path.join(self._volume.image_folder_path, self._volume.IMAGE_FILE_NAME)
            disk_size = self._volume.available_size
        elif self._volume.image_type_is_delta:
            layer_file_path = os.path.join(self._volume.image_folder_path, self._volume.DELTA_FILE_NAME)
            disk_size = self._volume.available_size - helpers.get_image_size(backing_file)
        else:
            raise self.CreateImageError('Image Type: {} does not support'.format(self._volume.image_type_str))

        if os.path.exists(layer_file_path):
            raise self.CreateImageError("Can't create image , file '{}' already exists".format(layer_file_path))

        success, result = self._qemu_img_cmd.create(layer_file_path, disk_size, backing_file=backing_file)
        if not success:
            raise self.CreateImageError("Can't create image: {}".format(result))

        return layer_file_path

    def create_image_symlinks(self, image_file, delta_file=None):
        if os.path.exists(self._volume.image_file_path):
            if os.path.islink(self._volume.image_file_path):
                os.unlink(self._volume.image_file_path)
            else:
                raise self.ImageError("Can't create image symlink, image exists and is not symlink: {}".format(
                    self._volume.image_file_path))
        try:
            os.symlink(image_file, self._volume.image_file_path)
        except OSError as e:
            raise self.ImageError('Cannot make symlink to {}: {}'.format(image_file, e))

        if delta_file:
            if os.path.exists(self._volume.delta_file_path):
                if os.path.islink(self._volume.delta_file_path):
                    os.unlink(self._volume.delta_file_path)
                else:
                    raise self.ImageError("Can't create delta symlink, delta exists and is not symlink: {}".format(
                        self._volume.delta_file_path))
            try:
                os.symlink(delta_file, self._volume.delta_file_path)
            except OSError as e:
                raise self.ImageError('Cannot make symlink to {}: {}'.format(delta_file, e))

        # Find, check and set full path to /image symlink
        if not os.path.exists(self._volume.image_file_path):
            raise self.ImageError('Unable to find image after symlinking')

        if not os.path.realpath(self._volume.image_file_path) == os.path.realpath(image_file):
            raise self.ImageError('Invalid image symlink')

    class BadAction(Exception):
        pass

    class ImageError(Exception):
        pass

    class CreateImageError(ImageError):
        pass

    class DownloadingImageError(ImageError):
        pass

    class VirtualSizeNotValidError(ImageError):
        pass
