import collections
import getpass
import json
import sys
from functools import partial
from multiprocessing.pool import ThreadPool

from infra.qyp.vmctl.src.actions.list_helpers import ListField, int_to_str_gigabytes
from tabulate import tabulate
from infra.qyp.vmctl.src import defines
from infra.qyp.proto_lib import vmset_pb2


def list_to_str(value):
    """
    :type value: list
    """
    return ' '.join(value)


def version_str(value):
    """
    Adds v character to make tabulate consider this value as string not as float
    :type value: str
    """
    if value == 'N/A':
        return value
    else:
        return 'v{}'.format(value)


def get_qemu_disk_size(vm, **kwargs):
    """
    :type vm: vmset_pb2.VM
    """
    for vol in vm.spec.qemu.volumes:
        if vol.name == '/qemu-persistent':
            return vol.capacity
    return 0


def host_name_getter(vm, cluster):
    """
    :type vm: vmset_pb2.VM
    :type cluster: str
    """
    name = AVAILABLE_FIELDS['name'](vm, cluster=cluster)
    location = AVAILABLE_FIELDS['location'](vm, cluster=cluster)

    return '{}.{}.yp-c.yandex.net'.format(name, location.lower())


def guarantees_to_str(value):
    return ', '.join('{} {}Mb/s'.format(disk_type.upper(), bandwidth / 1024 ** 2)
                     for disk_type, bandwidth in value.iteritems())


AVAILABLE_FIELDS = collections.OrderedDict([
    ('name', ListField('Name', attr='meta.id')),
    ('location', ListField('Location', attr='meta.cluster',
                           value_getter=lambda vm, cluster: cluster)),
    ('segment', ListField('Segment', attr='spec.qemu.node_segment')),
    ('cpu', ListField('CPU', attr='spec.qemu.resource_requests.vcpu_guarantee')),
    ('ram', ListField('RAM', attr='spec.qemu.resource_requests.memory_guarantee',
                      formatter=int_to_str_gigabytes)),
    ('disk', ListField('Disk',
                       value_getter=get_qemu_disk_size,
                       formatter=int_to_str_gigabytes)),
    ('io_guarantees', ListField('IO guarantees', attr='spec.qemu.io_guarantees_per_storage',
                                formatter=guarantees_to_str)),
    ('logins', ListField('Logins', attr='meta.auth.owners.logins',
                         formatter=list_to_str)),
    ('groups', ListField('Groups', attr='meta.auth.owners.group_ids',
                         formatter=list_to_str)),
    ('hostname', ListField('Host', value_getter=host_name_getter, formatter=str)),
    ('abc', ListField('ABC', attr='spec.account_id')),
    ('vmagent_version', ListField('Vmagent', attr='spec.vmagent_version', formatter=version_str))
])


def fetch_from_cluster(cluster, api_client, login, node_segment, abc,
                       filter_by_name=None, skip=0, limit=1000, labels_filters=None):
    try:
        vms = api_client.list_yp_vms(cluster, login, node_segment, abc, filter_by_name=filter_by_name,
                                     skip=skip, limit=limit, labels_filters=labels_filters)
    except Exception as e:
        print e
        sys.stderr.write("No response from cluster {}\n".format(cluster))
        vms = []
    return cluster, vms


def fetch_yp_vms(args, api_client, skip=0, limit=10):
    """

    :type args:
    :type api_client: infra.qyp.vmctl.src.api.VMProxyClient
    """
    # Login argument
    if args.login_all:
        login = None
    else:
        login = args.login or getpass.getuser()
    # Cluster argument
    if args.list_yp_cluster:
        cluster_list = args.list_yp_cluster
    else:
        cluster_list = defines.VMPROXY_LOCATION.keys()
    # List field argument
    if args.fields:
        field_set = set(args.fields)
        fields = [f for key, f in AVAILABLE_FIELDS.iteritems() if key in field_set]
    else:
        fields = AVAILABLE_FIELDS.values()
    labels_filters = {}
    if args.unused_only:
        labels_filters['qyp_unused_for_30_days'] = vmset_pb2.YpVmFindQueryBuilder(equal='true')
    # Making list
    rows = []
    count = 0

    if api_client.custom_proxyhost:
        cluster_list = ['---']

    pool = ThreadPool(3)
    _fetch_from_cluster = partial(fetch_from_cluster, api_client=api_client, login=login,
                                  node_segment=args.node_segment,
                                  abc=args.abc,
                                  filter_by_name=args.name_filter,
                                  skip=skip, limit=limit, labels_filters=labels_filters)

    res = pool.map(_fetch_from_cluster, cluster_list)

    for cluster, vms in res:
        for vm in vms:
            count += 1
            row = [count]
            for field in fields:
                row.append(field(vm, cluster=cluster))
            rows.append(row)

    return rows, ['Index'] + [f.field_name for f in fields]


def list_yp_vms(args, vmproxy_client=None, skip=0, limit=10):
    rows, headers = fetch_yp_vms(args, vmproxy_client, skip=skip, limit=limit)

    res = tabulate(rows, headers=headers, tablefmt='fancy_grid')
    return res


def dump_json_yp_vms(args, vmproxy_client=None, skip=0, limit=10):
    rows, headers = fetch_yp_vms(args, vmproxy_client, skip=skip, limit=limit)

    return json.dumps([dict(zip(headers, x)) for x in rows], indent=4, sort_keys=True)
