import getpass
import os
import time
import requests
import base64
from functools import partial
from distutils.version import LooseVersion

import nanny_rpc_client

import library.python.oauth as lpo
from infra.qyp.vmctl.src import defines
from infra.qyp.proto_lib import vmset_api_pb2, vmset_pb2, vmagent_api_pb2, vmset_api_stub, vmagent_pb2
from infra.qyp.vmctl.src import errors, helpers


class VMProxyClient(object):
    @classmethod
    def find_token(cls, args_token=None):  # type: (str) -> str
        return (args_token
                or lpo.get_token(defines.CLIENT_ID, defines.CLIENT_SECRET)
                )

    def __init__(self, token="", proxyhost=None, ssl_none=False, timeout=defines.TIMEOUT, user_agent=None):
        """

        :type token: str
        :type proxyhost: str
        :type ssl_none: bool
        :type timeout: int
        """
        self.ssl_verify = not ssl_none
        self.headers = {}

        self.token = token
        self.user_agent = user_agent
        self.headers['Authorization'] = "OAuth {}".format(token)
        if self.user_agent:
            self.headers['User-Agent'] = self.user_agent
        self.timeout = timeout

        self.custom_proxyhost = proxyhost

    def get_rpc_stub(self, cluster):
        """
        :type cluster: str
        rtype: vmset_api_stub.VmSetServiceStub
        """
        if self.custom_proxyhost:
            rpc_url = self.custom_proxyhost
        else:
            rpc_url = defines.VMPROXY_LOCATION[cluster.upper()]
        rpc = nanny_rpc_client.RetryingRpcClient(rpc_url + '/api',
                                                 oauth_token=self.token,
                                                 request_timeout=300,
                                                 user_agent=self.user_agent
                                                 )
        return vmset_api_stub.VmSetServiceStub(rpc)

    def _any_cluster_request(self, req, handler):
        """
        :type req: google.protobuf.message.Message
        :type handler: collections.Callable
        """
        errors_dict = {}
        for cluster_ in defines.VMPROXY_LOCATION.iterkeys():
            try:
                handler(req, cluster_)
            except Exception as e:
                errors_dict[cluster_] = e.message
                continue
            else:
                break
        else:
            errors_str = '\n'.join('{}: {}'.format(key, value) for key, value in errors_dict.iteritems())
            raise errors.VMAgentParamsError("Can't allocate resources in any cluster:\n{}".format(errors_str))

    def _create_request(self, req, cluster):
        """
        :type req: vmset_api_pb2.CreateVmRequest
        :type cluster: str
        """
        rpc_stub = self.get_rpc_stub(cluster)
        rpc_stub.create_vm(req)
        print 'Creating {} in {}'.format(req.meta.id, cluster)

    def create(self, config_id, vcpu, mem, rb_torrent, disk_size, vm_type, autorun, cluster, pod_id, network_id,
               node_segment, volume_size, mem_config, vcpu_limit, vcpu_guarantee, storage_class, enable_internet,
               use_nat64, abc, image_type, logins=None, groups=None, layer_url=None, custom_pod_resources=None,
               vmagent_version='', node_id='', extra_volumes=None, gpu_count=None, gpu_model=None,
               io_guarantee_ssd=None, io_guarantee_hdd=None, audio=None, network_bandwidth_guarantee=0,
               ip4_address_pool_id=None):
        """
        :type config_id: str
        :type image_type: str
        :type cluster: str
        :type pod_id: str
        :type network_id: str
        :type node_segment: str
        :type volume_size: int
        :type mem: int
        :type vcpu_limit: int
        :type vcpu_guarantee: int
        :type storage_class: str
        :type enable_internet: bool
        :type use_nat64: bool
        :type abc: str
        :type logins: list[str] | NoneType
        :type groups: list[str] | NoneType
        :type layer_url: str | NoneType
        :type id: str
        :type vcpu: int
        :type mem_config: int
        :type rb_torrent: str
        :type disk_size: int
        :type vm_type: str
        :type autorun: bool
        :type custom_pod_resources: str
        :type vmagent_version: str | NoneType
        :type node_id: str
        :type extra_volumes: list[vmset_pb2.Volume] | NoneType
        :type gpu_count: int
        :type gpu_model: str
        :type io_guarantee_ssd: int | NoneType
        :type io_guarantee_hdd: int | NoneType
        :type audio: int | NoneType
        :type network_bandwidth_guarantee: int
        :type ip4_address_pool_id: str | NoneType
        """
        req = vmset_api_pb2.CreateVmRequest()
        req.forced_node_id = node_id
        req.meta.id = pod_id
        if logins:
            req.meta.auth.owners.logins.extend(logins)
        if groups:
            req.meta.auth.owners.group_ids.extend(groups)
        req.spec.type = vmset_pb2.VMSpec.QEMU_VM
        req.spec.qemu.network_id = network_id
        req.spec.qemu.node_segment = node_segment
        req.spec.qemu.resource_requests.dirty_memory_limit = 0
        req.spec.qemu.resource_requests.memory_limit = mem
        req.spec.qemu.resource_requests.anonymous_memory_limit = 0
        req.spec.qemu.resource_requests.vcpu_guarantee = vcpu_guarantee
        req.spec.qemu.resource_requests.vcpu_limit = vcpu_limit
        req.spec.qemu.resource_requests.memory_guarantee = mem
        req.spec.qemu.resource_requests.network_bandwidth_guarantee = network_bandwidth_guarantee
        req.spec.qemu.enable_internet = enable_internet
        req.spec.qemu.use_nat64 = use_nat64
        if ip4_address_pool_id is not None:
            req.spec.qemu.ip4_address_pool_id = ip4_address_pool_id
        req.spec.qemu.autorun = autorun
        req.spec.qemu.vm_type = vmagent_pb2.VMConfig.VMType.Value(vm_type.upper())
        req.spec.vmagent_version = vmagent_version
        if layer_url:
            req.spec.qemu.porto_layer.url = layer_url

        if isinstance(custom_pod_resources, dict):
            for resource_key, resource_info in custom_pod_resources.items():
                new_resource = req.spec.qemu.pod_resources[resource_key]
                new_resource.CopyFrom(resource_info)

        v = req.spec.qemu.volumes.add()  # type: vmset_pb2.Volume
        v.name = defines.DEFAULT_VOLUME_MP
        v.capacity = volume_size
        v.storage_class = storage_class
        v.resource_url = rb_torrent
        v.image_type = vmset_pb2.Volume.ImageType.Value(image_type)
        v.vm_mount_path = '/'

        if audio is not None:
            req.spec.qemu.qemu_options.audio = audio

        if extra_volumes is not None:
            for extra_volume in extra_volumes:
                v = req.spec.qemu.volumes.add()
                v.CopyFrom(extra_volume)

        if gpu_model:
            req.spec.qemu.gpu_request.model = gpu_model
        if gpu_count:
            req.spec.qemu.gpu_request.capacity = gpu_count

        if io_guarantee_ssd is not None:
            req.spec.qemu.io_guarantees_per_storage['ssd'] = io_guarantee_ssd
        if io_guarantee_hdd is not None:
            req.spec.qemu.io_guarantees_per_storage['hdd'] = io_guarantee_hdd

        if abc == 'personal' or abc == defines.PERSONAL_ACCOUNT.split('abc:service:')[1]:
            self._validate_personal_quota_size(req.spec)
            req.spec.account_id = defines.PERSONAL_ACCOUNT
        elif abc:
            req.spec.account_id = 'abc:service:{}'.format(abc)
        if cluster == 'ANY':
            self._any_cluster_request(req, self._create_request)
        else:
            self._create_request(req, cluster)

    def _allocate_request(self, req, cluster):
        """
        :type req: vmset_api_pb2.AllocateVmRequest
        :type cluster: str
        """
        rpc_stub = self.get_rpc_stub(cluster)
        rpc_stub.allocate_vm(req)
        print 'Allocating {} in {}'.format(req.meta.id, cluster)

    def allocate(self, cluster, pod_id, network_id, node_segment, volume_size, mem, vcpu_limit, vcpu_guarantee,
                 storage_class, enable_internet, use_nat64, abc, logins=None, groups=None, layer_url=None):
        """
        :type cluster: str
        :type pod_id: str
        :type network_id: str
        :type node_segment: str
        :type volume_size: int
        :type mem: int
        :type vcpu_limit: int
        :type vcpu_guarantee: int
        :type storage_class: str
        :type enable_internet: bool
        :type use_nat64: bool
        :type abc: str
        :type logins: list[str] | NoneType
        :type groups: list[str] | NoneType
        :type layer_url: str | NoneType
        """
        req = vmset_api_pb2.AllocateVmRequest()
        req.meta.id = pod_id
        if logins:
            req.meta.auth.owners.logins.extend(logins)
        if groups:
            req.meta.auth.owners.group_ids.extend(groups)
        req.spec.type = vmset_pb2.VMSpec.QEMU_VM
        req.spec.qemu.network_id = network_id
        req.spec.qemu.node_segment = node_segment
        req.spec.qemu.resource_requests.dirty_memory_limit = 0
        req.spec.qemu.resource_requests.memory_limit = mem
        req.spec.qemu.resource_requests.anonymous_memory_limit = 0
        req.spec.qemu.resource_requests.vcpu_guarantee = vcpu_guarantee
        req.spec.qemu.resource_requests.vcpu_limit = vcpu_limit
        req.spec.qemu.resource_requests.memory_guarantee = mem
        req.spec.qemu.enable_internet = enable_internet
        req.spec.qemu.use_nat64 = use_nat64
        if layer_url:
            req.spec.qemu.porto_layer.url = layer_url
        v = req.spec.qemu.volumes.add()
        v.name = defines.DEFAULT_VOLUME_MP
        v.capacity = volume_size
        v.storage_class = storage_class
        if abc == 'personal' or abc == defines.PERSONAL_ACCOUNT.split('abc:service:')[1]:
            self._validate_personal_quota_size(req.spec)
            req.spec.account_id = defines.PERSONAL_ACCOUNT
        elif abc:
            req.spec.account_id = 'abc:service:{}'.format(abc)
        if cluster == 'ANY':
            self._any_cluster_request(req, self._allocate_request)
        else:
            self._allocate_request(req, cluster)

    def list_accounts(self, cluster, login, node_segment=None):
        """
        :type cluster: str
        :type login: str
        :type node_segment: str | NoneType
        :rtype: list
        """
        req = vmset_api_pb2.ListUserAccountsRequest()
        req.login = login
        if node_segment:
            req.segment = node_segment
        return self.get_rpc_stub(cluster).list_user_accounts(req).accounts

    def list_yp_vms(self, cluster, login=None, node_segment=None, abc=None, filter_by_name=None,
                    skip=0, limit=0, labels_filters=None):
        """
        :type cluster: str
        :type login: str | NoneType
        :type node_segment: list[str] | NoneType
        :type abc: list[str] | NoneType
        :type filter_by_name: str | NoneType
        :type skip: int
        :type limit: int
        :type labels_filters: dict[str, vmset_pb2.YpVmFindQueryBuilder] | NoneType
        :rtype collections.Iterable[vmset_pb2.VM]
        """
        req = vmset_api_pb2.ListYpVmRequest()
        if login:
            req.query.login = login
        if node_segment:
            req.query.segment.extend(node_segment)
        if filter_by_name:
            req.query.name = filter_by_name
        if abc:
            req.query.account.extend(abc)

        if isinstance(labels_filters, dict) and labels_filters:
            for k, v in labels_filters.items():
                req.query.labels_filters[k].CopyFrom(v)

        req.skip = skip
        req.limit = limit
        rpc_stub = self.get_rpc_stub(cluster)
        rsp = rpc_stub.list_yp_vm(req)
        return rsp.vms

    def get_vm_stats(self, cluster, node_segment=None, abc=None, consistency=vmset_api_pb2.WEAK):
        """

        :type cluster: str
        :type node_segment: [str]
        :type abc: [str]
        :type consistency:
        :return:
        """
        req = vmset_api_pb2.GetVmStatsRequest(consistency=consistency)
        if abc:
            req.account.extend(abc)

        if node_segment:
            req.segment.extend(node_segment)

        rpc_stub = self.get_rpc_stub(cluster)
        rsp = rpc_stub.get_vm_stats(req)
        return rsp

    def deallocate(self, cluster, pod_id):
        """
        :type cluster: str
        :type pod_id: str
        """
        req = vmset_api_pb2.DeallocateVmRequest()
        req.id = pod_id
        rpc_stub = self.get_rpc_stub(cluster)
        rpc_stub.deallocate_vm(req)

    def _init_update_vm_request(self, rpc_stub, pod_id):
        """

        :type rpc_stub: vmset_api_stub.VmSetServiceStub
        :type pod_id: str
        :rtype: (vmset_api_pb2.UpdateVmRequest, vmset_pb2.VM)
        """
        req = vmset_api_pb2.GetVmRequest()
        req.vm_id = pod_id
        vm = rpc_stub.get_vm(req).vm
        req = vmset_api_pb2.UpdateVmRequest()
        req.meta.CopyFrom(vm.meta)
        req.spec.CopyFrom(vm.spec)
        req.config.CopyFrom(vm.config)
        return req, vm

    def update(self,
               cluster,
               pod_id,
               abc,
               logins=None,
               groups=None,
               clear_groups=False,
               network_id=None,
               use_nat64=None,
               vcpu_limit=None,
               vcpu_guarantee=None,
               autorun=None,
               vcpu=None,
               mem_config=None,
               rb_torrent=None,
               vm_type=None,
               image_type=None,
               mem=None,
               update_vmagent=False,
               node_id=None,
               remove_node_id=None,
               changed_volumes=None,
               removed_volumes=None,
               conf_volumes=None,
               gpu_count=None,
               gpu_model=None,
               io_guarantee_ssd=None,
               io_guarantee_hdd=None,
               audio=None,
               network_bandwidth_guarantee=None,
               ):
        """
        :type cluster: str
        :type pod_id: str
        :type abc: str | None
        :type logins: list[str]
        :type groups: list[str]
        :type clear_groups: bool
        :type network_id: str
        :type use_nat64: bool
        :type vcpu_limit: int
        :type vcpu_guarantee: int
        :type autorun: bool
        :type vcpu: int
        :type mem_config: int
        :type rb_torrent: str
        :type vm_type: str | NoneType
        :type image_type: str | NoneType
        :type mem: int
        :type update_vmagent: bool
        :type node_id: str
        :type remove_node_id: bool
        :type changed_volumes: list[vmset_pb2.Volume] | NoneType
        :type removed_volumes: list[str] | NoneType
        :type conf_volumes: list[vmset_pb2.Volume] | NoneType
        :type gpu_count: int
        :type gpu_model: str
        :type io_guarantee_ssd: int | NoneType
        :type io_guarantee_hdd: int | NoneType
        :type audio: int | NoneType
        :type network_bandwidth_guarantee: int | NoneType
        :rtype:
        """
        rpc_stub = self.get_rpc_stub(cluster)
        req, vm = self._init_update_vm_request(rpc_stub, pod_id)
        req.update_vmagent = update_vmagent
        main_volume = None
        for v in req.spec.qemu.volumes:
            if v.name == defines.DEFAULT_VOLUME_MP:
                main_volume = v
                break

        if logins:
            req.meta.auth.owners.ClearField("logins")
            req.meta.auth.owners.logins.extend(sorted(list(set(logins))))

        if groups:
            req.meta.auth.owners.ClearField("group_ids")
            req.meta.auth.owners.group_ids.extend(sorted(list(set(groups))))

        if clear_groups:
            req.meta.auth.owners.ClearField("group_ids")

        if vm.spec.qemu.node_segment in (defines.NODE_SEGMENT_DEFAULT, defines.ARM64_SEGMENT_DEV):
            if vcpu_guarantee:
                vcpu_limit = vcpu_guarantee
            elif not vcpu_guarantee and vcpu_limit and vcpu_limit != req.spec.qemu.resource_requests.vcpu_guarantee:
                vcpu_limit = req.spec.qemu.resource_requests.vcpu_guarantee

        if vm.spec.qemu.node_segment == defines.NODE_SEGMENT_DEV:
            if vcpu_guarantee and not vcpu_limit and vcpu_guarantee > req.spec.qemu.resource_requests.vcpu_limit:
                vcpu_limit = vcpu_guarantee

        if vcpu_limit:
            vcpu = int(vcpu_limit / 1000)

        if removed_volumes:
            new_volumes = [v for v in req.spec.qemu.volumes if v.name not in removed_volumes]
            del req.spec.qemu.volumes[:]
            req.spec.qemu.volumes.extend(new_volumes)

        if changed_volumes:
            _changed_volumes = {v.name: v for v in changed_volumes}
            for exists_volume in req.spec.qemu.volumes:
                updated = _changed_volumes.pop(exists_volume.name, None)
                if updated:
                    exists_volume.CopyFrom(updated)
            for new_volume in _changed_volumes.values():
                qemu_volume = req.spec.qemu.volumes.add()
                qemu_volume.CopyFrom(new_volume)

        if conf_volumes:
            del req.spec.qemu.volumes[1:]
            req.spec.qemu.volumes.extend(conf_volumes)

        new_spec = vmset_pb2.VMSpec()
        new_spec.qemu.network_id = network_id or ''
        new_spec.qemu.resource_requests.vcpu_guarantee = vcpu_guarantee or 0
        new_spec.qemu.resource_requests.vcpu_limit = vcpu_limit or 0
        new_spec.qemu.resource_requests.memory_limit = mem or 0
        new_spec.qemu.resource_requests.memory_guarantee = mem or 0
        new_spec.qemu.resource_requests.network_bandwidth_guarantee = network_bandwidth_guarantee or 0
        if node_id:
            new_spec.qemu.forced_node_id = node_id

        if remove_node_id:
            new_spec.qemu.forced_node_id = ''

        new_config = vmagent_pb2.VMConfig()
        new_config.vcpu = vcpu or 0
        new_config.mem = mem_config or 0

        req.spec.MergeFrom(new_spec)
        req.config.MergeFrom(new_config)

        if use_nat64 is not None:
            req.spec.qemu.use_nat64 = use_nat64

        if audio is not None:
            req.spec.qemu.qemu_options.audio = audio
            req.config.audio = audio

        if autorun is not None:
            req.spec.qemu.autorun = autorun
            req.config.autorun = autorun

        if rb_torrent is not None:
            req.config.disk.resource.rb_torrent = rb_torrent
            main_volume.resource_url = rb_torrent

        if gpu_count is not None:
            if not gpu_model and not req.spec.qemu.gpu_request.model:
                raise ValueError('Gpu model has not been set')
            req.spec.qemu.gpu_request.capacity = gpu_count

        if gpu_model is not None:
            if not gpu_count and not req.spec.qemu.gpu_request.capacity:
                raise ValueError('Gpu count has not been set')
            req.spec.qemu.gpu_request.model = gpu_model

        if io_guarantee_ssd is not None:
            req.spec.qemu.io_guarantees_per_storage['ssd'] = io_guarantee_ssd
        if io_guarantee_hdd is not None:
            req.spec.qemu.io_guarantees_per_storage['hdd'] = io_guarantee_hdd

        # Explicitly set enum type fields
        if vm_type is not None:
            vm_type_value = vmagent_pb2.VMConfig.VMType.Value(vm_type.upper())
            req.config.type = vm_type_value
            req.spec.qemu.vm_type = vm_type_value
        if image_type is not None:
            req.config.disk.type = image_type
            main_volume.image_type = image_type

        if abc == 'personal' or abc == defines.PERSONAL_ACCOUNT.split('abc:service:')[1]:
            # Set personal account validate new spec
            self._validate_personal_quota_size(req.spec)
            req.spec.account_id = defines.PERSONAL_ACCOUNT
        elif abc:
            # Set non-personal account, no validation needed
            req.spec.account_id = 'abc:service:{}'.format(abc)
        elif vm.spec.account_id == defines.PERSONAL_ACCOUNT:
            # VM already in personal account, no changes. Validate spec
            self._validate_personal_vm_spec_change(vm.spec, req.spec, vm.meta.author)

        if node_id and req.spec.account_id == defines.PERSONAL_ACCOUNT:
            raise ValueError('Forcing node in personal quota is not allowed')

        return rpc_stub.update_vm(req)

    def update_vmagent(self,
                       cluster,
                       pod_id,
                       update_vmagent_version=None,
                       custom_pod_resources=None,
                       ):
        """
        update_vmagent_version and custom_pod_resources can use only root users

        :type cluster: str
        :type pod_id: str
        :type update_vmagent_version: str | NoneType
        :type custom_pod_resources: dict[str, vmset_pb2.PodResource]
        :rtype: vmset_api_pb2.UpdateVmResponse
        """
        rpc_stub = self.get_rpc_stub(cluster)
        update_request, vm = self._init_update_vm_request(rpc_stub, pod_id)

        update_request.update_vmagent = True
        if isinstance(custom_pod_resources, dict):
            for resource_key, resource_info in custom_pod_resources.items():
                new_resource = update_request.spec.qemu.pod_resources[resource_key]
                new_resource.CopyFrom(resource_info)

        if update_vmagent_version:
            update_request.spec.vmagent_version = update_vmagent_version

        return rpc_stub.update_vm(update_request)

    def update_labels(self, cluster, pod_id, labels, clear_exists_labels=False):
        """
        :type cluster: str
        :type pod_id: str
        :type labels: dict
        :type clear_exists_labels: bool
        :rtype: vmset_api_pb2.UpdateVmResponse
        """
        rpc_stub = self.get_rpc_stub(cluster)
        req, vm = self._init_update_vm_request(rpc_stub, pod_id)
        if not isinstance(labels, dict):
            raise ValueError('labels should be instance of dict')
        req.update_labels = True
        if clear_exists_labels:
            for k, v in req.spec.labels.items():
                del req.spec.labels[k]
        req.spec.labels.update(labels)
        return rpc_stub.update_vm(req)

    def get_vm(self, cluster, pod_id):
        """
        :type cluster: str
        :type pod_id: str
        :rtype: vmset_pb2.VM
        """
        rpc_stub = self.get_rpc_stub(cluster)
        req = vmset_api_pb2.GetVmRequest()
        req.vm_id = pod_id
        return rpc_stub.get_vm(req).vm

    def exists_vm(self, cluster, pod_id):
        """
        :type cluster: str
        :type pod_id: str
        :rtype: vmset_pb2.VM
        """
        rpc_stub = self.get_rpc_stub(cluster)
        req = vmset_api_pb2.GetVmRequest()
        req.vm_id = pod_id
        try:
            return rpc_stub.get_vm(req).vm.meta.id == pod_id
        except nanny_rpc_client.exceptions.NotFoundError, nanny_rpc_client.exceptions.ForbiddenError:
            return False

    def list_backups(self, cluster, pod_id):
        """
        :type cluster: str
        :type pod_id: str
        :rtype: collections.Iterable[vmset_pb2.Backup]
        """
        rpc_stub = self.get_rpc_stub(cluster)
        req = vmset_api_pb2.ListBackupRequest()
        req.vm_id = pod_id
        return rpc_stub.list_backup(req).backups

    def remove_backup(self, cluster, pod_id, backup_id):
        """
        :type cluster: str
        :type pod_id: str
        :type backup_id: str
        """
        rpc_stub = self.get_rpc_stub(cluster)
        req = vmset_api_pb2.RemoveBackupRequest()
        req.vm_id = pod_id
        req.id = backup_id
        rpc_stub.remove_backup(req)

    def get_status(self, cluster, pod_id):
        """

        :type cluster: str
        :type pod_id: str
        :rtype: vmset_api_pb2.GetStatusResponse

        """
        req = vmset_api_pb2.GetStatusRequest()
        req.vm_id.pod_id = pod_id
        rpc_stub = self.get_rpc_stub(cluster)
        return rpc_stub.get_status(req)

    def wait_status_change(self, cluster, pod_id, wait_iteration_timeout=defines.WAIT_ITERATION_TIMEOUT):
        # type: (str, str, int) -> list[tuple[bool, (str, )]]
        while True:
            try:
                status = self.get_status(cluster=cluster, pod_id=pod_id)
            except requests.exceptions.HTTPError as e:
                yield False, str(e.response.content) or str(e)
            except nanny_rpc_client.exceptions.BadRequestError as e:
                _msg_starts_with = 'Agent has not been started yet. Current status: '
                if e.message != 'Yp pod has not been scheduled yet' and not e.message.startswith(_msg_starts_with):
                    raise e
                yield False, e.message.replace(_msg_starts_with, '')
            else:
                if status.state.type in defines.FINITE_STATES:
                    yield True, status
                    break
                yield False, status
            time.sleep(wait_iteration_timeout)

    def backup(self, cluster=None, pod_id=None, host=None, port=None, service=None, backup_storage=None):
        req = vmset_api_pb2.BackupVmRequest()
        if cluster and pod_id:
            req.vm_id.pod_id = pod_id
            if backup_storage:
                req.backup_storage = backup_storage
            rpc_stub = self.get_rpc_stub(cluster)
        elif host and port and service:
            req.vm_id.nanny_args.host = host
            req.vm_id.nanny_args.port = port
            req.vm_id.nanny_args.service = service
            rpc_stub = self.get_rpc_stub(defines.DEFAULT_LOCATION)
        else:
            raise ValueError('(yp_cluster and pod_id) or (host and port and service) should be passed')
        rpc_stub.backup_vm(req)

    def restore_backup(self, cluster, pod_id, resource_url):
        req = vmset_api_pb2.RestoreBackupRequest()
        req.vm_id = pod_id
        req.resource_url = resource_url
        rpc_stub = self.get_rpc_stub(cluster)
        rpc_stub.restore_backup(req)

    def list_free_nodes(self, cluster, node_segment):
        rpc_stub = self.get_rpc_stub(cluster)
        req = vmset_api_pb2.ListFreeNodesRequest()
        req.segment = node_segment
        return rpc_stub.list_free_nodes(req)

    def check_qyp_vmagent_version(self, yp_cluster, pod_id, version):
        vm = self.get_vm(yp_cluster, pod_id)
        if vm.spec.vmagent_version == 'N/A':
            used_version = LooseVersion('0')
        else:
            used_version = LooseVersion(vm.spec.vmagent_version)
        return used_version >= version

    def guess_proper_yp_cluster(self, pod_id):
        def iter_clusters():
            for cluster in defines.VMPROXY_LOCATION.keys():
                if self.exists_vm(cluster, pod_id):
                    yield cluster

        if self.custom_proxyhost:
            return 'ANY' if self.exists_vm('ANY', pod_id) else None
        else:
            clusters = list(frozenset(iter_clusters()))
            if len(clusters) == 1:
                return clusters[0]

    def _init_personal_account_helper(self, login):
        """
        :type login: str
        :rtype: helpers.PersonalAccountHelper
        """
        accounts = []
        for cluster in defines.VMPROXY_LOCATION.keys():
            accounts.extend(self.list_accounts(cluster, login, 'dev'))
        pa_helper = helpers.PersonalAccountHelper(login=login)
        pa_helper.init(accounts=accounts)
        return pa_helper

    def _validate_personal_quota_size(self, vm_spec):
        """
        :type vm_spec: vmset_pb2.VMSpec
        """
        login = getpass.getuser()
        pa_helper = self._init_personal_account_helper(login)
        vol = next(v for v in vm_spec.qemu.volumes if v.name == defines.DEFAULT_VOLUME_MP)
        has_ip4 = vm_spec.qemu.enable_internet or bool(vm_spec.qemu.ip4_address_pool_id)
        pa_helper.validate_new_resource_fit(cpu=vm_spec.qemu.resource_requests.vcpu_guarantee,
                                            mem=vm_spec.qemu.resource_requests.memory_guarantee,
                                            disk_size=vol.capacity,
                                            storage_class=vol.storage_class,
                                            enable_internet=has_ip4)

    def _validate_personal_vm_spec_change(self, vm_spec, new_spec, login):
        """
        Validate that updated spec fits personal quota. Check only cpu and mem

        :type vm_spec: vmset_pb2.VMSpec
        :type new_spec: vmset_pb2.VMSpec
        :type login: str
        """
        pa_helper = self._init_personal_account_helper(login)
        vol = next(v for v in vm_spec.qemu.volumes if v.name == defines.DEFAULT_VOLUME_MP)
        new_cpu = new_spec.qemu.resource_requests.vcpu_guarantee
        cpu = new_cpu - vm_spec.qemu.resource_requests.vcpu_guarantee if new_cpu else 0
        new_mem = new_spec.qemu.resource_requests.memory_guarantee
        mem = new_mem - vm_spec.qemu.resource_requests.memory_guarantee if new_mem else 0
        pa_helper.validate_new_resource_fit(
            cpu=cpu,
            mem=mem,
            disk_size=0,
            storage_class=vol.storage_class,
            enable_internet=False
        )

    def get_vm_from_backup(self,  cluster, qdm_res_id):
        rpc_stub = self.get_rpc_stub(cluster)
        req = vmset_api_pb2.GetVmFromBackupRequest()
        req.qdm_res_id = qdm_res_id
        return rpc_stub.get_vm_from_backup(req)

    def acknowledge_eviction(self, cluster, vm_id, qdm_res_id):
        rpc_stub = self.get_rpc_stub(cluster)
        req = vmset_api_pb2.AcknowledgeEvictionRequest()
        req.vm_id = vm_id
        req.qdm_res_id = qdm_res_id
        return rpc_stub.acknowledge_eviction(req)

    def stop_backup(self, cluster, pod_id):
        rpc_stub = self.get_rpc_stub(cluster)
        req = vmset_api_pb2.MakeActionRequest()
        req.vm_id.pod_id = pod_id
        req.action = vmagent_api_pb2.VMActionRequest.STOP_QDMUPLOAD
        rpc_stub.make_action(req)


class VMAgentClient(object):
    @classmethod
    def from_action_options(cls, args, vm_proxy_client):
        return cls(
            vm_proxy_client=vm_proxy_client,
            host=args.host,
            port=args.port,
            service=args.service,
            pod_id=args.pod_id,
            yp_cluster=args.yp_cluster,
            direct=args.direct
        )

    def __init__(self, vm_proxy_client=None, host='', port='', service='', pod_id='', yp_cluster='', direct=False):
        # type: (VMProxyClient, str, str, str, str, str, bool) -> None
        self.headers = vm_proxy_client.headers if vm_proxy_client else {}
        self.timeout = vm_proxy_client.timeout if vm_proxy_client else defines.TIMEOUT
        self.ssl_verify = vm_proxy_client.ssl_verify if vm_proxy_client else True
        self.vmproxy_client = vm_proxy_client

        self.direct = direct

        self.pod_id = pod_id or ''
        self.yp_cluster = yp_cluster.upper() if yp_cluster else None

        self.host = host
        self.port = port
        self.service = service

        if self.direct:
            self.use_vmproxy_stub = False
            if not self.host:
                self.host = "[::1]"
            if not self.port:
                self.port = os.environ.get('SERVICE_PORT', 7255)
            link_tpl = defines.DIRECT_LINK
            if os.path.exists(defines.LOCAL_TOKEN_PATH):
                with open(defines.LOCAL_TOKEN_PATH) as f:
                    self.headers['Local-Token'] = f.read()
            self.action_link = link_tpl.format(path="action", host=self.host, port=self.port)
            self.status_link = link_tpl.format(path="status", host=self.host, port=self.port)
            self.debug_link = link_tpl.format(path="debug", host=self.host, port=self.port)

        elif self.host and self.service and self.port:
            # GENCFG VM
            self.use_vmproxy_stub = False
            proxyhost = vm_proxy_client.custom_proxyhost or defines.GENCFG_PROXYHOST

            url_path = partial(defines.VMPROXY_LINK_TPL.format,
                               host=self.host,
                               port=self.port,
                               service=self.service,
                               pod_id=self.pod_id)

            self.action_link = proxyhost + url_path(path="action")
            self.status_link = proxyhost + url_path(path="status")
            self.debug_link = proxyhost + url_path(path="debug")
        elif self.pod_id and self.yp_cluster:
            # YP VM
            self.use_vmproxy_stub = True
        else:
            raise errors.VMAgentParamsError('(yp_cluster and pod_id) or '
                                            '(host and port and service) or '
                                            '(direct and host and port) '
                                            'should be passed')

        self.vm_id_proto = self.build_vm_id_proto(self.pod_id, self.host, self.port, self.service)

    def build_vm_id_proto(self, pod_id=None, host='', port='', service=''):
        vm_id_proto = vmset_api_pb2.VmId()
        if pod_id:
            vm_id_proto.pod_id = pod_id
        else:
            vm_id_proto.nanny_args.host = host
            vm_id_proto.nanny_args.port = port
            vm_id_proto.nanny_args.service = service
        return vm_id_proto

    def get_status(self, find_issues=False):
        """

        :rtype: vmset_api_pb2.GetStatusResponse
        """
        if self.use_vmproxy_stub:
            return self.vmproxy_client.get_status(
                cluster=self.yp_cluster,
                pod_id=self.pod_id,
            )
        else:
            status_link = self.status_link
            if find_issues:
                status_link += 'find_issues=1'
            resp = requests.get(
                url=status_link,
                headers=self.headers,
                timeout=self.timeout,
                verify=self.ssl_verify
            )
            resp.raise_for_status()
            r = vmagent_api_pb2.VMStatusResponse()
            r.ParseFromString(base64.decodestring(resp.content))

            result = vmset_api_pb2.GetStatusResponse()
            result.state.CopyFrom(r.state)
            result.config.CopyFrom(r.config)
            result.vmagent_version = r.vmagent_version
            return result

    def send_action(self, req):
        """
        :type req: vmset_api_pb2.MakeActionRequest
        """
        if self.use_vmproxy_stub:
            stub = self.vmproxy_client.get_rpc_stub(self.yp_cluster)
            req.vm_id.CopyFrom(self.vm_id_proto)
            stub.make_action(req)
        else:
            direct_request = helpers.cast_vmset_to_vmagent_action_req(req)
            resp = requests.post(
                url=self.action_link,
                data=base64.encodestring(direct_request.SerializeToString()),
                headers=self.headers,
                timeout=self.timeout,
                verify=self.ssl_verify
            )
            resp.raise_for_status()

    def start(self):
        req = vmset_api_pb2.MakeActionRequest()
        req.action = vmagent_api_pb2.VMActionRequest.START
        self.send_action(req)

    def rescue(self):
        req = vmset_api_pb2.MakeActionRequest()
        req.action = vmagent_api_pb2.VMActionRequest.RESCUE
        self.send_action(req)

    def poweroff(self):
        req = vmset_api_pb2.MakeActionRequest()
        req.action = vmagent_api_pb2.VMActionRequest.POWEROFF
        self.send_action(req)

    def reset(self):
        req = vmset_api_pb2.MakeActionRequest()
        req.action = vmagent_api_pb2.VMActionRequest.RESET
        self.send_action(req)

    def restart(self):
        req = vmset_api_pb2.MakeActionRequest()
        req.action = vmagent_api_pb2.VMActionRequest.RESTART
        self.send_action(req)

    def shutdown(self):
        req = vmset_api_pb2.MakeActionRequest()
        req.action = vmagent_api_pb2.VMActionRequest.SHUTDOWN
        self.send_action(req)

    def revert(self):
        req = vmset_api_pb2.MakeActionRequest()
        req.action = vmagent_api_pb2.VMActionRequest.HARD_RESET
        self.send_action(req)

    def share_image(self):
        req = vmset_api_pb2.MakeActionRequest()
        req.action = vmagent_api_pb2.VMActionRequest.SHARE_IMAGE
        self.send_action(req)

    def push_config(self, id, vcpu, mem, rb_torrent, disk_size, vm_type, autorun, image_type, raw_req=None):
        """
        :type image_type: str
        :type id: str
        :type vcpu: int
        :type mem: int
        :type rb_torrent: str
        :type disk_size: int
        :type vm_type: str
        :type autorun: bool
        :type raw_req: google.protobuf.message.Message | NoneType
        """
        if raw_req:
            return self.send_action(raw_req)
        req = vmset_api_pb2.MakeActionRequest()
        req.action = vmagent_api_pb2.VMActionRequest.PUSH_CONFIG
        req.config.id = id
        req.config.vcpu = vcpu
        req.config.mem = mem
        req.config.disk.resource.rb_torrent = rb_torrent
        req.config.disk.delta_size = disk_size
        req.config.autorun = autorun
        req.config.disk.type = vmagent_pb2.VMDisk.ImageType.Value(image_type)
        if vm_type.lower() == "linux":
            req.config.type = vmagent_pb2.VMConfig.LINUX
        elif vm_type.lower() == "windows":
            req.config.type = vmagent_pb2.VMConfig.WINDOWS
        else:
            raise errors.VMAgentParamsError('Type {} not supported'.format(vm_type))
        self.send_action(req)

    def send_debug(self):
        if self.use_vmproxy_stub:
            raise RuntimeError('send_debug work only in direct mode')
        resp = requests.post(
            url=self.debug_link,
            headers=self.headers,
            timeout=self.timeout,
            verify=self.ssl_verify
        )
        resp.raise_for_status()
