import click


class OptionEatAll(click.Option):

    def __init__(self, *args, **kwargs):
        nargs = kwargs.pop('nargs', -1)
        assert nargs == -1, 'nargs, if set, must be -1 not {}'.format(nargs)
        super(OptionEatAll, self).__init__(*args, **kwargs)
        self._previous_parser_process = None
        self._eat_all_parser = None

    def add_to_parser(self, parser, ctx):

        def parser_process(value, state):
            # method to hook to the parser.process
            done = False
            value = [value]
            while state.rargs and not done:
                for prefix in self._eat_all_parser.prefixes:
                    if state.rargs[0].startswith(prefix):
                        done = True
                if not done:
                    value.append(state.rargs.pop(0))

            value = tuple(value)

            self._previous_parser_process(value, state)

        retval = super(OptionEatAll, self).add_to_parser(parser, ctx)
        for name in self.opts:
            our_parser = parser._long_opt.get(name) or parser._short_opt.get(name)
            if our_parser:
                self._eat_all_parser = our_parser
                self._previous_parser_process = our_parser.process
                our_parser.process = parser_process
                break
        return retval

    def value_from_envvar(self, ctx):
        rv = self.resolve_envvar_value(ctx)
        if rv is None:
            return None
        return self.type.split_envvar_value(rv)


class OptionValue(object):
    def __init__(self, *param_decls, **attrs):
        self._param_decls = param_decls
        self._attrs = attrs
        self.help = attrs.get('help')
        self.order = attrs.pop('order', 1000)
        self.destination_name = None

    def _fix_up(self, name):
        self.destination_name = name

    def __get__(self, instance, owner):
        if instance is None:
            return self

        if isinstance(instance, CommandOptions):
            return instance.get_value(self.destination_name)

    def __set__(self, instance, value):
        if instance is None:
            raise ValueError('Can not set OptionValue value for class')

        if isinstance(instance, CommandOptions):
            instance.set_value(self.destination_name, value)

    @property
    def decorate(self):
        _param_decls = list(self._param_decls)
        if self.destination_name not in self._param_decls:
            _param_decls.append(self.destination_name)
        return click.option(*_param_decls, **self._attrs)


class CommandOptionsMeta(type):
    def __init__(cls, name, bases, attrs):
        super(CommandOptionsMeta, cls).__init__(name, bases, attrs)
        cls._fix_up_properties()


class CommandOptions(object):
    __metaclass__ = CommandOptionsMeta
    _properties = None

    @classmethod
    def _fix_up_properties(cls):
        cls._properties = []
        for name in set(dir(cls)):
            attr = getattr(cls, name, None)
            if isinstance(attr, OptionValue):
                attr._fix_up(name)
                cls._properties.append((attr.order, name))

    @classmethod
    def build_from_kwargs(cls, kw, validate=True):
        args = cls(kw)
        if validate:
            try:
                args.validate()
            except ValueError as e:
                ctx = click.get_current_context()
                verbose = ctx.parent.params.get('verbose', False)
                if verbose:
                    raise e
                else:
                    raise click.BadOptionUsage(e.message)
        return args

    @classmethod
    def decorate(cls, f):
        for _, name in sorted(cls._properties, reverse=True):
            attr = getattr(cls, name, None)
            if not attr.help:
                raise ValueError('help required for class {} property {}'.format(cls.__name__, name))

            if isinstance(attr, OptionValue):
                f = attr.decorate(f)
        return f

    def __init__(self, command_options):
        self._command_options = command_options

    def get_value(self, name):
        return self._command_options[name]

    def set_value(self, name, value):
        self._command_options[name] = value

    @property
    def command_options(self):
        return self._command_options

    def debug(self):
        for n, v in self._command_options.items():
            print n, v

    def validate(self):
        pass
