import argparse
import getpass
import re
from infra.qyp.proto_lib import vmset_api_pb2, vmset_pb2

import click
import yaml
from infra.qyp.vmctl.src import defines, api, helpers
from infra.qyp.vmctl.src.actions import list_action
from infra.qyp.vmctl.src.common import CommandOptions, OptionValue, OptionEatAll


class PodIdType(click.ParamType):
    POD_ID_PATTERN = re.compile("^[a-z0-9\-]+$")

    def convert(self, value, param, ctx):  # type: (str, click.Option, click.Context) -> str
        if not self.POD_ID_PATTERN.match(value.lstrip()):
            self.fail("{} should match the pattern: [a-z0-9\-]+".format(value), param, ctx)
        return value

    def get_metavar(self, param):
        return 'POD_ID'


class NetworkIdType(click.ParamType):
    MACRO_PATTERN = re.compile("^_[A-Z0-9_]+_$")

    def convert(self, value, param, ctx):  # type: (str, click.Option, click.Context) -> str
        if not self.MACRO_PATTERN.match(value.lstrip()):
            msg = 'Invalid racktables macro \"{}\", ' \
                  'should match the pattern: _[A-Z0-9_]+_'.format(value)
            self.fail(msg, param, ctx)
        return value

    def get_metavar(self, param):
        return "NETWORK_ID"


class RepeatedChoices(click.ParamType):
    def __init__(self, choices, fail_msg='Invalid value'):
        self._choices = choices
        self._fail_msg = fail_msg

    def convert(self, value, param, ctx):  # type: (str, click.Option, click.Context) -> list[str]
        if isinstance(value, basestring):
            value = [value]
        for v in value:
            if v not in self._choices:
                msg = '{} "{}", should in {}'.format(self._fail_msg, v, self._choices)
                self.fail(msg, param, ctx)
        return value

    def get_metavar(self, param):
        return "[[{}] ...]".format("|".join(self._choices))


class RepeatedString(click.ParamType):

    def convert(self, value, param, ctx):  # type: (str, click.Option, click.Context) -> list[str]
        if isinstance(value, basestring):
            value = [value]
        return value

    def get_metavar(self, param):
        return "[TEXT ...]"


class SizeSuffixStringType(click.ParamType):
    SIZE_PATTERN = re.compile('^([0-9]+)([KkMmGgTt]?)$')

    def convert(self, value, param, ctx):
        if not value:
            return value

        res = self.SIZE_PATTERN.match(value)

        if not res:
            msg = 'Invalid size value \"{}\", valid suffixes are Kk, Mm, Gg and Tt or none'.format(value)
            self.fail(msg, param, ctx)

        num_str, suffix = res.groups()

        numeric = int(num_str)

        if suffix.lower() == 'k':
            numeric *= 1024

        elif suffix.lower() == "m":
            numeric *= 1048576

        elif suffix.lower() == "g":
            numeric *= 1048576 * 1024

        elif suffix.lower() == "t":
            numeric *= 1048576 * 1048576

        return numeric

    def get_metavar(self, param):
        return "SIZE"


def cast_dict_to_volume(obj_dict, param, ctx):
    if 'name' not in obj_dict:
        msg = '"name" should be set for all extra volumes'
        raise click.BadParameter(msg, ctx=ctx, param=param)
    if 'size' not in obj_dict:
        msg = '"size" should be set for all extra volumes'
        raise click.BadParameter(msg, ctx=ctx, param=param)
    if 'storage' not in obj_dict:
        msg = '"storage" should be set for all extra volumes'
        raise click.BadParameter(msg, ctx=ctx, param=param)
    name = obj_dict['name']
    capacity = obj_dict['size']
    storage_class = obj_dict['storage']
    resource_url = obj_dict.get('image', None)
    if storage_class.lower() not in defines.STORAGE_CLASSES:
        msg = 'Invalid storage value "{}", choose from {}'.format(storage_class, defines.STORAGE_CLASSES)
        raise click.BadParameter(msg, ctx=ctx, param=param)

    volume_pb = vmset_pb2.Volume()
    volume_pb.name = name
    volume_pb.capacity = SizeSuffixStringType().convert(capacity, param, ctx)
    volume_pb.storage_class = storage_class.lower()
    volume_pb.image_type = vmset_pb2.Volume.ImageType.RAW
    if resource_url:
        volume_pb.resource_url = resource_url
    return volume_pb


class ExtraVolumeType(click.ParamType):
    def convert(self, value, param, ctx):
        obj_dict = {}
        for option in value.split(','):
            parsed_option = option.split('=')
            if len(parsed_option) != 2:
                self.fail('Cannot parse volume arg {}'.format(value), ctx=ctx, param=param)

            key_, value_ = parsed_option
            if key_ in obj_dict:
                self.fail('Duplicate key {} in {}'.format(key_, value), ctx=ctx, param=param)
            obj_dict[key_] = value_
        return cast_dict_to_volume(obj_dict, param, ctx)

    def get_metavar(self, param):
        return 'name=TEXT,size=SIZE[,storage=hdd|ssd][,image=TEXT]'


class ExtraVolumeConfType(click.File):
    def convert(self, value, param, ctx):
        ret_value = super(ExtraVolumeConfType, self).convert(value, param, ctx)
        try:
            parsed_value = yaml.safe_load(ret_value)
        except yaml.YAMLError as e:
            self.fail('Cannot parse conf file: {}'.format(e), ctx=ctx, param=param)

        if not parsed_value:
            self.fail('File is empty', ctx=ctx, param=param)
        if 'volumes' not in parsed_value:
            self.fail('Key "volumes" not found', ctx=ctx, param=param)
        volumes_list = parsed_value['volumes']
        if not volumes_list:
            self.fail('Volumes list is empty', ctx=ctx, param=param)
        return [cast_dict_to_volume(obj_dict, param, ctx) for obj_dict in volumes_list]

    def get_metavar(self, param):
        return 'FILENAME'


class ConfirmOptions(CommandOptions):
    assume_yes = OptionValue('-y', is_flag=True, default=False, help="Automatic yes to prompts", order=20000)


class BasePodOptions(CommandOptions):
    pod_id = OptionValue("--pod-id", help="Pod ID", type=PodIdType(), required=True, order=1)

    yp_cluster = OptionValue('--cluster',
                             type=click.Choice(defines.VMPROXY_LOCATION.keys()),
                             envvar='VMCTL_CLUSTER',
                             show_default=True,
                             help='YP cluster location',
                             order=2,
                             required=True,
                             )


class BaseActionOptions(CommandOptions):
    pod_id = OptionValue("--pod-id", help="Pod Id", type=PodIdType(), order=1)
    yp_cluster = OptionValue('--cluster',
                             type=click.Choice(defines.VMPROXY_LOCATION.keys()),
                             envvar='VMCTL_CLUSTER',
                             show_default=True,
                             help='YP cluster location (default from env VMCTL_CLUSTER)',
                             order=2)

    host = OptionValue("-H", "--host", type=str, help="Instance host")
    port = OptionValue("-P", "--port", type=int, help="Instance port")
    service = OptionValue("-S", "--service", type=str, help="Service name", show_default=True, default='')
    timeout = OptionValue("--timeout", type=int, help="Connection timeout", default=None)
    direct = OptionValue("--direct", is_flag=True,
                         help='Direct Connection to vmagent (use --host and --port)',
                         envvar=defines.VMCTL_FORCE_DIRECT_CONNECTION_ENV_NAME)

    def correctness_of_vm_address(self):
        ctx = click.get_current_context()  # type: click.Context
        vmproxy_client = ctx.find_object(api.VMProxyClient)  # type: api.VMProxyClient
        if self.pod_id or self.yp_cluster:
            if self.service or self.host or self.port:
                raise click.UsageError("incorrect address for VMProxy (connection with YP): "
                                       "port or host or service specified")
            if vmproxy_client and (self.pod_id and not self.yp_cluster):
                self.yp_cluster = vmproxy_client.guess_proper_yp_cluster(self.pod_id)

            if not self.yp_cluster or not self.pod_id:
                raise click.UsageError("incorrect address for VMProxy (connection with YP): "
                                       "no pod_id or cluster specified")
        # check for VMAgent
        elif self.direct:
            if self.pod_id or self.yp_cluster or self.service:
                raise click.UsageError("incorrect address for VMAgent: pod_id or cluster "
                                       "or service specified")
        # check for gencfg/nanny
        elif not self.host or not self.port or not self.service:
            raise click.UsageError("incorrect address for VMProxy (connection with gencfg/nanny): "
                                   "no host or port or service specified")


class BaseAllocateOptions(CommandOptions):
    abc = OptionValue('--abc', type=str, help="ABC Service ID (if not pass use tmp quota)", order=100)
    network_id = OptionValue('--network-id',
                             type=NetworkIdType(),
                             help="Network ID (default from env VMCTL_NETWORK_ID)",
                             envvar='VMCTL_NETWORK_ID'
                             )
    node_segment = OptionValue('--node-segment',
                               help="Node Segment (default from env VMCTL_NODE_SEGMENT)",
                               type=click.Choice(defines.NODE_SEGMENTS),
                               envvar='VMCTL_NODE_SEGMENT',
                               order=400,
                               required=True
                               )

    memory = OptionValue('--memory',
                         type=SizeSuffixStringType(),
                         default='8G',
                         show_default=True,
                         help="Pod memory (VM memory = Pod Memory - 1Gb)"
                         )
    logins = OptionValue('--logins',
                         type=RepeatedString(),
                         help="Users Logins (default from env VMCTL_LOGINS variable)",
                         show_default=True,
                         envvar='VMCTL_LOGINS',
                         default=getpass.getuser(),
                         cls=OptionEatAll, order=300)

    groups = OptionValue('--groups', type=RepeatedString(), cls=OptionEatAll, help="Stuff Group IDs", order=300)
    layer_url = OptionValue('--layer-url', type=str, help='for internal usage')

    storage_class = OptionValue('--storage', type=click.Choice(defines.STORAGE_CLASSES),
                                help='Storage class', order=700)
    volume_size = OptionValue('--volume-size',
                              type=SizeSuffixStringType(),
                              default='20G',
                              show_default=True,
                              help="Volume size",
                              order=700)

    enable_internet = OptionValue('--use-white-ipv4', default=False, is_flag=True, help='Enable internet', order=5000)
    use_nat64 = OptionValue('--use-nat64', default=False, is_flag=True, help="Use nat64", order=5000)

    def prepare_network_id(self, value=None, node_segment=None):
        if not value and node_segment == defines.NODE_SEGMENT_DEV:
            value = "_SEARCHSAND_"

        if value is None or not value:
            raise click.BadOptionUsage('argument --network-id: Non empty value required')

        return value

    def prepare_storage_class(self, value=None, node_segment=None):
        _defaults = {defines.NODE_SEGMENT_DEV: defines.STORAGE_CLASS_SSD}
        return value or _defaults.get(node_segment, defines.STORAGE_CLASS_HDD)

    def prepare_memory(self, memory):
        if memory - defines.VM_MEMORY_GAP <= 0:
            raise click.BadOptionUsage('argument --memory: should be greater than 1Gb')

        return memory


class BaseConfigOptions(CommandOptions):
    d_image = OptionValue("--default-image",
                          type=click.Choice(defines.DISTRO_DICT.keys()),
                          help="Choose one of prebuilt ubuntu image",
                          order=600)  # type: str
    rb_torrent = OptionValue("-i", "--image", type=str, help="VM image rbtorrent", order=601)
    type = OptionValue("-t", "--type", type=click.Choice(defines.VM_TYPES), help="VM type", order=601)
    image_type = OptionValue("--image-type", type=click.Choice(defines.IMAGE_TYPES),
                             default=defines.IMAGE_TYPE_DELTA, order=601, help="image type"
                             )

    vcpu = OptionValue("-c", "--vcpu", type=int, help="VM vcpu count", order=1000, required=True)  # type: int

    def prepare_os_choice(self, d_image=None, rb_torrent=None, vm_type=None):
        if d_image and vm_type:
            raise ValueError("you cannot choose type of OS (windows or linux) if you chose image from ready images")

        if rb_torrent and not vm_type:
            raise ValueError("choose type of OS (windows or linux)")

        if d_image:
            vm_type = "linux"
            rb_torrent = defines.DISTRO_DICT[d_image]

        if not rb_torrent:
            raise ValueError("--image: Non empty value required (if --default-image empty)")

        if not vm_type:
            click.echo("No VM type specified, considering Linux")
            vm_type = defines.VM_TYPE_LINUX

        return d_image, rb_torrent, vm_type


class CreateOptions(BasePodOptions, BaseAllocateOptions, BaseConfigOptions, ConfirmOptions):
    yp_cluster = OptionValue('--cluster',
                             type=click.Choice(defines.LOCATION_LIST),
                             envvar='VMCTL_CLUSTER',
                             show_default=True,
                             help='YP cluster location',
                             order=2,
                             required=True,
                             )
    vcpu = OptionValue("-c", "--vcpu", type=int, help="DEPRECATED (use --vcpu-limit instead)", order=1000,
                       required=False)  # type: int
    vcpu_guarantee = OptionValue('--vcpu-guarantee', type=int, default=2, help="VCPU guarantee (1 = 1 core)",
                                 order=1000)
    vcpu_limit = OptionValue('--vcpu-limit', type=int, default=0, help="VCPU limit (1 = 1 core)", order=1000)

    node_id = OptionValue('--node-id', type=str, default='', help='Forced node id', order=1000)
    wait = OptionValue('--wait', is_flag=True, order=3000, help='Wait until vm change status state'
                                                                ' to (CONFIGURED, STOPPED, RUNNING)')
    extra_volume_list = OptionValue('--extra-volume', type=ExtraVolumeType(), multiple=True, order=1000,
                                    help='Disk volume options')
    extra_volumes_conf = OptionValue('--extra-volumes-conf', type=ExtraVolumeConfType(), order=1000,
                                     help='YAML config file for extra volumes configuration')
    gpu_count = OptionValue('--gpu-count', type=int, help='Number of gpu cards', order=1000)
    gpu_model = OptionValue('--gpu-model', type=click.Choice(defines.GPU_MODELS),
                            help='Model of each gpu card', order=1000)
    io_guarantee_ssd = OptionValue('--io-guarantee-ssd', type=SizeSuffixStringType(), order=1000,
                                   help='Cumulative IO guarantee for all ssd volumes')
    io_guarantee_hdd = OptionValue('--io-guarantee-hdd', type=SizeSuffixStringType(), order=1000,
                                   help='Cumulative IO guarantee for all hdd volumes')
    audio = OptionValue('--audio', type=click.Choice(defines.AUDIO_TYPES), help='Audio type')
    network_bandwidth_guarantee = OptionValue('--network-guarantee', type=SizeSuffixStringType(),
                                              default=0,
                                              order=1000, help='Network bandwidth guarantee')
    ip4_address_pool_id = OptionValue('--ip4-address-pool', type=str, help='IP4 pool id for allocating ip4 address',
                                      order=1000)

    def validate(self):
        if self.audio:
            self.audio = vmset_pb2.QemuOptions.AudioType.Value(self.audio)

        self.d_image, self.rb_torrent, self.type = self.prepare_os_choice(self.d_image, self.rb_torrent, self.type)

        self.network_id = self.prepare_network_id(self.network_id, self.node_segment)

        self.storage_class = self.prepare_storage_class(self.storage_class, self.node_segment)

        self.memory = self.prepare_memory(self.memory)

        if self.node_segment in (defines.NODE_SEGMENT_DEFAULT, defines.ARM64_SEGMENT_DEV):
            if self.vcpu_limit > self.vcpu_guarantee:
                msg = "--vcpu-limit: in segment {} --vcpu-limit should be equal --vcpu-guarantee"
                click.echo(msg.format(self.node_segment), err=True)
            self.vcpu_limit = self.vcpu_guarantee

        if self.node_segment == defines.NODE_SEGMENT_DEV:
            self.vcpu_limit = defines.NODE_SEGMENT_DEV_DEFAULT_VCPU_LIMIT if not self.vcpu_limit else self.vcpu_limit

        self.vcpu = self.vcpu_limit

        if not self.abc and not self.assume_yes:
            click.confirm(defines.TMP_ACC_WARN_PROMPT, abort=True)

        if self.node_id and (self.abc == 'personal' or self.abc == defines.PERSONAL_ACCOUNT.split('abc:service:')[1]):
            raise ValueError('Forcing node in personal quota is not allowed')

        if self.extra_volume_list and self.extra_volumes_conf:
            raise ValueError('Use only extra-volumes-conf or multiple extra-volume values, not both')

        if self.extra_volume_list or self.extra_volumes_conf:
            volumes_name_set = set()
            for v in self.extra_volume_list or self.extra_volumes_conf:
                if v.name in volumes_name_set:
                    raise ValueError('Duplicate name {} in extra volumes'.format(v.name))
                else:
                    volumes_name_set.add(v.name)

        if self.gpu_count and not self.gpu_model:
            raise ValueError('Gpu model has not been set')
        if self.gpu_model and not self.gpu_count:
            raise ValueError('Gpu count has not been set')


class AllocateOptions(BasePodOptions, BaseAllocateOptions, ConfirmOptions):
    yp_cluster = OptionValue('--cluster',
                             type=click.Choice(defines.LOCATION_LIST),
                             envvar='VMCTL_CLUSTER',
                             show_default=True,
                             help='YP cluster location',
                             order=2,
                             required=True,
                             )
    vcpu_limit = OptionValue('--vcpu-limit', type=int, default=32000,
                             help='VCPU limit(1000 = 1 core)', order=1000)
    vcpu_guarantee = OptionValue('--vcpu-guarantee', type=int,
                                 help='VCPU guarantee(1000 = 1 core)',
                                 default=2000, order=1000)

    def validate(self):
        self.network_id = self.prepare_network_id(self.network_id, self.node_segment)

        self.storage_class = self.prepare_storage_class(self.storage_class, self.node_segment)

        self.memory = self.prepare_memory(self.memory)

        if not self.abc and not self.assume_yes:
            click.confirm(defines.TMP_ACC_WARN_PROMPT, abort=True)


class ListVmOptions(CommandOptions):
    mode = OptionValue('--mode', type=click.Choice(defines.VM_MODES),
                       help='Filter VMs by mode',
                       show_default=True,
                       default=defines.VM_MODE_YP)
    format = OptionValue('--format', type=click.Choice(('text', 'json')), help="Output format", default='text')
    list_yp_cluster = OptionValue('--cluster',
                                  help="YP clusters",
                                  type=RepeatedChoices(defines.VMPROXY_LOCATION.keys()),
                                  cls=OptionEatAll)
    login = OptionValue('--login',
                        type=str,
                        default=getpass.getuser(),
                        show_default=True,
                        help="Filter by login")
    login_all = OptionValue('--login-all', default=False,
                            help='Do not filter VMs by login',
                            is_flag=True)
    node_segment = OptionValue('--node-segment',
                               help='List of node segments',
                               type=RepeatedChoices(defines.NODE_SEGMENTS),
                               cls=OptionEatAll)

    name_filter = OptionValue('--name-filter', help='Filter pod-id with substr', order=8000)
    fields = OptionValue('--fields', cls=OptionEatAll,
                         type=RepeatedChoices(list_action.AVAILABLE_FIELDS.keys()),
                         help='List of output fields', order=10000)
    abc = OptionValue('--abc', type=RepeatedString(), cls=OptionEatAll, help='List of abc service ids')
    unused_only = OptionValue('--unused-only', default=False, help='Show only unused vms', is_flag=True)

    skip = OptionValue('--skip', type=int, default=0, help='Skip', order=9000)
    limit = OptionValue('--limit', type=int, default=1000, help='Limit', order=9000)


class DeallocateOptions(BasePodOptions, ConfirmOptions):
    def validate(self):
        if not self.assume_yes:
            click.confirm(defines.DESTRUCTIVE_MSG, abort=True)


class UpdateVmagentOptions(BasePodOptions, ConfirmOptions):
    wait = OptionValue('--wait', is_flag=True, order=3000, help='Wait until vm change status state'
                                                                ' to (CONFIGURED, STOPPED, RUNNING)')

    vmagent_url = OptionValue('--vmagent-url', type=click.STRING,
                              help='Only QYP Admin can use this option')
    vmagent_version = OptionValue('--vmagent-version', type=click.STRING,
                                  help='Only QYP Admin can use this option')

    def validate(self):
        if self.vmagent_url and not self.vmagent_version:
            raise ValueError('--vmagent-version: should be passed with --vmagent-url')


class UpdateOptions(BasePodOptions, BaseConfigOptions, ConfirmOptions):
    abc = OptionValue('--abc', type=str, required=False, help='Update ABC service id')
    logins = OptionValue('--logins', help="Update logins. If not pass, has no effect",
                         type=RepeatedString(), cls=OptionEatAll)
    groups = OptionValue('--groups', help="Update groups. If not pass, has no effect",
                         type=RepeatedString(), cls=OptionEatAll)
    clear_groups = OptionValue('--clear-groups', help="Clear all groups from owners",
                               default=False, is_flag=True)
    network_id = OptionValue('--network-id', type=NetworkIdType(), help="Network ID")
    use_nat64 = OptionValue('--use-nat64', default='skip', type=click.Choice(['skip', 'enable', 'disable']),
                            help="Use nat64", order=5000, show_default=True)
    vcpu_guarantee = OptionValue('--vcpu-guarantee', type=int, help="VCPU guarantee (1 = 1 core)",
                                 order=1000)
    vcpu_limit = OptionValue('--vcpu-limit', type=int, help="VCPU limit (1 = 1 core)", order=1000)

    memory = OptionValue('--memory', type=SizeSuffixStringType(), help="Pod memory (VM memory = Pod Memory - 1Gb)")

    autorun = OptionValue("--autorun", default='skip', type=click.Choice(['skip', 'enable', 'disable']),
                          help='Enable Autorun', show_default=True)

    d_image = OptionValue("--default-image",
                          type=click.Choice(defines.DISTRO_DICT.keys()),
                          help="Choose one of prebuilt ubuntu image",
                          order=600)  # type: str
    rb_torrent = OptionValue("-i", "--image", type=str, help="VM image rbtorrent", order=601)
    type = OptionValue("-t", "--type", type=click.Choice(defines.VM_TYPES), help="VM type", order=601)
    image_type = OptionValue("--image-type", type=click.Choice(defines.IMAGE_TYPES), order=601, help="image type")
    vcpu = OptionValue("-c", "--vcpu", type=int, help="VM vcpu count", order=1000)
    wait = OptionValue('--wait', is_flag=True, order=3000, help='Wait until vm change status state'
                                                                ' to (CONFIGURED, STOPPED, RUNNING)')
    node_id = OptionValue('--node-id', type=str, default='', help='Forced node id', order=1000)
    remove_node_id = OptionValue('--remove-node-id', help="Move from forced node. VM will be recreated",
                                 default=False, is_flag=True)
    extra_volume_list = OptionValue('--extra-volume', type=ExtraVolumeType(), multiple=True, order=1000,
                                     help='Disk volume options')
    removed_volumes = OptionValue('--remove-volumes', help='List of volume names to be removed',
                                  type=RepeatedString(), cls=OptionEatAll)
    extra_volumes_conf = OptionValue('--extra-volumes-conf', type=ExtraVolumeConfType(), order=1000,
                                     help='YAML config file for extra volumes configuration')
    gpu_count = OptionValue('--gpu-count', type=int, help='Number of gpu cards', order=1000)
    gpu_model = OptionValue('--gpu-model', type=click.Choice(defines.GPU_MODELS),
                            help='Model of each gpu card', order=1000)
    io_guarantee_ssd = OptionValue('--io-guarantee-ssd', type=SizeSuffixStringType(), order=1000,
                                   help='Cumulative IO guarantee for all ssd volumes')
    io_guarantee_hdd = OptionValue('--io-guarantee-hdd', type=SizeSuffixStringType(), order=1000,
                                   help='Cumulative IO guarantee for all hdd volumes')
    audio = OptionValue('--audio', type=click.Choice(defines.AUDIO_TYPES), help='Audio type')
    network_bandwidth_guarantee = OptionValue('--network-guarantee', type=SizeSuffixStringType(),
                                              order=1000, help='Network bandwidth guarantee')

    def validate(self):
        if self.image_type:
            self.image_type = vmset_pb2.Volume.ImageType.Value(self.image_type)

        if self.audio:
            self.audio = vmset_pb2.QemuOptions.AudioType.Value(self.audio)

        if self.d_image or self.rb_torrent:
            self.d_image, self.rb_torrent, self.type = self.prepare_os_choice(self.d_image, self.rb_torrent, self.type)

        _flags = {'skip': None, 'enable': True, 'disable': False}
        self.use_nat64 = _flags[self.use_nat64]
        self.autorun = _flags[self.autorun]

        poweroff = (self.network_id or self.use_nat64 is not None or self.vcpu_limit or self.vcpu_guarantee
                    or self.memory or self.autorun is not None or self.vcpu or self.extra_volume_list
                    or self.removed_volumes or self.extra_volumes_conf)
        destroy = (self.d_image or self.rb_torrent or self.type is not None or self.image_type is not None
                   or self.node_id or self.remove_node_id)

        if poweroff and not self.assume_yes:
            click.confirm(defines.VM_POWEROFF_PROMPT, abort=True)
        if destroy and not self.assume_yes:
            click.confirm(defines.DESTRUCTIVE_MSG, abort=True)
        if self.extra_volumes_conf and (self.extra_volume_list or self.removed_volumes):
            raise ValueError('Use only extra-volumes-conf or extra-volume/remove-volumes, not both')

        if self.extra_volumes_conf or self.extra_volume_list:
            volumes_name_set = set()
            for v in self.extra_volume_list or self.extra_volumes_conf:
                if v.name in volumes_name_set:
                    raise ValueError('Duplicate name {} in extra volumes'.format(v.name))
                else:
                    volumes_name_set.add(v.name)


class ListAccountsOptions(CommandOptions):
    cluster = OptionValue('--cluster', type=RepeatedChoices(defines.VMPROXY_LOCATION.keys()),
                          cls=OptionEatAll, help='List of clusters')
    node_segment = OptionValue('--segment', type=click.Choice(defines.NODE_SEGMENTS), help='Node segment')
    detailed = OptionValue('--detailed', default=False, is_flag=True, help='Detailed output', order=10001)


class HostnameOptions(BaseActionOptions):
    pass


class ActionOptions(BaseActionOptions):
    def validate(self):
        self.correctness_of_vm_address()


class GetStatusOptions(ActionOptions):
    if helpers.force_direct_connection_to_vmagent():
        find_issues = OptionValue('--find-issues', default=False, is_flag=True,
                                  help='Find Issues (only for direct mode)',
                                  order=10001)


class RevertOptions(BaseActionOptions, ConfirmOptions):
    def validate(self, vmproxy_client=None):
        self.correctness_of_vm_address()
        if not self.assume_yes:
            click.confirm(defines.DESTRUCTIVE_MSG, abort=True)


class ConfigOptions(BaseActionOptions, BaseConfigOptions, ConfirmOptions):
    mem = OptionValue("-m", "--mem", type=SizeSuffixStringType(), required=True,
                      help="Desired memory size in bytes")
    autorun = OptionValue("--autorun", default=False, is_flag=True, help='Enable Autorun')
    disk_size = OptionValue("-s", "--disk", type=SizeSuffixStringType(),
                            help="Desired disk size in bytes", default=0)

    def validate(self, vmproxy_client=None):
        self.correctness_of_vm_address()

        self.d_image, self.rb_torrent, self.type = self.prepare_os_choice(self.d_image, self.rb_torrent, self.type)

        if not self.assume_yes:
            click.confirm(defines.DESTRUCTIVE_MSG, abort=True)


class BackupOptions(BaseActionOptions, ConfirmOptions):
    direct = OptionValue("--direct", is_flag=True, help=argparse.SUPPRESS,
                         default=False)

    backup_storage = OptionValue("--backup-storage",
                                 type=click.Choice(vmset_api_pb2.BackupStorage.keys()),
                                 default=vmset_api_pb2.BackupStorage.Name(vmset_api_pb2.DEFAULT_BACKUP_STORAGE),
                                 help='Backup Storage',
                                 show_default=True
                                 )

    def validate(self, vmproxy_client=None):
        self.direct = False
        self.backup_storage = vmset_api_pb2.BackupStorage.Value(self.backup_storage)
        self.correctness_of_vm_address()
        if not self.assume_yes:
            click.confirm(defines.VM_POWEROFF_PROMPT, abort=True)


class ShareImageOptions(BaseActionOptions, ConfirmOptions):
    def validate(self, vmproxy_client=None):
        self.correctness_of_vm_address()
        if not self.assume_yes:
            click.confirm(defines.VM_POWEROFF_PROMPT, abort=True)


class StopBackupOptions(BaseActionOptions):
    def validate(self, vmproxy_client=None):
        self.correctness_of_vm_address()


class ListBackupOptions(BasePodOptions):
    pass


class RemoveBackupOption(BasePodOptions, ConfirmOptions):
    id = OptionValue('--id', type=str, help='Backup ID', required=True)

    def validate(self):
        if not self.assume_yes:
            click.confirm(defines.OBJECT_REMOVE_PROMPT_TMPL.format('Backup with ID {}'.format(self.id)), abort=True)


class VmStatsOptions(CommandOptions):
    list_yp_cluster = OptionValue('--cluster',
                                  help="YP clusters",
                                  type=RepeatedChoices(defines.VMPROXY_LOCATION.keys()),
                                  cls=OptionEatAll)

    node_segment = OptionValue('--node-segment',
                               help='List of node segments',
                               type=RepeatedChoices(defines.NODE_SEGMENTS),
                               cls=OptionEatAll)
    abc = OptionValue('--abc', type=RepeatedString(), cls=OptionEatAll, help='List of abc service ids')

    consistency = OptionValue('--consistency', type=click.Choice(vmset_api_pb2.ConsistencyLevel.keys()),
                              default=vmset_api_pb2.ConsistencyLevel.Name(vmset_api_pb2.WEAK),
                              help='Backend Consistency Level')

    def validate(self):
        self.consistency = vmset_api_pb2.ConsistencyLevel.Value(self.consistency)

        if not self.list_yp_cluster:
            self.list_yp_cluster = defines.VMPROXY_LOCATION.keys()


class ListFreeNodes(CommandOptions):
    yp_cluster = OptionValue('--cluster',
                             type=click.Choice(defines.VMPROXY_LOCATION.keys()),
                             envvar='VMCTL_CLUSTER',
                             show_default=True,
                             help='YP cluster location',
                             order=2,
                             required=True)
    node_segment = OptionValue('--node-segment',
                               type=click.Choice(defines.NODE_SEGMENTS),
                               envvar='VMCTL_SEGMENT',
                               show_default=True,
                               help='Node segment',
                               order=2,
                               required=True)
    limit = OptionValue('--limit', type=int, default=10, help='Limit', order=9000)
    pretty = OptionValue('--pretty', default=False, is_flag=True, help='Convert bytes in gigabytes', order=10001)
