import math
import os

from infra.qyp.proto_lib import vmset_pb2, vmagent_api_pb2
from infra.qyp.vmctl.src import errors, defines


class PersonalAccountHelper(object):
    DEV_SEGMENT = 'dev'
    TMP_ACCOUNT = 'tmp'

    def __init__(self, login):
        """
        :type login: str
        """
        self.group_usage = UsageAcc()
        self.personal_usage = UsageAcc()
        self.total_usage = UsageAcc()
        self.limits = None
        self.login = login

    def init(self, accounts):
        """
        :type accounts: list[vmset_pb2.Account]
        """
        for account in accounts:
            if account.id == self.TMP_ACCOUNT:
                continue
            if account.type == vmset_pb2.Account.PERSONAL:
                self.limits = account.personal.limits.per_segment[self.DEV_SEGMENT]
                cluster_usage = account.personal.usage.per_segment[self.DEV_SEGMENT]
                factor = 1
                self.personal_usage.plus(cluster_usage, factor)
                self.total_usage.plus(cluster_usage, factor)
            elif account.type == vmset_pb2.Account.SERVICE and account.members_count:
                cluster_usage = account.limits.per_segment[self.DEV_SEGMENT]
                factor = float(1) / account.members_count
                self.group_usage.plus(cluster_usage, factor)
                self.total_usage.plus(cluster_usage, factor)
        self.personal_usage.finalize()
        self.group_usage.finalize()
        self.total_usage.finalize()

    def get_available_usage(self):
        """
        :rtype: UsageAcc
        """
        return self.total_usage.diff(self.limits)

    def validate_new_resource_fit(self, cpu, mem, disk_size, storage_class, enable_internet):
        """
        :type cpu: int
        :type mem: int
        :type disk_size: int
        :type storage_class: str
        :type enable_internet: bool
        """
        if self.limits is None:
            raise errors.VMAgentParamsError('User {} has no access to personal account'.format(self.login))
        req = UsageAcc(cpu=cpu, mem=mem,
                       hdd=disk_size if storage_class == 'hdd' else 0,
                       ssd=disk_size if storage_class == 'ssd' else 0,
                       addrs=int(enable_internet))
        res_available = self.get_available_usage()
        if is_overcommited(req, res_available):
            msg = "Can't allocate resources in quota:\n" \
                  "Available {}\n" \
                  "Requested {}"
            raise errors.VMAgentParamsError(msg.format(res_available.to_str(), req.to_str()))

    def to_account(self):
        """
        :rtype: vmset_pb2.Account
        """
        acc = vmset_pb2.Account()
        acc.limits.per_segment[self.DEV_SEGMENT].CopyFrom(self.limits)
        acc.usage.per_segment[self.DEV_SEGMENT].CopyFrom(self.personal_usage.to_resource_info())
        return acc

    def group_usage_account(self):
        """
        :rtype: vmset_pb2.Account
        """
        acc = vmset_pb2.Account()
        acc.limits.per_segment[self.DEV_SEGMENT].CopyFrom(self.limits)
        acc.usage.per_segment[self.DEV_SEGMENT].CopyFrom(self.group_usage.to_resource_info())
        return acc


class UsageAcc(object):
    def __init__(self, cpu=None, mem=None, hdd=None, ssd=None, addrs=None):
        self.cpu = cpu or 0
        self.mem = mem or 0
        self.disk_per_storage = {'hdd': hdd or 0, 'ssd': ssd or 0}
        self.internet_address = addrs or 0

    def plus(self, res, factor):
        """
        :type res: vmset_pb2.ResourceInfo
        :type factor: float
        """
        self.cpu += res.cpu * factor
        self.mem += res.mem * factor
        self.disk_per_storage['hdd'] += res.disk_per_storage['hdd'] * factor
        self.disk_per_storage['ssd'] += res.disk_per_storage['ssd'] * factor
        self.internet_address += res.internet_address * factor

    def diff(self, res):
        """
        :type res: vmset_pb2.ResourceInfo
        :rtype: UsageAcc
        """
        diff = UsageAcc()
        diff.cpu = res.cpu - self.cpu
        diff.mem = res.mem - self.mem
        diff.disk_per_storage = {
            'hdd': res.disk_per_storage['hdd'] - self.disk_per_storage['hdd'],
            'ssd': res.disk_per_storage['ssd'] - self.disk_per_storage['ssd'],
        }
        diff.internet_address = res.internet_address - self.internet_address
        return diff

    def finalize(self):
        self.cpu = self._round(self.cpu, 1000)
        self.mem = self._round(self.mem, 1024 ** 3)
        self.disk_per_storage['hdd'] = self._round(self.disk_per_storage['hdd'], 1024 ** 3)
        self.disk_per_storage['ssd'] = self._round(self.disk_per_storage['ssd'], 1024 ** 3)
        self.internet_address = self._round(self.internet_address)

    def _round(self, value, up_to=1):
        """
        :rtype: int
        """
        return int(math.ceil(value / up_to)) * up_to

    def to_str(self):
        """
        :rtype: str
        """
        return 'CPU {}, MEM {}, HDD {}, SSD {}, IPv4 {}'.format(
            self.cpu / 1000,
            int_to_str_gigabytes(self.mem),
            int_to_str_gigabytes(self.disk_per_storage['hdd']),
            int_to_str_gigabytes(self.disk_per_storage['ssd']),
            self.internet_address,
        )

    def to_resource_info(self):
        """
        :rtype: vmset_pb2.ResourceInfo
        """
        res = vmset_pb2.ResourceInfo()
        res.cpu = self.cpu
        res.mem = self.mem
        res.disk_per_storage['hdd'] = self.disk_per_storage['hdd']
        res.disk_per_storage['ssd'] = self.disk_per_storage['ssd']
        res.internet_address = self.internet_address
        return res


def is_overcommited(res, av_res):
    """
    :type res: UsageAcc
    :type av_res: UsageAcc
    """
    return (res.cpu > 0 and res.cpu > av_res.cpu or
            res.mem > 0 and res.mem > av_res.mem or
            res.disk_per_storage['hdd'] > 0 and res.disk_per_storage['hdd'] > av_res.disk_per_storage['hdd'] or
            res.disk_per_storage['ssd'] > 0 and res.disk_per_storage['ssd'] > av_res.disk_per_storage['ssd'] or
            res.internet_address > 0 and res.internet_address > av_res.internet_address)


def is_empty(res):
    """
    :type res: vmset_pb2.ResourceInfo
    :rtype: bool
    """
    return (res.cpu == 0 and
            res.mem == 0 and
            res.disk_per_storage['hdd'] == 0 and
            res.disk_per_storage['ssd'] == 0 and
            res.internet_address == 0)


def int_to_str_gigabytes(value):
    """
    :type value: int
    """
    return '{}G'.format(value / 1024 ** 3)


def cast_vmset_to_vmagent_action_req(req):
    """
    :type req: vmset_api_pb2.MakeActionRequest
    :rtype: vmagent_api_pb2.VMActionRequest
    """
    vmagent_req = vmagent_api_pb2.VMActionRequest()
    vmagent_req.action = req.action
    vmagent_req.config.CopyFrom(req.config)
    vmagent_req.timeout = req.timeout
    return vmagent_req


def force_direct_connection_to_vmagent():
    return os.environ.get(defines.VMCTL_FORCE_DIRECT_CONNECTION_ENV_NAME, None)
