import click
import time
import os
import sys
import uuid
import requests
import urllib
import webbrowser
from collections import defaultdict
from functools import update_wrapper, partial
from multiprocessing.pool import ThreadPool

import nanny_rpc_client
from google.protobuf import json_format
from library.python.svn_version import svn_revision

import library.python.oauth as lpo
from infra.qyp.proto_lib import vmagent_pb2, vmset_pb2

from infra.qyp.vmctl.src import defines
from infra.qyp.vmctl.src import api
from infra.qyp.vmctl.src import helpers
from infra.qyp.vmctl.src import ctl_options, errors
from infra.qyp.vmctl.src.actions import list_action, list_accounts, list_backup, list_free_nodes


class VmctlClickGroup(click.Group):
    def __init__(self, *args, **kwargs):
        self.help_priorities = []
        self.help_disabled_commands = []
        super(VmctlClickGroup, self).__init__(*args, **kwargs)

    def get_help(self, ctx):
        self.list_commands = self.list_commands_for_help
        help_str = super(VmctlClickGroup, self).get_help(ctx)
        if helpers.force_direct_connection_to_vmagent():
            return help_str.replace('%HELP_INFO%', 'VMCTL DIRECT MODE, use for '
                                                   'control vmagent (no args required)')
        else:
            return help_str.replace('%HELP_INFO%', 'https://wiki.yandex-team.ru/qyp/cli/')

    def list_commands_for_help(self, ctx):
        """reorder the list of commands when listing the help"""
        commands = super(VmctlClickGroup, self).list_commands(ctx)
        disabled_commands = [c.name for c in self.help_disabled_commands]
        priority = [c.name for c in self.help_priorities]
        return (c[1] for c in sorted(
            (priority.index(command), command)
            for command in commands if command in priority and command not in disabled_commands))


@click.group(cls=VmctlClickGroup)
@click.option('-T', '--token', 'token',
              help="OAuth token (default from env VMCTL_TOKEN)",
              envvar='VMCTL_TOKEN')
@click.option('-L', '--server', 'server', help="VMProxy hostname")
@click.option('-k', '--no-check-certificate', 'no_check_certificate', is_flag=True,
              help="Do not verify proxy host SSL certificate")
@click.option('-V', '--verbose', 'verbose', is_flag=True, help="Verbose output", expose_value=True)
@click.version_option(version=svn_revision(), message='vmctl svn revision: r%(version)s')
@click.pass_context
def vmctl(*args, **kwargs):
    """
    %HELP_INFO%
    """


def get_oauth_token():
    ctx = click.get_current_context()
    return ctx.parent.params.get('token') or lpo.get_token(defines.CLIENT_ID, defines.CLIENT_SECRET)


def pass_vmproxy_client(f):
    def new_func(*args, **kwargs):
        ctx = click.get_current_context()  # type: click.Context
        obj = ctx.find_object(api.VMProxyClient)
        if not obj:
            ctx.obj = api.VMProxyClient(
                proxyhost=ctx.parent.params.get('server'),
                token=api.VMProxyClient.find_token(ctx.parent.params.get('token')),
                ssl_none=ctx.parent.params.get('no_check_certificate'),
                user_agent="{name}/r{revision} ({platform})".format(
                    name='ya-tools-qyp' if 'ya' in os.path.basename(sys.executable) else 'vmctl',
                    revision=svn_revision(),
                    platform=sys.platform,
                )
            )
            if not ctx.obj.token and not ctx.params.get('direct', False):
                raise click.BadOptionUsage(
                    'argument -T/--token: Non empty value required (default from env VMCTL_TOKEN)')
        verbose = ctx.parent.params.get('verbose', False)
        if verbose:
            return ctx.invoke(f, ctx.obj, *args[1:], **kwargs)
        else:
            try:
                return ctx.invoke(f, ctx.obj, *args[1:], **kwargs)
            except (errors.VMAgentParamsError, ValueError) as e:
                raise click.UsageError(e.message or e, ctx)
            except requests.HTTPError as e:
                raise click.ClickException(e.response.content or str(e))
            except requests.ConnectionError as e:
                is_direct_connection = ctx.params.get('direct', False)
                if is_direct_connection:
                    msg = "Cannot connect to vmagent {}'.format(e)".format(e)
                else:
                    msg = 'Cannot connect to vmproxy server: {}'.format(e)
                raise click.ClickException(msg)
            except nanny_rpc_client.exceptions.RpcError as e:
                raise click.ClickException(str(e))
            except nanny_rpc_client.exceptions.BalancerRetriesExceeded:
                raise click.ClickException('Request failed. Try again later')

    return update_wrapper(new_func, f)


def wait_vm_change_status(vmproxy_client, cluster, pod_id, wait_iteration_timeout):
    done, reason_or_status = False, 'Undefined'
    ctx = click.get_current_context()
    verbose = ctx.parent.params.get('verbose', False)
    if verbose:
        click.echo('Wait until vm change status state to (CONFIGURED, STOPPED, RUNNING)...')
    try:
        for done, reason_or_status in vmproxy_client.wait_status_change(cluster, pod_id, wait_iteration_timeout):
            if verbose:
                click.echo('{}, current status: {}'.format('In Progress' if not done else 'Success', reason_or_status))

    except KeyboardInterrupt:
        if verbose:
            click.echo('Interrupt by Keyboard with result: {} -> {}'.format(done, reason_or_status))


def confirm_volumes_changing(args, vmproxy_client):
    old_vm = vmproxy_client.get_vm(args.yp_cluster, args.pod_id)
    changed_volumes = []
    removed_volumes = []
    msg = ''
    if args.extra_volumes_conf:
        new_volumes_dict = {v.name: v for v in args.extra_volumes_conf}
        for v in old_vm.spec.qemu.volumes[1:]:
            new_volume_spec = new_volumes_dict.get(v.name, None)
            if new_volume_spec is None:
                removed_volumes.append(v.name)
            # one can only change resource url
            elif new_volume_spec.resource_url != v.resource_url:
                changed_volumes.append(v.name)
        if changed_volumes:
            msg += defines.VOLUME_CHANGED_MSG.format(volumes_list=changed_volumes) + '\n'
        if removed_volumes:
            msg += defines.VOLUME_REMOVED_MSG.format(volumes_list=removed_volumes) + '\n'
        if msg:
            click.confirm(msg[:-1], abort=True)
    else:
        if args.extra_volume_list:
            new_volumes_dict = {v.name: v for v in args.extra_volume_list}
            for v in old_vm.spec.qemu.volumes[1:]:
                new_volume_spec = new_volumes_dict.get(v.name, None)
                # one can only change resource url
                if new_volume_spec is not None and new_volume_spec.resource_url != v.resource_url:
                    changed_volumes.append(v.name)
            if changed_volumes:
                msg += defines.VOLUME_CHANGED_MSG.format(volumes_list=changed_volumes) + '\n'
        if args.removed_volumes:
            selected_volumes_set = set()
            for value in args.removed_volumes:
                selected_volumes_set.update(value.split())
            for v in old_vm.spec.qemu.volumes[1:]:
                if v.name in selected_volumes_set:
                    removed_volumes.append(v.name)
            if removed_volumes:
                msg += defines.VOLUME_REMOVED_MSG.format(volumes_list=removed_volumes) + '\n'
        if msg:
            click.confirm(msg[:-1], abort=True)


@vmctl.command(help='Create VM')
@ctl_options.CreateOptions.decorate
@pass_vmproxy_client
def create(vmproxy_client, **kwargs):  # type: (api.VMProxyClient, dict) -> None
    args = ctl_options.CreateOptions.build_from_kwargs(kwargs)
    vmproxy_client.create(
        config_id=str(uuid.uuid4()),
        vcpu=args.vcpu,
        mem_config=args.memory - 1024 ** 3,
        rb_torrent=args.rb_torrent,
        disk_size=0,
        vm_type=args.type,
        image_type=args.image_type,
        autorun=True,
        cluster=args.yp_cluster,
        pod_id=args.pod_id,
        network_id=args.network_id,
        node_segment=args.node_segment,
        volume_size=args.volume_size,
        mem=args.memory,
        vcpu_limit=args.vcpu_limit * 1000,
        vcpu_guarantee=args.vcpu_guarantee * 1000,
        logins=args.logins,
        groups=args.groups,
        layer_url=args.layer_url,
        storage_class=args.storage_class,
        enable_internet=args.enable_internet,
        use_nat64=args.use_nat64,
        abc=args.abc,
        node_id=args.node_id,
        extra_volumes=args.extra_volume_list or args.extra_volumes_conf,
        gpu_count=args.gpu_count,
        gpu_model=args.gpu_model,
        io_guarantee_ssd=args.io_guarantee_ssd,
        io_guarantee_hdd=args.io_guarantee_hdd,
        audio=args.audio,
        network_bandwidth_guarantee=args.network_bandwidth_guarantee,
        ip4_address_pool_id=args.ip4_address_pool_id,
    )
    if args.wait:
        wait_vm_change_status(vmproxy_client, args.yp_cluster, args.pod_id, defines.CREATE_WAIT_ITERATION_TIMEOUT)


@vmctl.command('list', help='Show list VMs')
@ctl_options.ListVmOptions.decorate
@pass_vmproxy_client
def list_vms(vmproxy_client, **kwargs):  # type: (api.VMProxyClient, dict) -> None
    args = ctl_options.ListVmOptions.build_from_kwargs(kwargs)
    if args.mode == 'gencfg':
        raise click.BadOptionUsage('--mode: gencfg not supported yet')

    if args.format == 'text':
        click.echo(list_action.list_yp_vms(args, vmproxy_client, skip=args.skip, limit=args.limit))
    else:
        click.echo(list_action.dump_json_yp_vms(args, vmproxy_client, skip=args.skip, limit=args.limit))


@vmctl.command(help='Allocate pod for VM in YP')
@ctl_options.AllocateOptions.decorate
@pass_vmproxy_client
def allocate(vmproxy_client, **kwargs):  # type: (api.VMProxyClient, dict) -> None
    args = ctl_options.AllocateOptions.build_from_kwargs(kwargs)
    storage_class = args.storage_class or {'dev': 'ssd'}.get(args.node_segment, 'hdd')
    vmproxy_client.allocate(
        cluster=args.yp_cluster,
        pod_id=args.pod_id,
        network_id=args.network_id,
        node_segment=args.node_segment,
        volume_size=args.volume_size,
        mem=args.memory,
        vcpu_limit=args.vcpu_limit,
        vcpu_guarantee=args.vcpu_guarantee,
        logins=args.logins,
        groups=args.groups,
        layer_url=args.layer_url,
        storage_class=storage_class,
        enable_internet=args.enable_internet,
        use_nat64=args.use_nat64,
        abc=args.abc,
    )


@vmctl.command(help='Deallocate pod for VM in YP')
@ctl_options.DeallocateOptions.decorate
@pass_vmproxy_client
def deallocate(vmproxy_client, **kwargs):  # type: (api.VMProxyClient, dict) -> None
    args = ctl_options.DeallocateOptions.build_from_kwargs(kwargs)

    vmproxy_client.deallocate(
        cluster=args.yp_cluster,
        pod_id=args.pod_id
    )


@vmctl.command('update-vmagent', help='Update VMAgent to latest version')
@ctl_options.UpdateVmagentOptions.decorate
@pass_vmproxy_client
def update_vmagent(vmproxy_client, **kwargs):  # type: (api.VMProxyClient, dict) -> None
    args = ctl_options.UpdateVmagentOptions.build_from_kwargs(kwargs)

    version_valid = vmproxy_client.check_qyp_vmagent_version(args.yp_cluster, args.pod_id,
                                                             defines.VMAGENT_UPDATE_WITHOUT_RESTART_VERSION)

    if not version_valid and not args.assume_yes:
        click.confirm(defines.VM_POWEROFF_PROMPT, abort=True)

    custom_pod_resources = None
    update_vmagent_version = None
    if args.vmagent_url:
        custom_pod_resources = {'vmagent': vmset_pb2.PodResource(dynamic=True, url=args.vmagent_url)}
        update_vmagent_version = args.vmagent_version

    resp = vmproxy_client.update_vmagent(cluster=args.yp_cluster, pod_id=args.pod_id,
                                         custom_pod_resources=custom_pod_resources,
                                         update_vmagent_version=update_vmagent_version)
    click.echo('Current VMAgent Version : {}'.format(resp.vm.spec.vmagent_version))
    if args.wait:
        wait_vm_change_status(vmproxy_client, args.yp_cluster, args.pod_id, defines.CREATE_WAIT_ITERATION_TIMEOUT)


@vmctl.command(help='Update VM')
@ctl_options.UpdateOptions.decorate
@pass_vmproxy_client
def update(vmproxy_client, **kwargs):  # type: (api.VMProxyClient, dict) -> None
    args = ctl_options.UpdateOptions.build_from_kwargs(kwargs)

    update_vmagent_flag = False
    if args.use_nat64 or args.memory or args.autorun or args.vcpu:
        version_valid = vmproxy_client.check_qyp_vmagent_version(args.yp_cluster, args.pod_id,
                                                                 defines.VMAGENT_SAFE_UPDATE_VERSION)
        # Update vmagent to avoid vm destruction
        if not version_valid:
            raise click.UsageError('Vmagent version is too old, please update to the latest version first')

    if args.extra_volume_list or args.removed_volumes or args.extra_volumes_conf:
        confirm_volumes_changing(args, vmproxy_client)

    update_response = vmproxy_client.update(
        cluster=args.yp_cluster,
        pod_id=args.pod_id,
        abc=args.abc,
        logins=args.logins,
        groups=args.groups,
        clear_groups=args.clear_groups,
        network_id=args.network_id,
        use_nat64=args.use_nat64,
        vcpu_limit=args.vcpu_limit * 1000 if args.vcpu_limit else None,
        vcpu_guarantee=args.vcpu_guarantee * 1000 if args.vcpu_guarantee else None,
        mem=args.memory,
        autorun=args.autorun,
        vcpu=args.vcpu,
        mem_config=args.memory - 1024 ** 3 if args.memory else None,
        rb_torrent=args.rb_torrent,
        vm_type=args.type,
        image_type=args.image_type,
        update_vmagent=update_vmagent_flag,
        node_id=args.node_id,
        remove_node_id=args.remove_node_id,
        changed_volumes=args.extra_volume_list,
        removed_volumes=args.removed_volumes,
        conf_volumes=args.extra_volumes_conf,
        gpu_count=args.gpu_count,
        gpu_model=args.gpu_model,
        io_guarantee_ssd=args.io_guarantee_ssd,
        io_guarantee_hdd=args.io_guarantee_hdd,
        audio=args.audio,
        network_bandwidth_guarantee=args.network_bandwidth_guarantee,
    )
    if args.logins or args.groups or args.clear_groups:
        click.echo("Owners success updated: ")
        click.echo(update_response.vm.meta.auth.owners)

    if args.wait:
        wait_vm_change_status(vmproxy_client, args.yp_cluster, args.pod_id, defines.UPDATE_WAIT_ITERATION_TIMEOUT)


@vmctl.command('list-accounts', help='Show list YP accounts')
@ctl_options.ListAccountsOptions.decorate
@pass_vmproxy_client
def list_accounts_cmd(vmproxy_client, **kwargs):  # type: (api.VMProxyClient, dict) -> None
    args = ctl_options.ListAccountsOptions.build_from_kwargs(kwargs)
    group_quotas, personal_quotas = list_accounts.run(args, api_client=vmproxy_client)
    click.echo('Group Quotas')
    click.echo(group_quotas)
    if personal_quotas:
        click.echo('Personal Quotas')
        click.echo(personal_quotas)


@vmctl.command(help='Show VM hostname')
@ctl_options.HostnameOptions.decorate
def hostname(**kwargs):
    args = ctl_options.HostnameOptions.build_from_kwargs(kwargs)
    if args.pod_id and args.yp_cluster:
        click.echo('{}.{}.yp-c.yandex.net'.format(args.pod_id, args.yp_cluster.lower()))
    elif args.host:
        click.echo(args.host)


@vmctl.command(help='Retrieve VM instance status')
@ctl_options.GetStatusOptions.decorate
@pass_vmproxy_client
def status(vmproxy_client, **kwargs):  # type: (api.VMProxyClient, dict) -> None
    args = ctl_options.GetStatusOptions.build_from_kwargs(kwargs)
    find_issues = False
    if hasattr(args, 'find_issues') and args.find_issues:
        find_issues = True
    vmagent_client = api.VMAgentClient.from_action_options(args, vmproxy_client)
    m = vmagent_client.get_status(find_issues=find_issues)
    click.echo(json_format.MessageToJson(m, including_default_value_fields=True))


@vmctl.command(help='Monitor instance status')
@ctl_options.ActionOptions.decorate
@pass_vmproxy_client
def monitor(vmproxy_client, **kwargs):  # type: (api.VMProxyClient, dict) -> None
    args = ctl_options.ActionOptions.build_from_kwargs(kwargs)
    vmagent_client = api.VMAgentClient.from_action_options(args, vmproxy_client)
    try:
        while True:
            click.echo(vmagent_client.get_status())
            time.sleep(1)
    except KeyboardInterrupt:
        click.echo('Interrupted by user...')


@vmctl.command(help='Open link to instance vnc in external browser')
@ctl_options.ActionOptions.decorate
@pass_vmproxy_client
def vnc(vmproxy_client, **kwargs):  # type: (api.VMProxyClient, dict) -> None
    args = ctl_options.ActionOptions.build_from_kwargs(kwargs)
    vmagent_client = api.VMAgentClient.from_action_options(args, vmproxy_client)
    rsp = vmagent_client.get_status()

    if rsp.state.type != vmagent_pb2.VMState.RUNNING:
        raise click.UsageError("VM is not running, VNC is not accessible now")

    if args.pod_id and args.yp_cluster:
        query = {
            'pod_id': args.pod_id,
            'cluster': args.yp_cluster
        }
    else:
        query = {
            'host': os.environ.get('PORTO_HOST') or args.host,
            'port': args.port,
            'service': args.service
        }

    link = defines.VNC_LINK.format(path=urllib.quote(urllib.urlencode(query)))
    pwd = rsp.config.access_info.vnc_password
    if pwd:
        link += "&password={}".format(pwd)

    click.echo("VNC url: {}".format(link))
    click.echo("VNC password: {}".format(rsp.config.access_info.vnc_password))

    webbrowser.open_new(link)


@vmctl.command(help='Start VM instance')
@ctl_options.ActionOptions.decorate
@pass_vmproxy_client
def start(vmproxy_client, **kwargs):  # type: (api.VMProxyClient, dict) -> None
    args = ctl_options.ActionOptions.build_from_kwargs(kwargs)
    vmagent_client = api.VMAgentClient.from_action_options(args, vmproxy_client)
    if args.yp_cluster and args.pod_id:
        backups = vmproxy_client.list_backups(args.yp_cluster, args.pod_id)
        for _backup in backups:
            if _backup.status.state in (vmset_pb2.BackupStatus.PLANNED, vmset_pb2.BackupStatus.IN_PROGRESS):
                click.confirm(defines.VM_BACKUP_FAIL_PROMPT, abort=True)
    vmagent_client.start()


@vmctl.command(help='Try to gracefully shutdown VM in instance')
@ctl_options.ActionOptions.decorate
@pass_vmproxy_client
def shutdown(vmproxy_client, **kwargs):  # type: (api.VMProxyClient, dict) -> None
    args = ctl_options.ActionOptions.build_from_kwargs(kwargs)
    vmagent_client = api.VMAgentClient.from_action_options(args, vmproxy_client)
    vmagent_client.shutdown()


@vmctl.command('poweroff', help='Stop VM in instance')
@ctl_options.ActionOptions.decorate
@pass_vmproxy_client
def power_off(vmproxy_client, **kwargs):  # type: (api.VMProxyClient, dict) -> None
    args = ctl_options.ActionOptions.build_from_kwargs(kwargs)
    vmagent_client = api.VMAgentClient.from_action_options(args, vmproxy_client)
    vmagent_client.poweroff()


@vmctl.command(help='Launch rescue shell with instance disk')
@ctl_options.ActionOptions.decorate
@pass_vmproxy_client
def rescue(vmproxy_client, **kwargs):  # type: (api.VMProxyClient, dict) -> None
    args = ctl_options.ActionOptions.build_from_kwargs(kwargs)
    vmagent_client = api.VMAgentClient.from_action_options(args, vmproxy_client)
    vmagent_client.rescue()


@vmctl.command(help='Reset VM in instance')
@ctl_options.ActionOptions.decorate
@pass_vmproxy_client
def reset(vmproxy_client, **kwargs):  # type: (api.VMProxyClient, dict) -> None
    args = ctl_options.ActionOptions.build_from_kwargs(kwargs)
    vmagent_client = api.VMAgentClient.from_action_options(args, vmproxy_client)
    vmagent_client.reset()


@vmctl.command(help='Revert VM to its initial state')
@ctl_options.RevertOptions.decorate
@pass_vmproxy_client
def revert(vmproxy_client, **kwargs):  # type: (api.VMProxyClient, dict) -> None
    args = ctl_options.RevertOptions.build_from_kwargs(kwargs)
    vmagent_client = api.VMAgentClient.from_action_options(args, vmproxy_client)
    vmagent_client.revert()


@vmctl.command('share-image', help='Share image of VM disk')
@ctl_options.ShareImageOptions.decorate
@pass_vmproxy_client
def share_image(vmproxy_client, **kwargs):  # type: (api.VMProxyClient, dict) -> None
    args = ctl_options.ShareImageOptions.build_from_kwargs(kwargs)
    vmagent_client = api.VMAgentClient.from_action_options(args, vmproxy_client)
    vmagent_client.share_image()


@vmctl.command(help='Set or replace instance VM configuration')
@ctl_options.ConfigOptions.decorate
@pass_vmproxy_client
def config(vmproxy_client, **kwargs):  # type: (api.VMProxyClient, dict) -> None
    args = ctl_options.ConfigOptions.build_from_kwargs(kwargs)
    vmagent_client = api.VMAgentClient.from_action_options(args, vmproxy_client)

    vmagent_client.push_config(id=str(uuid.uuid4()),
                               vcpu=args.vcpu,
                               mem=args.mem,
                               rb_torrent=args.rb_torrent,
                               disk_size=args.disk_size,
                               vm_type=args.type,
                               image_type=args.image_type,
                               autorun=args.autorun)


@vmctl.command(help='Backup VM disk')
@ctl_options.BackupOptions.decorate
@pass_vmproxy_client
def backup(vmproxy_client, **kwargs):  # type: (api.VMProxyClient, dict) -> None
    args = ctl_options.BackupOptions.build_from_kwargs(kwargs)

    if args.yp_cluster and args.pod_id:
        version_valid = vmproxy_client.check_qyp_vmagent_version(args.yp_cluster, args.pod_id,
                                                                 defines.VMAGENT_QDM_BACKUP_VERSION)
        if not version_valid:
            if not args.assume_yes:
                click.confirm(defines.VMAGENT_UPDATE_PROMPT, abort=True)
            vmproxy_client.update_vmagent(
                cluster=args.yp_cluster,
                pod_id=args.pod_id,
            )
            wait_vm_change_status(vmproxy_client, args.yp_cluster, args.pod_id, defines.WAIT_ITERATION_TIMEOUT)

    vmproxy_client.backup(
        cluster=args.yp_cluster,
        pod_id=args.pod_id,
        host=args.host,
        port=args.port,
        service=args.service,
        backup_storage=args.backup_storage,
    )


@vmctl.command('stop-backup', help='Stop VM backup')
@ctl_options.StopBackupOptions.decorate
@pass_vmproxy_client
def stop_backup(vmproxy_client, **kwargs):  # type: (api.VMProxyClient, dict) -> None
    args = ctl_options.StopBackupOptions.build_from_kwargs(kwargs)
    vmproxy_client.stop_backup(args.yp_cluster, args.pod_id)


@vmctl.command('list-backup', help='Show list VM backups')
@ctl_options.ListBackupOptions.decorate
@pass_vmproxy_client
def list_backup_cmd(vmproxy_client, **kwargs):  # type: (api.VMProxyClient, dict) -> None
    args = ctl_options.ListBackupOptions.build_from_kwargs(kwargs)
    click.echo(list_backup.run(args, vmproxy_client))


@vmctl.command('remove-backup', help='Remove VM backup')
@ctl_options.RemoveBackupOption.decorate
@pass_vmproxy_client
def remove_backup(vmproxy_client, **kwargs):  # type: (api.VMProxyClient, dict) -> None
    args = ctl_options.RemoveBackupOption.build_from_kwargs(kwargs)

    vmproxy_client.remove_backup(
        cluster=args.yp_cluster,
        pod_id=args.pod_id,
        backup_id=args.id,
    )


@vmctl.command('ui', help='Open UI in default webbrowser')
def ui():
    open_new = webbrowser.open_new
    open_new('https://qyp.yandex-team.ru')


@vmctl.command('vm-stats', help='Show Vm Usage Stats')
@ctl_options.VmStatsOptions.decorate
@pass_vmproxy_client
def vm_stats(vmproxy_client, **kwargs):  # type: (api.VMProxyClient, dict) -> None
    args = ctl_options.VmStatsOptions.build_from_kwargs(kwargs)

    cluster_list = ['---'] if vmproxy_client.custom_proxyhost else args.list_yp_cluster

    pool = ThreadPool(3)
    _fetch_from_cluster = partial(vmproxy_client.get_vm_stats,
                                  node_segment=args.node_segment,
                                  abc=args.abc,
                                  consistency=args.consistency
                                  )

    res = pool.map(_fetch_from_cluster, cluster_list)
    total, cpu, mem, storage = 0, 0, 0, defaultdict(lambda: 0)
    vmagent_version_stats = defaultdict(lambda: 0)

    for item in res:
        total += item.total
        cpu += item.usage.cpu
        mem += item.usage.mem
        for vmagent_version, count in item.vmagent_versions.items():
            vmagent_version_stats[vmagent_version] += count
        for storage_type, capacity in item.usage.disk_per_storage.items():
            storage[storage_type] += capacity

    click.echo('Total: {}'.format(total))
    click.echo('CPU: {} c'.format(cpu / 1000))
    click.echo('RAM: {} Gb'.format(mem / 1024 ** 3))
    for storage_type, capacity in storage.items():
        click.echo('Disc {}: {} Gb'.format(storage_type, capacity / 1024 ** 3))

    click.echo('VM Agent versions: ')
    for vmagent_version, count in sorted(vmagent_version_stats.items(), key=lambda o: o[1], reverse=True):
        click.echo('{} : {}'.format(vmagent_version, count))


@vmctl.command('upgrade-vmctl', help='Upgrade vmctl binary to most recent version')
@click.option('--beta', is_flag=True, default=False)
@click.option('--sandbox-token', 'sandbox_token', help="OAuth token for sandbox")
def upgrade_vmctl(beta, sandbox_token):
    if sys.platform == "linux" or sys.platform == "linux2":
        url = defines.VMCTL_LINUX_RELEASE_URL if not beta else defines.VMCTL_LINUX_URL
    elif sys.platform == "darwin":
        url = defines.VMCTL_MACOS_RELEASE_URL if not beta else defines.VMCTL_MACOS_URL
    else:
        # FIXME: implement self-upgrade for win32
        click.echo('upgrade-vmctl is not yet supported for %s' % sys.platform)
        sys.exit(1)
    me = os.path.abspath(sys.argv[0])
    # If we are executed from Arcadia PY_PROGRAM binary container
    # sys.argv[0] path is incorrect, we should sys.executable instead
    me_exe = os.path.abspath(sys.executable)
    if os.path.basename(me) == os.path.basename(me_exe):
        me = me_exe
    me_tmp = me + '.tmp'
    click.echo("Downloading " + url + " -> " + me)
    oauth_token = sandbox_token or get_oauth_token()
    headers = {'Authorization': 'OAuth {}'.format(oauth_token)}
    r = requests.get(url, headers=headers, stream=True)
    r.raise_for_status()
    with open(me_tmp, 'wb') as f:
        for chunk in r.iter_content(chunk_size=65536):
            if chunk:  # filter out keep-alive new chunks
                f.write(chunk)

    os.chmod(me_tmp, 0755)
    os.rename(me_tmp, me)
    click.echo("Upgrade successful")


@vmctl.command('list-free-nodes', help='Show list free nodes')
@ctl_options.ListFreeNodes.decorate
@pass_vmproxy_client
def list_free_nodes_cmd(vmproxy_client, **kwargs):
    args = ctl_options.ListFreeNodes.build_from_kwargs(kwargs)
    free_nodes = list_free_nodes.run(args, vmproxy_client)
    click.echo('List free nodes')
    click.echo(free_nodes)


vmctl.help_priorities = [
    ui,
    list_vms,
    vm_stats,
    list_accounts_cmd,
    create,
    deallocate,
    start,
    shutdown,
    power_off,
    reset,
    backup,
    share_image,
    list_backup_cmd,
    remove_backup,
    allocate,
    config,
    hostname,
    monitor,
    rescue,
    revert,
    status,
    update,
    update_vmagent,
    vnc,
    upgrade_vmctl,
    list_free_nodes_cmd
]

vmctl.help_disabled_commands = []

# Disable self upgrade for "ya tools vmctl"
if '.ya/tools' in sys.executable:
    vmctl.help_disabled_commands.append(upgrade_vmctl)

# Disable help commands for vmctl in porto layer
if helpers.force_direct_connection_to_vmagent():
    vmctl.help_disabled_commands += [
        ui,
        list_vms,
        vm_stats,
        list_accounts_cmd,
        create,
        deallocate,
        backup,
        list_backup_cmd,
        remove_backup,
        allocate,
        hostname,
        update,
        update_vmagent,
        vnc,
        list_free_nodes_cmd
    ]

if __name__ == "__main__":
    vmctl()
