import inject
import yp.data_model as data_model
from sepelib.core import config
# import semantic_version
from yt import yson

from infra.swatlib.auth import abc
# from infra.qyp.vmproxy.src import errors
from infra.qyp.vmproxy.src.action import helpers
# from infra.qyp.vmproxy.src.action import validation
# from infra.qyp.vmproxy.src.action import config as config_action
from infra.qyp.vmproxy.src.lib.yp import yputil
from infra.qyp.proto_lib import vmset_pb2


VM_CONFIG_RESOURCE_NAME = 'vm.config'
DUTY_SUBJECT = 'abc:service-scope:2900:16'
ROBOT_LOGIN = 'robot-vmagent-rtc'
ROOT_SUBJECTS = frozenset([
    'abc:service-scope:3389:8',
    'abc:service-scope:2900:8',
    'abc:service:2285',
    ROBOT_LOGIN,
])


def get_scopes_dict():
    abc_client = inject.instance(abc.IAbcClient)
    abc_resp = abc_client._client.get('/api/v4/roles/scopes/')
    return {item['slug']: item['id'] for item in abc_resp['results']}


def make_pod_acl(meta):
    """
    :type meta: vmset_pb2.VMMeta
    :rtype: list[data_model.TAccessControlEntry]
    """
    root_ace = data_model.TAccessControlEntry()
    root_ace.action = data_model.ACA_ALLOW
    root_ace.permissions.extend([
        data_model.ACA_WRITE,
        data_model.ACP_READ,
    ])
    root_ace.subjects.extend(ROOT_SUBJECTS)

    owners_ace = data_model.TAccessControlEntry()
    owners_ace.action = data_model.ACA_ALLOW
    owners_ace.permissions.extend([
        data_model.ACA_GET_QYP_VM_STATUS,
        data_model.ACA_SSH_ACCESS,
        data_model.ACP_READ,
        ])
    owners_ace.subjects.extend(meta.auth.owners.logins)
    if len(meta.auth.owners.group_ids):
        scopes = get_scopes_dict()
    else:
        scopes = None

    for group_id in meta.auth.owners.group_ids:
        subject = helpers.cast_group_id_to_subject(group_id, scopes)
        owners_ace.subjects.append(subject)

    return [root_ace, owners_ace]


def prepare_pod(meta, spec, login, pod_ctl):
    """
    :type meta: vmset_pb2.VMMeta
    :type spec: vmset_pb2.VMSpec
    :type login: str
    :type pod_ctl: PodController
    :rtype: data_model.TPod
    """
    object_id = meta.id
    if config.get_value('vmproxy.pass_ssh_keys', False):
        request = pod_ctl.get_keys_by_logins(meta.auth.owners.logins)
        keys = []
        for user in request:
            for key in user:
                keys.append(key['key'])
    else:
        keys = None

    pod_spec = yputil.cast_vmspec_to_pod_spec(
        pod_id=object_id,
        yp_cluster=config.get_value('yp.default_cluster'),
        spec=spec,
        login=login,
    )

    pod = data_model.TPod()
    pod.meta.id = object_id
    pod.meta.pod_set_id = object_id
    pod.meta.inherit_acl = True
    pod.spec.CopyFrom(pod_spec)
    yputil.cast_dict_to_attr_dict(yputil.make_qyp_labels_dict(spec), pod.labels)

    owners_dict = helpers.cast_owners_to_dict(meta.auth.owners)
    owners_dict['author'] = login

    pod.annotations.attributes.add(key='owners', value=yson.dumps(owners_dict))

    vm = vmset_pb2.VM()
    vm.meta.CopyFrom(meta)
    vm.spec.CopyFrom(spec)

    pod.annotations.attributes.add(key='qyp_vm_spec', value=helpers.yson_dumps_vm(vm))
    pod.spec.dynamic_attributes.annotations.append('qyp_vm_spec')

    if keys:
        pod.annotations.attributes.add(key='qyp_ssh_authorized_keys', value=yson.dumps(keys))
        pod.spec.dynamic_attributes.annotations.append('qyp_ssh_authorized_keys')
    return pod


def prepare_pod_set(meta, spec, login=None):
    """
    :type meta: vmset_pb2.VMMeta
    :type spec: vmset_pb2.VMSpec
    :type login: str
    :rtype: data_model.TPodSet
    """
    pod_set = data_model.TPodSet()
    pod_set.meta.id = meta.id
    pod_set.meta.acl.extend(make_pod_acl(meta))
    pod_set.spec.node_segment_id = spec.qemu.node_segment
    yputil.cast_dict_to_attr_dict(yputil.make_qyp_labels_dict(spec), pod_set.labels)
    if spec.account_id:
        pod_set.spec.account_id = spec.account_id
    return pod_set


def set_vm_spec_defaults(spec):
    """

    :type spec: vmset_pb2.VMSpec
    """
    allocate_action_defaults = config.get_value('vmproxy.allocate_action_defaults', default={})
    if not allocate_action_defaults:
        return False, 'Disabled'

    if spec.qemu.volumes:
        main_volume = spec.qemu.volumes[0]

        if not main_volume.resource_url:
            main_volume_image = allocate_action_defaults['main_volume_image']
            main_volume_image_resource_url = None
            for image in config.get_value('images', []):
                if image['name'] == main_volume_image:
                    main_volume_image_resource_url = image['url']

            if not main_volume_image_resource_url:
                return False, 'Wrong config vmproxy.allocate_action_defaults.main_volume_image: {}, image not found'.format(
                    main_volume_image)
            main_volume.resource_url = main_volume_image_resource_url

    spec.qemu.autorun = allocate_action_defaults['autorun']
    spec.qemu.vm_type = vmset_pb2.VMType.Value(allocate_action_defaults['vm_type'])

    return True, None
