import base64
import uuid

import inject
import semantic_version
from collections import defaultdict

from infra.swatlib import sandbox
from infra.qyp.proto_lib import vmagent_api_pb2, vmset_pb2
from infra.qyp.vmproxy.src.action import helpers
from infra.qyp.vmproxy.src.lib.yp import yputil

BACKUP_VERSION = semantic_version.Version.coerce('0.15')
SB_TASK_BACKUP_VERSION = semantic_version.Version.coerce('0.18')
QDMUPLOAD_VERSION = semantic_version.Version.coerce('0.26')
QDMUPLOAD_BG_VERSION = semantic_version.Version.coerce('0.32')

QDM_BACKUP_SPEC_VERSION_1 = 1
SUPPORTED_QDM_BACKUP_SPEC_VERSIONS = (QDM_BACKUP_SPEC_VERSION_1,)


def run_vmagent_backup(instance, ctx):
    """
    :type instance: vmproxy.vm_instance.VMInstance
    :type ctx: vmproxy.web.app.Ctx
    """
    vmagent_req = vmagent_api_pb2.VMActionRequest()
    vmagent_req.action = vmagent_api_pb2.VMActionRequest.BACKUP
    encoded_data = base64.b64encode(vmagent_req.SerializeToString())
    return ctx.vmagent_client.action(url=instance.get_agent_url(), data=encoded_data)


def make_sandbox_task(instance, ctx, backup_storage):
    """
    :type instance: vmproxy.vm_instance.VMInstance
    :type ctx: vmproxy.web.app.Ctx
    :type backup_storage: str
    :rtype: str
    """
    sb_client = inject.instance(sandbox.ISandboxClient)
    task_data = {
        'type': 'BACKUP_QEMU_VM',
        'owner': 'QEMU_BACKUP',
        'description': 'QEMU backup task for pod "{}" from cluster "{}"'.format(
            instance.pod_id, ctx.pod_ctl_factory.cluster
        ),
        'priority': ('SERVICE', 'HIGH'),
        'custom_fields': [
            {
                'name': 'cluster',
                'value': ctx.pod_ctl_factory.cluster,
            },
            {
                'name': 'vm_id',
                'value': instance.pod_id,
            },
            {
                'name': 'storage',
                'value': backup_storage,
            },
        ],
        'requirements': {
            'disk_space': 1024 ** 3,  # 1Gb
        },
    }
    sb_task = sb_client._client.post('task', json=task_data)
    sb_task_id = sb_task['id']
    sb_client._client.put('batch/tasks/start', json={'id': [sb_task_id]})
    return str(sb_task_id)


def check_processing_backup(instance, ctx):
    """
    :type instance: vmproxy.vm_instance.VMInstance
    :type ctx: vmproxy.web.app.Ctx
    """
    backup_list = ctx.pod_ctl.get_backup_list(instance.pod)
    for item in backup_list:
        backup_pb = yputil.loads_proto(str(item), vmset_pb2.Backup)
        if backup_pb.status.state in (vmset_pb2.BackupStatus.PLANNED, vmset_pb2.BackupStatus.IN_PROGRESS):
            raise ValueError('Backup already in progress: {}, sb-task {}'.format(
                backup_pb.meta.id, backup_pb.spec.sandbox_task_id
            ))
    backup = ctx.qdm_client.backup_status(instance.pod_id)
    if backup and backup.spec.origin != vmset_pb2.BackupSpec.HOT:
        raise ValueError('Backup already in progress')


def start_sb_backup(instance, ctx, backup_storage):
    """
    :type instance: vmproxy.vm_instance.VMInstance
    :type ctx: vmproxy.web.app.Ctx
    :type backup_storage: str
    """
    backup_list = ctx.pod_ctl.get_backup_list(instance.pod)
    status = ctx.vmagent_client.status(url=instance.get_agent_url())
    status_resp = vmagent_api_pb2.VMStatusResponse()
    status_resp.ParseFromString(base64.b64decode(status))
    backup = vmset_pb2.Backup()
    backup.meta.id = str(uuid.uuid4())
    backup.meta.creation_time.GetCurrentTime()
    backup.spec.type = vmset_pb2.BackupSpec.SANDBOX_RESOURCE
    backup.spec.vm_id = instance.pod_id
    backup.spec.sandbox_task_id = make_sandbox_task(instance, ctx, backup_storage)
    backup.spec.generation = status_resp.state.generation
    backup.status.state = vmset_pb2.BackupStatus.PLANNED

    backup_list.append(yputil.dumps_proto(backup))
    ctx.pod_ctl.update_backup_list(instance.pod, backup_list)
    return backup.spec.sandbox_task_id


def rewrite_qemu_volumes_from_backup_spec(qemu_vm_spec, qdm_backup_spec):
    """

    :type qemu_vm_spec: infra.qyp.proto_lib.vmset_pb2.QemuVMSpec
    :type qdm_backup_spec: infra.qyp.proto_lib.qdm_pb2.QDMBackupSpec
    """
    if qdm_backup_spec.qdm_spec_version not in SUPPORTED_QDM_BACKUP_SPEC_VERSIONS:
        raise ValueError('Unsupported qdm backup spec version: {} no in {}'.format(
            qdm_backup_spec.qdm_spec_version, SUPPORTED_QDM_BACKUP_SPEC_VERSIONS))

    files_map = defaultdict(lambda: {'resource_urls': [], 'size': 0})
    qdm_rev_id = "qdm:{}".format(qdm_backup_spec.rev_id)

    for file_spec in qdm_backup_spec.filemap:
        file_info = files_map[file_spec.meta.volume_index]
        file_info['resource_urls'].append('{}/{}'.format(qdm_rev_id, file_spec.path))
        file_info['size'] += file_spec.size

    if not files_map:
        raise ValueError('qdm backup spec ({}) has no files'.format(qdm_rev_id))

    if len(files_map) < len(qemu_vm_spec.volumes):
        del qemu_vm_spec.volumes[len(files_map) + 1:]

    def update_volume_with_file_info(volume, _file_info):
        """
        :type volume:  infra.qyp.proto_lib.vmset_pb2.Volume
        :type _file_info: dict
        """
        volume.resource_url = ",".join(_file_info['resource_urls'])
        count_files = len(_file_info['resource_urls'])
        if count_files == 1:
            volume.image_type = vmset_pb2.Volume.RAW
        elif count_files == 2:
            volume.image_type = vmset_pb2.Volume.DELTA
        else:
            raise ValueError('Wrong count files: {}'.format(count_files))

    if not qdm_backup_spec.HasField('vmspec'):  #
        if not qemu_vm_spec.volumes:
            main_volume = qemu_vm_spec.volumes.add()
            main_volume.name = yputil.MAIN_VOLUME_NAME
            main_volume.pod_mount_path = yputil.MAIN_VOLUME_POD_MOUNT_PATH
            main_volume.vm_mount_path = yputil.MAIN_VOLUME_VM_MOUNT_PATH
            main_volume.capacity = files_map[0]['size']
        else:
            main_volume = qemu_vm_spec.volumes[0]
            main_volume.req_id = ''
            if main_volume.capacity < files_map[0]['size'] + 1024 ** 3:
                raise ValueError('Volume:{} has smaller capacity then image size {} < {} + 1G'.format(
                    main_volume.name, main_volume.capacity, files_map[0]['size']
                ))
        update_volume_with_file_info(main_volume, files_map[0])
    else:
        del qemu_vm_spec.volumes[:]
        for volume_index, qemu_volume_spec in enumerate(qdm_backup_spec.vmspec.spec.qemu.volumes):
            new_volume = qemu_vm_spec.volumes.add()
            new_volume.CopyFrom(qemu_volume_spec)
            new_volume.req_id = ''
            update_volume_with_file_info(new_volume, files_map[volume_index])


def run(instance, ctx, backup_storage='SANDBOX_RESOURCE'):
    """
    :type instance: vmproxy.vm_instance.VMInstance
    :type ctx: vmproxy.web.app.Ctx
    :type backup_storage: str
    """
    if instance.VM_TYPE == 'gencfg':
        return run_vmagent_backup(instance, ctx)

    check_processing_backup(instance, ctx)
    pod = instance.pod
    vmagent_version = helpers.get_vmagent_version(pod)
    if vmagent_version is None or vmagent_version < BACKUP_VERSION:
        raise ValueError('Minimal requested vmagent version {}, used {}'.format(BACKUP_VERSION, vmagent_version))
    elif vmagent_version < SB_TASK_BACKUP_VERSION:
        return run_vmagent_backup(instance, ctx)
    elif vmagent_version < QDMUPLOAD_VERSION:
        return start_sb_backup(instance, ctx, backup_storage)

    # With last vmagent one can choose storage type (yet)
    if backup_storage == 'SANDBOX_RESOURCE':
        return start_sb_backup(instance, ctx, backup_storage)
    elif backup_storage == 'QDM':
        return ctx.qdm_client.backup_create(vm_id=instance.pod_id)
    else:
        raise ValueError('Unknown storage type: {}'.format(backup_storage))
