import base64
import time
import uuid

import semantic_version
from google.protobuf import json_format

from infra.qyp.proto_lib import vmagent_api_pb2, vmagent_pb2
from infra.qyp.vmproxy.src.action import helpers
from infra.qyp.vmproxy.src.lib.yp import yputil


VM_CONFIG_RESOURCE_NAME = 'vm.config'
VM_CONFIG_ID_RESOURCE_NAME = 'vm_config_id'
ISS_PAYLOAD_CONFIG_STORAGE_VERSION = semantic_version.Version.coerce('0.3')
NO_DELTA_SIZE_VERSION = semantic_version.Version.coerce('0.10')
MULTI_STORAGE_VMAGENT_VERSION = semantic_version.Version.coerce('0.28')

MEMORY_GAP = 1024 ** 3  # 1gb

DISC_RESOURCE_HTTP_PREFIX = 'http://'
DISC_RESOURCE_HTTPS_PREFIX = 'https://'
DISC_RESOURCE_RBTORRENT_PREFIX = 'rbtorrent:'
DISK_RESOURCE_QDM_PREFIX = 'qdm:'


def size_to_gb(value):
    return value / (1024 ** 3)


def gen_config_id():
    return str(uuid.uuid4())


def validate_config_params(config_pb, vm_spec, vmagent_version):
    """
    :type config_pb: vmagent_pb2.VMConfig
    :type vm_spec: vmset_pb2.VMSpec
    :type vmagent_version: semantic_version.Version | NoneType
    """
    if not config_pb.id:
        config_pb.id = gen_config_id()
    # Validate CPU
    # vcpu_limit have been set in ms, so multiply
    if config_pb.vcpu * 1000 > vm_spec.qemu.resource_requests.vcpu_limit:
        msg = 'Config cpu should be less than allocation cpu limit {}, got {}'
        raise ValueError(msg.format(vm_spec.qemu.resource_requests.vcpu_limit / 1000, config_pb.vcpu))
    # Validate memory
    if config_pb.mem <= 0:
        raise ValueError('Config memory should be set')
    if config_pb.mem + MEMORY_GAP > vm_spec.qemu.resource_requests.memory_guarantee:
        msg = 'Config memory should be less than allocation memory guarantee {}Gb for at least {}Gb, got {}Gb'
        raise ValueError(msg.format(
            size_to_gb(vm_spec.qemu.resource_requests.memory_guarantee),
            size_to_gb(MEMORY_GAP),
            size_to_gb(config_pb.mem)
        ))
    # Validate disk size
    qemu_volume = None
    for v in vm_spec.qemu.volumes:
        if v.name == yputil.MAIN_VOLUME_NAME:
            qemu_volume = v
    if qemu_volume is None:
        # Should not happen
        raise ValueError('Volume request for qemu-persistent volume not found in allocation')
    if vmagent_version is None or vmagent_version < NO_DELTA_SIZE_VERSION:
        if config_pb.disk.delta_size <= 0:
            raise ValueError('Config disk.delta_size should be set')
        if config_pb.disk.delta_size > qemu_volume.capacity:
            msg = 'Config disk size should be less than allocation volume size {}Gb, got {}Gb'
            raise ValueError(msg.format(
                size_to_gb(qemu_volume.capacity),
                size_to_gb(config_pb.disk.delta_size)
            ))

    # Validate disc resource rb_torrent url
    if not config_pb.disk.resource.rb_torrent:
        raise ValueError('disk.resource.rb_torrent should be set')

    if not (config_pb.disk.resource.rb_torrent.startswith(DISC_RESOURCE_HTTP_PREFIX) or
            config_pb.disk.resource.rb_torrent.startswith(DISC_RESOURCE_HTTPS_PREFIX) or
            config_pb.disk.resource.rb_torrent.startswith(DISC_RESOURCE_RBTORRENT_PREFIX) or
            config_pb.disk.resource.rb_torrent.startswith(DISK_RESOURCE_QDM_PREFIX)):
        msg = "disk.resource.rb_torrent should start with '{}', '{}', '{}', '{}' (if config set)".format(
            DISC_RESOURCE_HTTP_PREFIX, DISC_RESOURCE_HTTPS_PREFIX,
            DISC_RESOURCE_RBTORRENT_PREFIX, DISK_RESOURCE_QDM_PREFIX
        )
        raise ValueError(msg)


def run_vmagent_config(instance, config, ctx):
    """
    :type instance: vmproxy.vm_instance.VMInstance
    :type config: vmagent_pb2.VMConfig
    :type ctx: vmproxy.web.app.Ctx
    """
    req = vmagent_api_pb2.VMActionRequest(config=config)
    req.action = vmagent_api_pb2.VMActionRequest.PUSH_CONFIG
    encoded_data = base64.b64encode(req.SerializeToString())
    return ctx.vmagent_client.action(url=instance.get_agent_url(), data=encoded_data)


def put_config_as_resource(iss, config):
    """
    :type iss: cluster_api_pb2.HostConfiguration
    :type config: vmagent_pb2.VMConfig
    """
    yputil.set_iss_payload_resource(
        iss,
        name=VM_CONFIG_RESOURCE_NAME,
        content=json_format.MessageToJson(config),
        dynamic=False
    )


def put_config_id_as_resource(iss, config_id=None):
    """
    :type iss: cluster_api_pb2.HostConfigurationInstance
    :type config_id: str | None
    """
    yputil.set_iss_payload_resource(
        iss,
        name=VM_CONFIG_ID_RESOURCE_NAME,
        content=config_id or str(uuid.uuid4()),
        dynamic=True
    )


def update_pod(pod, ctx):
    """
    :type pod: data_model_pb2.TPod
    :type ctx: vmproxy.web.app.Ctx
    """
    now = int(time.time())
    conf = '{}-{}'.format(pod.meta.id, now)
    iss_payload_pb = pod.spec.iss
    i = iss_payload_pb.instances[0]
    i.id.configuration.groupStateFingerprint = conf
    # Get vm version
    set_updates = {'/spec/iss': yputil.dumps_proto(iss_payload_pb)}
    version = yputil.cast_attr_dict_to_dict(pod.labels)['version']
    t_id, ts = ctx.pod_ctl.start_transaction()
    ctx.pod_ctl.update_pod(pod.meta.id, version, set_updates, t_id, ts)
    ctx.pod_ctl.commit_transaction(t_id)


def fill_vm_spec_from_vm_config(spec, config):
    """

    :type spec: infra.qyp.proto_lib.vmset_pb2.VMSpec
    :type config: infra.qyp.proto_lib.vmagent_pb2.VMConfig | None
    :return:
    """
    spec.qemu.volumes[0].resource_url = config.disk.resource.rb_torrent
    spec.qemu.volumes[0].image_type = config.disk.type
    spec.qemu.autorun = config.autorun
    spec.qemu.vm_type = config.type


def run(instance, config, ctx):
    """
    :type instance: vmproxy.vm_instance.VMInstance
    :type config: vmagent_pb2.VMConfig
    :type ctx: vmproxy.web.app.Ctx
    """
    if instance.VM_TYPE == 'gencfg':
        return run_vmagent_config(instance, config, ctx)
    else:
        pod = instance.pod
        vm = yputil.cast_pod_to_vm(
            pod_pb=pod,
            pod_set_pb=ctx.pod_ctl.get_pod_set(pod.meta.id)
        )
        vmagent_version = helpers.get_vmagent_version(pod)
        validate_config_params(config, vm.spec, vmagent_version)
        if vmagent_version is None or vmagent_version < ISS_PAYLOAD_CONFIG_STORAGE_VERSION:
            return run_vmagent_config(instance, config, ctx)
        elif ISS_PAYLOAD_CONFIG_STORAGE_VERSION <= vmagent_version < MULTI_STORAGE_VMAGENT_VERSION:
            put_config_as_resource(pod.spec.iss, config)
            return update_pod(pod, ctx)
        elif vmagent_version >= MULTI_STORAGE_VMAGENT_VERSION:
            raise ValueError('Vmagent with version >= {}, does not support Config Action, '
                             'use Update Action instead'.format(MULTI_STORAGE_VMAGENT_VERSION))
        else:
            raise ValueError('Incorrect vmagent version {}'.format(vmagent_version))
