from infra.qyp.vmproxy.src import errors
from infra.qyp.vmproxy.src.action import validation
from infra.qyp.vmproxy.src.action import helpers
from infra.qyp.vmproxy.src.action import allocate as allocate_action
from infra.qyp.vmproxy.src.action import config as config_action
from infra.qyp.vmproxy.src.action import list_free_nodes as list_free_nodes_action
from sepelib.core import config as vmproxy_config


def size_to_gb(value):
    return value / (1024 ** 3)


def update_spec_with_config(ctx, spec, config=None):
    """
    :type ctx: infra.qyp.vmproxy.src.web.app.Ctx
    :type spec: infra.qyp.proto_lib.vmset_pb2.VMSpec
    :type config: infra.qyp.proto_lib.vmagent_pb2.VMConfig | None
    """
    forced_node_id = spec.qemu.forced_node_id
    node_resources = list_free_nodes_action.get_free_resources_by_node(ctx, [forced_node_id])[forced_node_id]
    cpu = node_resources.cpu
    mem = node_resources.mem

    spec.qemu.resource_requests.vcpu_guarantee = cpu
    spec.qemu.resource_requests.vcpu_limit = cpu

    spec.qemu.resource_requests.memory_limit = mem
    spec.qemu.resource_requests.memory_guarantee = mem

    # QEMUKVM-723
    root_quota = vmproxy_config.get_value('vmproxy.default_porto_layer.root_quota')
    workdir_quota = vmproxy_config.get_value('vmproxy.default_porto_layer.workdir_quota')

    node_has_several_storage_classes = all([
        node_resources.disk_per_storage.get('hdd', None),
        node_resources.disk_per_storage.get('ssd', None)
    ])
    if not node_has_several_storage_classes and len(spec.qemu.volumes) == 1:
        # backward compatibility
        disk = node_resources.disk_per_storage.get(spec.qemu.volumes[0].storage_class)
        if not disk:
            raise ValueError('node {} has no disk with storage class {}'.format(
                forced_node_id, spec.qemu.volumes[0].storage_class))
        spec.qemu.volumes[0].capacity = disk - root_quota - workdir_quota
    else:
        node_resources.disk_per_storage[spec.qemu.volumes[0].storage_class] -= (root_quota + workdir_quota)
        for qemu_volume in spec.qemu.volumes:
            if qemu_volume.storage_class not in node_resources.disk_per_storage:
                raise ValueError('node {} has no disk with storage class {}'.format(
                    forced_node_id, qemu_volume.storage_class))
            node_resources.disk_per_storage[qemu_volume.storage_class] -= qemu_volume.capacity

        balance_result = {}
        for storage_class, balance in node_resources.disk_per_storage.items():
            if balance:
                balance_result[storage_class] = balance
        if balance_result:
            imbalanced_volumes = {k: "{}Gb".format(size_to_gb(v)) for k, v in balance_result.items()}
            raise ValueError('Qemu Volumes has wrong balance in node: {}'.format(imbalanced_volumes))

    if config:
        config.vcpu = cpu / 1000
        config.mem = mem - 1024 ** 3

    if node_resources.gpu_per_model:
        gpu_models = node_resources.gpu_per_model.keys()
        if len(gpu_models) > 1:
            raise ValueError('Node {} has more than one gpu model: {}'.format(forced_node_id, gpu_models))
        spec.qemu.gpu_request.model = gpu_models[0]
        spec.qemu.gpu_request.capacity = node_resources.gpu_per_model[gpu_models[0]]


def run(meta, spec, ctx, login, config=None):
    """
    :type meta: infra.qyp.proto_lib.vmset_pb2.VMMeta
    :type spec: infra.qyp.proto_lib.vmset_pb2.VMSpec
    :type ctx: infra.qyp.vmproxy.src.web.app.Ctx
    :type login: str
    :type config: infra.qyp.proto_lib.vmagent_pb2.VMConfig | None
    """
    if spec.qemu.forced_node_id:
        if not ctx.pod_ctl.forced_node_free(spec.qemu.forced_node_id):
            raise ValueError('Node has allocations')
        update_spec_with_config(ctx, spec, config)

    if config is not None:
        config_action.fill_vm_spec_from_vm_config(spec, config)
    if not spec.account_id:
        spec.account_id = 'tmp'

    validation.validate_allocate_request(ctx, meta, spec, login)
    validation.validate_macro(ctx, spec.qemu.network_id, login)

    for volume in spec.qemu.volumes:
        if volume.resource_url.startswith('qdm:'):
            backup_spec = ctx.qdm_client.get_revision_info(volume.resource_url)
            if backup_spec:
                validation.validate_qdm_backup_spec(ctx, backup_spec, login)
            else:
                raise ValueError('QDM not found resource with id: {}'.format(volume.resource_url))

    if spec.account_id and not ctx.pod_ctl.check_use_account_permission(spec.account_id, login, use_cache=False):
        msg = 'User {} has no access to account {}'.format(login, spec.account_id)
        raise errors.AuthorizationError(msg)

    validation.validate_personal_resource_fit(ctx, spec, login, ignore_disk_tax=False)

    if not ctx.sec_policy.is_root(login):
        # remove labels for non root users
        spec.ClearField('labels')

    helpers.add_admin_group_id_to_owners(ctx, meta, spec.account_id)
    pod = allocate_action.prepare_pod(meta, spec, login, ctx.pod_ctl)
    pod_set = allocate_action.prepare_pod_set(meta, spec)
    vmagent_version = helpers.get_vmagent_version(pod)
    if vmagent_version >= config_action.MULTI_STORAGE_VMAGENT_VERSION:
        config_action.put_config_id_as_resource(pod.spec.iss)
    else:
        raise ValueError('Incorrect vmagent version {}'.format(vmagent_version))

    ctx.pod_ctl.create_pod_with_pod_set(pod, pod_set)
