import base64

import cachetools.func

import inject
import semantic_version
from infra.qyp.proto_lib import vmagent_api_pb2
from infra.qyp.vmproxy.src.lib import abc_roles
from infra.qyp.vmproxy.src.lib.yp import yputil
from infra.swatlib.auth import staff
from yt import yson


def get_vmagent_version(pod):
    """
    :type pod: data_model_pb2.TPod
    :rtype: semantic_version.Version | NoneType
    """
    vmagent_version = yputil.cast_attr_dict_to_dict(pod.labels).get('vmagent_version')
    if vmagent_version:
        return semantic_version.Version.coerce(str(vmagent_version))
    else:
        return None


def make_encoded_resource(content):
    """
    :type content: str
    :rtype: str
    """
    encoded_content = base64.b64encode(content)
    return 'data:text/plain;charset=utf-8;base64,{}'.format(encoded_content)


def cast_vmset_to_vmagent_action_req(req):
    """
    :type req: vmset_api_pb2.MakeActionRequest
    :rtype: vmagent_api_pb2.VMActionRequest
    """
    vmagent_req = vmagent_api_pb2.VMActionRequest()
    vmagent_req.action = req.action
    vmagent_req.config.CopyFrom(req.config)
    vmagent_req.qdmreq.CopyFrom(req.qdmreq)
    vmagent_req.timeout = req.timeout
    return vmagent_req


def cast_role_scope_slug_to_group_id(role_scope_slug, service_id):
    """
    :type role_scope_slug: str
    :type service_id: str
    :rtype: str
    """
    staff_client = inject.instance(staff.IStaffClient)
    try:
        staff_resp = staff_client.list_groups({
            'parent.service.id': service_id,
            'role_scope': role_scope_slug,
        }, one=True, fields=('id',))
    except staff.StaffError:
        raise ValueError(
            'Cannot find group {} for service {}'.format(role_scope_slug, service_id)
        )
    return staff_resp['id']


@cachetools.func.ttl_cache(maxsize=500, ttl=60 * 60)
def get_staff_admin_group_ids(ctx, service_id):
    """
    :type ctx: infra.qyp.vmproxy.src.web.app.Ctx
    :type service_id: str
    :rtype: dict[str, str]
    """
    staff_client = inject.instance(staff.IStaffClient)
    service_scopes = ctx.pod_ctl.get_service_existing_scopes(service_id)
    try:
        staff_resp = staff_client.list_groups({
            'parent.service.id': service_id,
            'role_scope': service_scopes
        }, fields=('id', 'role_scope', 'parent.id'))
    except staff.StaffError:
        return None
    service_groups = {}
    for item in staff_resp['result']:
        # Will check that service itself is present in permissions
        service_groups['parent_id'] = str(item['parent']['id'])
        service_groups[item['role_scope']] = str(item['id'])
    return service_groups


def add_admin_group_id_to_owners(ctx, meta, account_id):
    """
    :type ctx: infra.qyp.vmproxy.src.web.app.Ctx
    :type meta: vmset_pb2.VMMeta
    :type account_id: str
    :rtype: bool
    """
    if account_id == abc_roles.TMP_ACCOUNT or account_id in ctx.personal_quotas_dict:
        return False
    service_id = account_id.split('abc:service:')[-1]
    service_groups = get_staff_admin_group_ids(ctx, service_id)
    if service_groups is None:
        # Staff error, do nothing
        return False
    for group_id in service_groups.values():
        if group_id in meta.auth.owners.group_ids:
            return False
    if abc_roles.ADMIN_ROLE_SCOPE_SLUG in service_groups:
        admin_group_id = service_groups[abc_roles.ADMIN_ROLE_SCOPE_SLUG]
    elif abc_roles.MANAGEMENT_ROLE_SCOPE_SLUG in service_groups:
        admin_group_id = service_groups[abc_roles.MANAGEMENT_ROLE_SCOPE_SLUG]
    else:
        # No service_management scope or no scopes at all
        # This will never happen
        return False
    meta.auth.owners.group_ids.append(admin_group_id)
    return True


def cast_group_id_to_subject(group_id, scopes):
    """
    :type group_id: str
    :type scopes: dict
    :rtype: str
    """
    staff_client = inject.instance(staff.IStaffClient)
    try:
        staff_resp = staff_client.list_groups({
            'id': group_id
        }, one=True, fields=('type', 'department', 'service', 'parent', 'role_scope'))
    except staff.StaffError:
        raise ValueError('Cannot find group {}'.format(group_id))
    group_type = staff_resp['type']
    if group_type == 'department':
        return 'staff:department:{}'.format(staff_resp['department']['id'])
    elif group_type == 'service':
        service_id = staff_resp['service']['id']
        return 'abc:service:{}'.format(service_id)
    elif group_type == 'servicerole':
        service_id = staff_resp['parent']['service']['id']
        scope_slug = staff_resp['role_scope']
        scope_id = scopes.get(scope_slug)
        return 'abc:service-scope:{}:{}'.format(service_id, scope_id)
    else:
        raise ValueError('Unknown group type: {} for group {}'.format(group_type, group_id))


def cast_owners_to_dict(owners):
    group_ids = map(int, owners.group_ids)
    owners_dict = {
        'logins': owners.logins,
        'groups': group_ids,
    }
    return owners_dict


def yson_dumps_list_of_proto(l):
    """

    :param l: collections.Iterable[google.protobuf.Message]
    :rtype: str
    """
    s = [yputil.dumps_proto(v) for v in l]
    return '[{}]'.format(';'.join(s))


def yson_dumps_vm(vm):
    """
    :type vm: vmset_pb2.VM
    :rtype: str
    """
    return yson.dumps(vm.SerializeToString(deterministic=True))
