from infra.qyp.proto_lib import vmset_pb2
from yt import yson

import cachetools.func


node_data_cache = cachetools.TTLCache(maxsize=10, ttl=6 * 60 * 60)
DISK_CAPACITY_RESERVE = 30 * 1024 ** 3
BLOCK_SIZE = 512


def quoted(items):
    for item in items:
        yield '"{}"'.format(item)


@cachetools.func.ttl_cache(maxsize=1, ttl=12 * 60 * 60)
def get_daemon_set_pod_set_ids(ctx):
    return ctx.pod_ctl.get_all_daemon_set_pod_set_ids()


def get_node_data(ctx, node_segment):
    """
    :type ctx: web.app.Ctx
    :type node_segment: str
    :rtype: dict[str, dict]
    """
    r = node_data_cache.get(node_segment)
    if r is not None:
        return r
    query_nodes = ' and '.join([
        "[/labels/extras/migration/source] != 'gencfg'",
        "[/labels/segment]='{}'".format(node_segment),
        "[/status/hfsm/state]='up'",
        "[/status/maintenance/state]='none'",
        "[/status/alerts] = #",
    ])
    result = ctx.pod_ctl.get_nodes_spec(query_nodes)
    node_data_cache[node_segment] = result
    return result


def get_free_disk_capacity(value):
    # QEMUKVM-1252
    if value:
        reserve_size = int(DISK_CAPACITY_RESERVE + value * 0.01)
        correct_size = (value - reserve_size) // BLOCK_SIZE * BLOCK_SIZE
        return correct_size
    else:
        return value


def get_free_resources_by_node(ctx, node_ids):
    result = {}
    resources = ctx.pod_ctl.list_resources_by_nodes(node_ids)
    for resource in resources:
        node_id, spec, free = [yson.loads(val) for val in resource.values]
        if node_id not in result:
            result[node_id] = vmset_pb2.ResourceInfo()
        res_info = result[node_id]
        for key, val in spec.iteritems():
            res_free = free[key]
            if key == 'cpu':
                res_info.cpu = res_free.get('capacity', 0)
            elif key == 'memory':
                res_info.mem = res_free.get('capacity', 0)
            elif key == 'disk':
                storage_class = val['storage_class']
                disk_capacity = res_free.get('capacity', 0)
                res_info.disk_per_storage[storage_class] = get_free_disk_capacity(disk_capacity)
                res_info.io_guarantees_per_storage[storage_class] = res_free.get('bandwidth', 0)
            elif key == 'gpu':
                res_info.gpu_per_model[val['model']] += res_free.get('capacity', 0)
    return result


def run(ctx, node_segment):
    """
    :type ctx: web.app.Ctx
    :type node_segment: str
    :rtype: dict[str, vmset_pb2.ResourceInfo]
    """
    node_spec_dict = get_node_data(ctx, node_segment)
    daemon_set_pod_set_ids = get_daemon_set_pod_set_ids(ctx)
    query_nodes = "not [/meta/pod_set_id] in ({})".format(','.join(quoted(daemon_set_pod_set_ids)))
    allocated_node_ids = set(ctx.pod_ctl.get_allocated_node_ids(query_nodes))
    free_node_ids = set(node_spec_dict.keys()) - allocated_node_ids
    node_free_resources = get_free_resources_by_node(ctx, free_node_ids)

    free_network_ids = ctx.pod_ctl.get_free_network_ids()
    free_nodes_with_network = set(key for key, data in node_spec_dict.iteritems()
                                  if data.get('network_module_id', None) in free_network_ids)

    for node_id in free_node_ids:
        node_data = node_free_resources[node_id]
        if not node_data.cpu or not node_data.mem or not len(node_data.disk_per_storage):
            del node_free_resources[node_id]

        node_data.internet_address = 1 if node_id in free_nodes_with_network else 0
    return node_free_resources


def match_node(nodes, pod_spec):
    """
    :type nodes: dict[str, vmset_pb2.ResourceInfo]
    :type pod_spec: data_model.TPodSpec
    :rtype: str|None
    """
    desired_resource_info = vmset_pb2.ResourceInfo()
    desired_resource_info.cpu = pod_spec.resource_requests.vcpu_limit
    desired_resource_info.mem = pod_spec.resource_requests.memory_limit
    if pod_spec.ip6_address_requests:
        ip_req = pod_spec.ip6_address_requests[0]
        if ip_req.ip4_address_pool_id or ip_req.enable_internet:
            desired_resource_info.internet_address = 1
    for volume in pod_spec.disk_volume_requests:
        desired_resource_info.disk_per_storage[volume.storage_class] += volume.quota_policy.capacity

    if pod_spec.gpu_requests:
        capacity = len(pod_spec.gpu_requests)
        model = pod_spec.gpu_requests[0].model
        desired_resource_info.gpu_per_model[model] = capacity

    for node_id, node in nodes.iteritems():
        if node.cpu != desired_resource_info.cpu:
            continue
        if node.mem != desired_resource_info.mem:
            continue
        if node.disk_per_storage != desired_resource_info.disk_per_storage:
            continue
        if node.gpu_per_model != desired_resource_info.gpu_per_model:
            continue
        if desired_resource_info.internet_address and not node.internet_address:
            continue
        return node_id
