import cachetools.func

from yt import yson

from infra.qyp.proto_lib import vmset_pb2
from infra.qyp.vmproxy.src import errors
from infra.qyp.vmproxy.src.lib import abc_roles


QYP_SEGMENTS = ['dev', 'default', 'gpu-dev']


def get_vms_dict(ctx, account_ids, segment, login):
    """
    :type ctx: vmproxy.web.app.Ctx
    :type account_ids: list[str]
    :type segment: str
    :type login: str
    :rtype: dict[str, vmset_pb2.Vm]
    """
    if not account_ids:
        return {}
    query = vmset_pb2.YpVmFindQuery()
    query.account.extend(account_ids)
    query.login = login
    if segment:
        query.segment.append(segment)
    vms_dict = {}
    for vm in ctx.pod_ctl.list_pods(query):
        vms_dict.setdefault(vm.spec.account_id, []).append(vm)
    return vms_dict


def set_personal_limits(limits, personal_quota):
    """
    :type limits: vmset_pb2.ResourceTotals
    :type personal_quota: dict
    """
    per_segment_limits = limits.per_segment[personal_quota['segment']]
    per_segment_limits.mem = personal_quota['mem']
    per_segment_limits.cpu = personal_quota['cpu']
    for disk in personal_quota['disk']:
        per_segment_limits.disk_per_storage[disk['storage']] = disk['capacity']
        per_segment_limits.io_guarantees_per_storage[disk['storage']] = disk['bandwidth_guarantee']
    per_segment_limits.internet_address = personal_quota['internet_address']


def set_personal_usage(usage, vms, login):
    """
    :type usage: vmset_pb2.ResourceTotals
    :type vms: list[vmset_pb2.Vm]
    :type login: str
    """
    for vm in vms:
        if vm.meta.author != login:
            continue
        per_segment_usage = usage.per_segment[vm.spec.qemu.node_segment]
        per_segment_usage.cpu += vm.spec.qemu.resource_requests.vcpu_guarantee
        per_segment_usage.mem += vm.spec.qemu.resource_requests.memory_guarantee
        for volume in vm.spec.qemu.volumes:
            per_segment_usage.disk_per_storage[volume.storage_class] += volume.capacity
        if vm.spec.qemu.enable_internet or vm.spec.qemu.ip4_address_pool_id:
            per_segment_usage.internet_address += 1
        for storage_class, value in vm.spec.qemu.io_guarantees_per_storage.items():
            per_segment_usage.io_guarantees_per_storage[storage_class] += value


def cast_yp_resources(pb, yp_resources):
    """
    :type pb: vmset_pb2.ResourceInfo
    :type yp_resources: data_model_pb2.TPerSegmentResourceTotals
    """
    pb.mem = yp_resources.memory.capacity
    pb.cpu = yp_resources.cpu.capacity
    for key, value in yp_resources.disk_per_storage_class.iteritems():
        pb.disk_per_storage[key] = value.capacity
        pb.io_guarantees_per_storage[key] = value.bandwidth
    pb.internet_address = yp_resources.internet_address.capacity
    for key, value in yp_resources.gpu_per_model.iteritems():
        pb.gpu_per_model[key] = value.capacity
    pb.network_guarantee = yp_resources.network.bandwidth


def set_limits(limits, account, segment=None):
    """
    :type limits: vmset_pb2.ResourceTotals
    :type account: data_model_pb2.TAccount
    :type segment: str | NoneType
    """
    if segment and segment in account.spec.resource_limits.per_segment:
        cast_yp_resources(limits.per_segment[segment], account.spec.resource_limits.per_segment[segment])
    else:
        for segment_, segment_limits in account.spec.resource_limits.per_segment.iteritems():
            if segment_ not in QYP_SEGMENTS:
                continue
            cast_yp_resources(limits.per_segment[segment_], segment_limits)


def set_usage(usage, account, segment=None):
    """
    :type usage: vmset_pb2.ResourceTotals
    :type account: data_model_pb2.TAccount
    :type segment: str | NoneType
    """
    if segment and segment in account.status.resource_usage.per_segment:
        cast_yp_resources(usage.per_segment[segment], account.status.resource_usage.per_segment[segment])
    else:
        for segment_, segment_usage in account.status.resource_usage.per_segment.iteritems():
            if segment_ not in QYP_SEGMENTS:
                continue
            cast_yp_resources(usage.per_segment[segment_], segment_usage)


@cachetools.func.ttl_cache(maxsize=1, ttl=3 * 60 * 60)
def get_search_members(pod_ctl):
    """
    :rtype: frozenset
    """
    # All search members are included in QYP personal account for now
    rsp = pod_ctl.yp_client.get_groups(['abc:service:4172'], selectors=['/meta/id', '/spec/members'])
    return frozenset(yson.loads(rsp.subresponses[0].result.values[1]))


@cachetools.func.ttl_cache(maxsize=1, ttl=5 * 60)
def get_yp_superusers(ctx):
    """
    :rtype: frozenset
    """
    group_rsp = ctx.pod_ctl.yp_client.get_group('superusers')
    members = yson.loads(group_rsp.result.values[0]).get('spec').get('members') or []
    return frozenset(members)


def run(ctx, login, account_id, segment, use_cache):
    """
    :type ctx: vmproxy.web.app.Ctx
    :type login: str
    :type account_id: str
    :type segment: str
    :type use_cache: bool
    :rtype: list[vmset_pb2.Account]
    """
    if account_id:
        if not ctx.pod_ctl.check_use_account_permission(account_id, login, use_cache):
            raise errors.AuthorizationError('{} must be a member of a {} account'.format(login, account_id))
        account_ids = [account_id]
    else:
        if login in get_yp_superusers(ctx):
            service_ids = abc_roles.list_user_services_from_abc(login)
            account_ids = ['abc:service:{}'.format(service_id) for service_id in service_ids]
        else:
            account_ids = ctx.pod_ctl.list_account_ids_by_login(login)
        account_ids = abc_roles.filter_accounts_by_roles(list(account_ids), login, use_cache)
        if not account_ids:
            return []

    accounts = ctx.pod_ctl.get_accounts(account_ids)
    groups = ctx.pod_ctl.yp_client.get_groups(account_ids, selectors=['/meta/id', '/spec/members'])
    members_dict = {}
    for item in groups.subresponses:
        if len(item.result.values) == 2:
            members_dict[yson.loads(item.result.values[0])] = item.result.values[1]
    result = []
    personal_account_ids = list(set(account_ids).intersection(set(ctx.personal_quotas_dict)))
    vms_dict = get_vms_dict(ctx, personal_account_ids, segment, login)
    for account in accounts:
        acc = vmset_pb2.Account()
        acc.id = account.meta.id
        set_limits(acc.limits, account, segment)
        set_usage(acc.usage, account, segment)
        if acc.id in members_dict:
            members = abc_roles.get_service_qyp_persons(acc.id, use_cache) or list(yson.loads(members_dict[acc.id]))
            search_members = get_search_members(ctx.pod_ctl)
            acc.members_count = sum(int(member in search_members) for member in members)
        if account.meta.id in ctx.personal_quotas_dict:
            acc.type = vmset_pb2.Account.PERSONAL
            set_personal_limits(acc.personal.limits, ctx.personal_quotas_dict[account.meta.id])
            set_personal_usage(acc.personal.usage, vms_dict.get(account.meta.id) or [], login)
        else:
            acc.type = vmset_pb2.Account.SERVICE
        result.append(acc)
    return result
