import base64
import semantic_version
import logging
import uuid

from yt import yson
import yp.data_model as data_model
from yp_proto.yp.client.api.proto import cluster_api_pb2

from sepelib.core import config as sepelib_config

from infra.qyp.vmproxy.src.lib.yp import yputil
from infra.qyp.vmproxy.src import errors, vm_instance
from infra.qyp.proto_lib import vmagent_api_pb2, vmset_pb2, vmagent_pb2
from infra.qyp.vmproxy.src.action import allocate as allocate_action
from infra.qyp.vmproxy.src.action import config as config_action
from infra.qyp.vmproxy.src.action import create as create_action
from infra.qyp.vmproxy.src.action import validation, helpers

UPDATE_VMAGENT_VM_MARK_LABEL = 'update_vmagent_process'

log = logging.getLogger('UpdateVm')


def get_vm_status(ctx, pod_pb):
    """

    :type ctx:
    :type pod_pb:
    :rtype:  vmagent_api_pb2.VMStatusResponse
    """
    container_ip = ctx.pod_ctl.get_pod_container_ip(pod_pb)
    vmagent_port = sepelib_config.get_value('vmproxy.default_agent_port')
    instance = vm_instance.PodVMInstance(pod_pb.meta.id, pod_pb, container_ip, vmagent_port)
    status = ctx.vmagent_client.status(instance.get_agent_url())
    vmagent_resp = vmagent_api_pb2.VMStatusResponse()
    vmagent_resp.ParseFromString(base64.b64decode(status))
    return vmagent_resp


def get_vm_config(ctx, pod_pb):
    """

    :type ctx:
    :type pod_pb:
    :rtype:  vmagent_pb2.VMConfig
    """
    vmagent_resp = get_vm_status(ctx, pod_pb)
    return vmagent_resp.config


def recreate_vm(ctx, meta, spec, login, pod_set_updates):
    new_pod_pb = allocate_action.prepare_pod(meta, spec, login, ctx.pod_ctl)
    config_action.put_config_id_as_resource(new_pod_pb.spec.iss)
    ctx.pod_ctl.update_pod_with_move(
        pod_id=meta.id,
        pod_version=meta.version.pod,
        pod_set_version=meta.version.pod_set,
        pod=new_pod_pb,
        pod_set_updates=pod_set_updates
    )


def fill_vm_spec_from_config_if_exists(spec, config=None):
    """

    :type spec:
    :type config:
    """
    if config is None:
        return False
    if not spec.qemu.volumes[0].resource_url:
        spec.qemu.volumes[0].resource_url = config.disk.resource.rb_torrent
        spec.qemu.volumes[0].image_type = config.disk.type
    spec.qemu.autorun = config.autorun
    return True


def update_node_id(ctx, old_vm, spec, config=None):
    if spec.qemu.forced_node_id == old_vm.spec.qemu.forced_node_id:
        return False

    if spec.qemu.forced_node_id:
        if not ctx.pod_ctl.forced_node_free(spec.qemu.forced_node_id):
            raise ValueError('Node has allocations')
        create_action.update_spec_with_config(ctx, spec, config=config)
    return True


def update_account(ctx, spec, old_vm, login, pod_updates, pod_set_updates):
    if spec.account_id == old_vm.spec.account_id:
        return False
    if not ctx.pod_ctl.check_use_account_permission(spec.account_id, login, use_cache=False):
        msg = 'User {} has no access to account {}'.format(login, spec.account_id)
        raise errors.AuthorizationError(msg)
    if spec.account_id in ctx.personal_quotas_dict:
        pod_updates['/annotations/owners/author'] = yson.dumps(login)
    pod_set_updates['/spec/account_id'] = yson.dumps(spec.account_id)
    return True


def update_owners(ctx, meta, account_id, old_vm, login, pod_updates, pod_set_updates):
    # First try to add admin group to owners.
    helpers.add_admin_group_id_to_owners(ctx, meta, account_id)

    if meta.auth == old_vm.meta.auth:
        return False

    owners_dict = helpers.cast_owners_to_dict(meta.auth.owners)

    pod_updates['/annotations/owners/logins'] = yson.dumps(owners_dict['logins'])
    pod_updates['/annotations/owners/groups'] = yson.dumps(owners_dict['groups'])

    pod_set_updates['/meta/acl'] = helpers.yson_dumps_list_of_proto(allocate_action.make_pod_acl(meta))
    return True


def update_network_project(ctx, old_vm, spec, pod_pb, pod_updates, login):
    if spec.qemu.network_id == old_vm.spec.qemu.network_id:
        return False

    validation.validate_macro(ctx, spec.qemu.network_id, login)
    for index, addr_req in enumerate(pod_pb.spec.ip6_address_requests):
        labels = yputil.cast_attr_dict_to_dict(addr_req.labels)
        if labels['owner'] == 'vm':
            ypath = '/spec/ip6_address_requests/{}/network_id'.format(index)
            pod_updates[ypath] = yson.dumps(spec.qemu.network_id)
    return True


def update_resource_requests(old_vm, spec, pod_updates):
    if spec.qemu.resource_requests == old_vm.spec.qemu.resource_requests:
        return False
    req = data_model.TPodSpec.TResourceRequests()
    req.dirty_memory_limit = spec.qemu.resource_requests.dirty_memory_limit
    req.memory_limit = spec.qemu.resource_requests.memory_limit
    req.anonymous_memory_limit = spec.qemu.resource_requests.anonymous_memory_limit
    req.vcpu_guarantee = spec.qemu.resource_requests.vcpu_guarantee
    req.vcpu_limit = spec.qemu.resource_requests.vcpu_limit
    req.memory_guarantee = spec.qemu.resource_requests.memory_guarantee
    req.network_bandwidth_guarantee = spec.qemu.resource_requests.network_bandwidth_guarantee
    # commented due to QEMUKVM-1679
    # req.network_bandwidth_limit = spec.qemu.resource_requests.network_bandwidth_guarantee
    pod_updates['/spec/resource_requests'] = yputil.dumps_proto(req)
    return True


def update_gpu_requests(old_vm, spec, new_iss_proto, pod_updates):
    """
    :type old_vm: vmset_pb2.Vm
    :type spec: vmset_pb2.VmSpec
    :type new_iss_proto: cluster_api_pb2.HostConfiguration
    :type pod_updates: dict[str, str]
    :rtype: bool
    """
    if spec.qemu.gpu_request == old_vm.spec.qemu.gpu_request:
        return False
    gpu_reqs = []
    for _ in range(spec.qemu.gpu_request.capacity):
        req = data_model.TPodSpec.TGpuRequest()
        req.id = str(uuid.uuid4())
        req.model = spec.qemu.gpu_request.model
        gpu_reqs.append(req)
    pod_updates['/spec/gpu_requests'] = helpers.yson_dumps_list_of_proto(gpu_reqs)

    i = new_iss_proto.instances[0]
    gpu_req = spec.qemu.gpu_request
    i.properties['QYP_GPU'] = '{}-{}-{}-{}'.format(
        gpu_req.model, gpu_req.capacity, gpu_req.min_memory, gpu_req.max_memory
    )
    return True


def update_qemu_options(old_vm, spec, new_iss_proto):
    """
    :type old_vm: vmset_pb2.Vm
    :type spec: vmset_pb2.VmSpec
    :type new_iss_proto: cluster_api_pb2.HostConfiguration
    :rtype: bool
    """
    if spec.qemu.qemu_options == old_vm.spec.qemu.qemu_options:
        return False
    return True


def update_use_nat64(old_vm, spec, new_iss_proto):
    if spec.qemu.use_nat64 == old_vm.spec.qemu.use_nat64:
        return False
    yputil.update_iss_payload_use_nat64(new_iss_proto, spec)
    return True


def update_config(ctx, spec, config, new_iss_proto, pod_pb, semantic_vmagent_version):
    if config.ListFields():
        old_config = get_vm_config(ctx, pod_pb)
        config.id = old_config.id
        if config == old_config:
            return False

        config.id = ""
        config_action.validate_config_params(config, spec, semantic_vmagent_version)
        config_action.put_config_as_resource(new_iss_proto, config)
        return True


def update_config_id(new_iss_proto):
    config_action.put_config_id_as_resource(new_iss_proto, config_action.gen_config_id())


def update_autorun(old_vm, spec):
    if spec.qemu.autorun == old_vm.spec.qemu.autorun:
        return False
    return True


def update_vm_type(old_vm, spec):
    if old_vm.spec.qemu.vm_type == spec.qemu.vm_type:
        return False

    return True


def update_volumes(ctx, pod_pb, current_iss_proto, old_vm, spec, meta, new_iss_proto, pod_updates, login):
    # update volumes
    current_volumes = {v.name: v for v in old_vm.spec.qemu.volumes}
    new_volumes = {v.name: v for v in spec.qemu.volumes}
    for new_volume_name, new_volume in new_volumes.items():
        current_volume = current_volumes.get(new_volume_name)  # type: vmset_pb2.Volume
        if current_volume is not None:
            if new_volume.resource_url.startswith('qdm:') and current_volume.resource_url != new_volume.resource_url:
                backup_spec = ctx.qdm_client.get_revision_info(new_volume.resource_url)
                if backup_spec:
                    validation.validate_qdm_backup_spec(ctx, backup_spec, login)
                else:
                    raise ValueError('QDM not found resource with id: {}'.format(new_volume.resource_url))
            new_volume.req_id = current_volume.req_id  # User can't change disk volume request id
            new_volume.pod_mount_path = current_volume.pod_mount_path  # User can't change disk pod_mount_path
            new_volume.vm_mount_path = current_volume.vm_mount_path  # User can't change vm_mount_path
            if current_volume.capacity != new_volume.capacity and not update_node_id:
                raise ValueError('Volume:{} has changed capacity, {} != {}'.format(
                    new_volume.name, current_volume.capacity, new_volume.capacity))
            if current_volume.storage_class != new_volume.storage_class:
                raise ValueError('Volume:{} has changed storage_class, {} != {}'.format(
                    new_volume.name, current_volume.storage_class, new_volume.storage_class))
        else:
            if new_volume.resource_url.startswith('qdm:'):
                backup_spec = ctx.qdm_client.get_revision_info(new_volume.resource_url)
                if backup_spec:
                    validation.validate_qdm_backup_spec(ctx, backup_spec, login)
                else:
                    raise ValueError('QDM not found resource with id: {}'.format(new_volume.resource_url))

    old_io_guarantees = old_vm.spec.qemu.io_guarantees_per_storage
    new_io_guarantees = dict(old_io_guarantees)
    if spec.qemu.io_guarantees_per_storage:
        new_io_guarantees.update(spec.qemu.io_guarantees_per_storage)
    if current_volumes == new_volumes and old_io_guarantees == new_io_guarantees:
        return False

    root_fs_disc_volume_req = data_model.TPodSpec.TDiskVolumeRequest()
    root_fs_disc_volume_req.CopyFrom(pod_pb.spec.disk_volume_requests[0])
    del pod_pb.spec.disk_volume_requests[:]

    yputil.cast_qemu_volumes_to_pod_disk_volume_requests(
        pod_id=meta.id,
        vm_spec=spec,
        disk_volume_requests=pod_pb.spec.disk_volume_requests,
        root_fs_req=root_fs_disc_volume_req)

    pod_updates['/spec/disk_volume_requests'] = helpers.yson_dumps_list_of_proto(pod_pb.spec.disk_volume_requests)

    _new_iss_proto = yputil.make_iss_payload(
        pod_id=meta.id,
        vm_spec=old_vm.spec,
        yp_cluster=ctx.pod_ctl.yp_cluster,
        volumes=pod_pb.spec.disk_volume_requests,
        root_storage=yputil.get_iss_payload_root_storage_class(current_iss_proto),
        pod_resources=yputil.get_iss_payload_pod_resources(current_iss_proto),
    )
    new_iss_proto.CopyFrom(_new_iss_proto)
    return True


def run_for_vmagent_ver_lt_0_26(meta, spec, ctx, login, old_vm):
    """
    Safe Update VM Actions
        - spec.qemu.account_id
        - meta.auth.owners

    Remove this after update all vm to vmagent gte 0.28

    :type meta: infra.qyp.proto_lib.vmset_pb2.VMMeta
    :type spec: infra.qyp.proto_lib.vmset_pb2.VMSpec
    :type ctx: infra.qyp.vmproxy.src.web.app.Ctx
    :type login: str
    :type old_vm: vmset_pb2.VM
    """
    pod_updates = {}
    pod_set_updates = {}

    account_changed = update_account(ctx, spec, old_vm, login, pod_updates, pod_set_updates)
    owners_changed = update_owners(ctx, meta, spec.account_id, old_vm, login, pod_updates, pod_set_updates)
    log.info('Update vm({}) with vmagent ver < 0.26: '
             '\naccount: {account_changed}'
             '\nowners: {owners_changed}'
             ''.format(meta.id, **locals()))
    new_labels = yputil.make_qyp_labels_dict(spec)
    if pod_updates:
        for key, value in new_labels.iteritems():
            pod_updates['/labels/{}'.format(key)] = yson.dumps(value)

    if pod_set_updates:
        for key, value in new_labels.iteritems():
            pod_set_updates['/labels/{}'.format(key)] = yson.dumps(value)

    if not pod_updates and not pod_set_updates:
        return

    # commit
    t_id, ts = ctx.pod_ctl.start_transaction()
    ctx.pod_ctl.update_pod(meta.id, meta.version.pod, pod_updates, t_id, ts)
    ctx.pod_ctl.update_pod_set(meta.id, meta.version.pod_set, pod_set_updates, t_id, ts)
    ctx.pod_ctl.commit_transaction(t_id)


def run_for_vmagent_ver_gte_0_26_and_lt_0_28(meta, spec, ctx, login, pod_pb, old_vm, config, vmagent_version):
    """
    :type meta: infra.qyp.proto_lib.vmset_pb2.VMMeta
    :type spec: infra.qyp.proto_lib.vmset_pb2.VMSpec
    :type config: infra.qyp.proto_lib.vmagent_pb2.VMConfig
    :type ctx: infra.qyp.vmproxy.src.web.app.Ctx
    :type pod_pb: data_model.TPod
    :type old_vm: vmset_pb2.VM
    :type login: str
    :type vmagent_version: str
    """
    pod_updates = {}
    pod_set_updates = {}
    current_iss_proto = pod_pb.spec.iss
    new_iss_proto = cluster_api_pb2.HostConfiguration()
    new_iss_proto.CopyFrom(current_iss_proto)

    node_id_changed = update_node_id(ctx, old_vm, spec, config=config)

    account_changed = update_account(ctx, spec, old_vm, login, pod_updates, pod_set_updates)
    network_changed = update_network_project(ctx, old_vm, spec, pod_pb, pod_updates, login)
    use_nat64_changed = update_use_nat64(old_vm, spec, new_iss_proto)

    owners_changed = update_owners(ctx, meta, spec.account_id, old_vm, login, pod_updates, pod_set_updates)

    resource_requests_changed = update_resource_requests(old_vm, spec, pod_updates)
    config_changed = update_config(ctx, spec, config, new_iss_proto, pod_pb, vmagent_version)
    log.info('Update vm({}) with vmagent ver >= 0.26 and < 0.28: '
             '\n\tnode_id: {node_id_changed}'
             '\n\taccount: {account_changed}'
             '\n\tnetwork: {network_changed}'
             '\n\tuse_nat64: {use_nat64_changed}'
             '\n\towners: {owners_changed}'
             '\n\tresource_requests: {resource_requests_changed}'
             '\n\tconfig: {config_changed}'
             ''.format(meta.id, **locals()))
    if new_iss_proto != current_iss_proto:
        yputil.update_iss_payload_config_fingerprint(new_iss_proto, meta.id)
        pod_updates['/spec/iss'] = yputil.dumps_proto(new_iss_proto)

    new_labels = yputil.make_qyp_labels_dict(spec)
    if pod_updates:
        for key, value in new_labels.iteritems():
            pod_updates['/labels/{}'.format(key)] = yson.dumps(value)

    if pod_set_updates:
        for key, value in new_labels.iteritems():
            pod_set_updates['/labels/{}'.format(key)] = yson.dumps(value)

    if node_id_changed:
        recreate_vm(ctx, meta, spec, login, pod_set_updates)
        return

    if not pod_updates and not pod_set_updates:
        return

    # commit
    t_id, ts = ctx.pod_ctl.start_transaction()
    ctx.pod_ctl.update_pod(meta.id, meta.version.pod, pod_updates, t_id, ts)
    ctx.pod_ctl.update_pod_set(meta.id, meta.version.pod_set, pod_set_updates, t_id, ts)
    ctx.pod_ctl.commit_transaction(t_id)


def run_for_vmagent_ver_gte_0_28(ctx, meta, spec, login, pod_pb, old_vm):
    """

    :type ctx: infra.qyp.vmproxy.src.web.app.Ctx
    :type meta: infra.qyp.proto_lib.vmset_pb2.VMMeta
    :type spec: infra.qyp.proto_lib.vmset_pb2.VMSpec
    :type pod_pb: data_model.TPod
    :type login: str
    :type old_vm: vmset_pb2.VM
    """
    pod_updates = {}
    pod_set_updates = {}
    current_iss_proto = pod_pb.spec.iss
    new_iss_proto = cluster_api_pb2.HostConfiguration()
    new_iss_proto.CopyFrom(current_iss_proto)

    vm_type_changed = update_vm_type(old_vm, spec)
    if vm_type_changed:
        raise ValueError('Change vm type deprecated!!')

    node_id_changed = update_node_id(ctx, old_vm, spec)

    volumes_changed = update_volumes(ctx, pod_pb, current_iss_proto, old_vm, spec, meta, new_iss_proto, pod_updates,
                                     login)

    account_changed = update_account(ctx, spec, old_vm, login, pod_updates, pod_set_updates)
    network_changed = update_network_project(ctx, old_vm, spec, pod_pb, pod_updates, login)
    use_nat64_changed = update_use_nat64(old_vm, spec, new_iss_proto)

    owners_changed = update_owners(ctx, meta, spec.account_id, old_vm, login, pod_updates, pod_set_updates)

    resource_requests_changed = update_resource_requests(old_vm, spec, pod_updates)
    gpu_requests_changed = update_gpu_requests(old_vm, spec, new_iss_proto, pod_updates)
    qemu_options_changed = update_qemu_options(old_vm, spec, new_iss_proto)

    autorun_changed = update_autorun(old_vm, spec)

    log.info('Update vm({}) with vmagent ver >= 0.28: '
             '\n\tnode_id: {node_id_changed}'
             '\n\tvolumes: {volumes_changed}'
             '\n\taccount: {account_changed}'
             '\n\tnetwork: {network_changed}'
             '\n\tuse_nat64: {use_nat64_changed}'
             '\n\towners: {owners_changed}'
             '\n\tresource_requests: {resource_requests_changed}'
             '\n\tgpu_requests: {gpu_requests_changed}'
             '\n\tqemu_options: {qemu_options_changed}'
             '\n\tautorun_changed: {autorun_changed}'
             ''.format(meta.id, **locals()))

    if (volumes_changed or autorun_changed or resource_requests_changed or owners_changed or gpu_requests_changed or
        qemu_options_changed):
        config_action.put_config_id_as_resource(new_iss_proto, config_action.gen_config_id())

    if new_iss_proto != current_iss_proto:
        yputil.update_iss_payload_config_fingerprint(new_iss_proto, meta.id)
        pod_updates['/spec/iss'] = yputil.dumps_proto(new_iss_proto)

    if (resource_requests_changed or autorun_changed or volumes_changed or owners_changed or account_changed or
        gpu_requests_changed or qemu_options_changed):
        pod_updates['/annotations/qyp_vm_spec'] = helpers.yson_dumps_vm(vmset_pb2.VM(spec=spec, meta=meta))

    new_labels = yputil.make_qyp_labels_dict(spec)
    if pod_updates:
        for key, value in new_labels.iteritems():
            pod_updates['/labels/{}'.format(key)] = yson.dumps(value)

    if pod_set_updates:
        for key, value in new_labels.iteritems():
            pod_set_updates['/labels/{}'.format(key)] = yson.dumps(value)

    if node_id_changed:
        recreate_vm(ctx, meta, spec, login, pod_set_updates)
        return

    if not pod_updates and not pod_set_updates:
        return

    # commit
    t_id, ts = ctx.pod_ctl.start_transaction()
    ctx.pod_ctl.update_pod(meta.id, meta.version.pod, pod_updates, t_id, ts)
    ctx.pod_ctl.update_pod_set(meta.id, meta.version.pod_set, pod_set_updates, t_id, ts)
    ctx.pod_ctl.commit_transaction(t_id)


def run_update_vmagent(ctx, pod_pb, old_vm, spec, meta, login, force=False):
    """

    :type ctx: infra.qyp.vmproxy.src.web.app.Ctx
    :type meta: infra.qyp.proto_lib.vmset_pb2.VMMeta
    :type spec: infra.qyp.proto_lib.vmset_pb2.VMSpec
    :type login: str
    :type force: bool
    """
    pod_updates = {}
    pod_set_updates = {}
    current_iss_proto = pod_pb.spec.iss
    new_vm = vmset_pb2.VM(spec=old_vm.spec, meta=old_vm.meta)

    vmagent_ver_0_26 = semantic_version.Version.coerce('0.26')
    vmagent_ver_0_28 = semantic_version.Version.coerce('0.28')

    current_vmagent_version = helpers.get_vmagent_version(pod_pb)

    pod_resources = yputil.get_iss_payload_pod_resources(current_iss_proto)
    vm_config = yputil.get_config_from_iss_resource(current_iss_proto)

    if vmagent_ver_0_26 <= current_vmagent_version < vmagent_ver_0_28:
        disk_is_raw = new_vm.spec.qemu.volumes[0].image_type == vmagent_pb2.VMDisk.RAW
        disk_resource_is_qdm = new_vm.spec.qemu.volumes[0].resource_url.startswith('qdm:')
        if disk_is_raw and disk_resource_is_qdm:
            vm_status = get_vm_status(ctx, pod_pb)
            if vm_status.state.type in (vmagent_pb2.VMState.EMPTY, vmagent_pb2.VMState.INVALID):
                raise ValueError("Can't update vmagent for vm ({}) with state: {}".format(
                    new_vm.meta.id, vmagent_pb2.VMState.VMStateType.Name(vm_status.state.type)))
            vm_config = vm_status.config
            if vm_config.disk.type == vmagent_pb2.VMDisk.DELTA:
                new_vm.spec.qemu.volumes[0].image_type = vmset_pb2.Volume.DELTA
                log.info('VM with id: {} replace image type from current vm config'.format(old_vm.meta.id))

    if current_vmagent_version < vmagent_ver_0_28 and vm_config:
        # Check VM vcpu
        vm_vcpu = vm_config.vcpu * 1000
        vcpu_limit_changed = False
        if not force and vm_vcpu != new_vm.spec.qemu.resource_requests.vcpu_limit:
            if vm_vcpu == 0:
                raise ValueError("Can't update vmagent for vm ({}) with vcpu == 0".format(new_vm.meta.id))
            if vm_vcpu < new_vm.spec.qemu.resource_requests.vcpu_guarantee:
                raise ValueError("Can't update vmagent for vm ({}) with vcpu({}) < vcpu_guarantee({})".format(
                    new_vm.meta.id, vm_vcpu, new_vm.spec.qemu.resource_requests.vcpu_guarantee))

            new_vm.spec.qemu.resource_requests.vcpu_limit = vm_vcpu
            vcpu_limit_changed = True

        # Check VM memory
        vm_memory = vm_config.mem + 1024 ** 3
        memory_changed = False
        if not force and vm_memory != new_vm.spec.qemu.resource_requests.memory_limit:
            if vm_memory < 2 * 1024 ** 3:
                raise ValueError("Can't update vmagent for vm ({}) with vm.memory({}) < 2G".format(
                    vm_memory, new_vm.meta.id
                ))
            new_vm.spec.qemu.resource_requests.memory_limit = vm_memory
            new_vm.spec.qemu.resource_requests.memory_guarantee = vm_memory
            memory_changed = True

        if vcpu_limit_changed or memory_changed:
            update_resource_requests(old_vm, new_vm.spec, pod_updates)

    default_pod_resource = yputil.get_default_pod_resources()
    pod_resources.update(default_pod_resource)

    new_vm.spec.vmagent_version = sepelib_config.get_value('vmproxy.default_vmagent.version')

    if yputil.can_user_change_pod_resources(login) and spec.qemu.pod_resources:
        pod_resources.update(spec.qemu.pod_resources)

        if yputil.VMAGENT_RESOURCE_NAME in spec.qemu.pod_resources:
            new_vm.spec.vmagent_version = spec.vmagent_version

    new_iss_proto = yputil.make_iss_payload(
        pod_id=meta.id,
        vm_spec=new_vm.spec,
        yp_cluster=ctx.pod_ctl.yp_cluster,
        volumes=pod_pb.spec.disk_volume_requests,
        root_storage=yputil.get_iss_payload_root_storage_class(current_iss_proto),
        pod_resources=pod_resources,
        porto_layer_url=spec.qemu.porto_layer.url
    )

    config_action.put_config_id_as_resource(new_iss_proto, config_action.gen_config_id())

    if current_vmagent_version >= vmagent_ver_0_26:
        # prevent from restart instance after update vmagent with ver > 0.26
        exists_hook_time_limits = current_iss_proto.instances[0].entity.instance.timeLimits
        new_iss_proto.instances[0].entity.instance.ClearField('timeLimits')
        for hook_name, time_limit in exists_hook_time_limits.iteritems():
            new_iss_proto.instances[0].entity.instance.timeLimits[hook_name].CopyFrom(time_limit)

    root_fs_disc_volume_req = data_model.TPodSpec.TDiskVolumeRequest()
    root_fs_disc_volume_req.CopyFrom(pod_pb.spec.disk_volume_requests[0])
    del pod_pb.spec.disk_volume_requests[:]

    yputil.cast_qemu_volumes_to_pod_disk_volume_requests(
        pod_id=meta.id,
        vm_spec=new_vm.spec,
        disk_volume_requests=pod_pb.spec.disk_volume_requests,
        root_fs_req=root_fs_disc_volume_req)

    pod_updates['/spec/disk_volume_requests'] = helpers.yson_dumps_list_of_proto(pod_pb.spec.disk_volume_requests)

    pod_updates['/spec/iss'] = yputil.dumps_proto(new_iss_proto)
    pod_updates['/spec/dynamic_attributes/annotations'] = yson.dumps(['qyp_vm_spec', 'qyp_ssh_authorized_keys'])

    new_vm.meta.ClearField('version')
    new_vm.meta.ClearField('creation_time')
    new_vm.meta.ClearField('last_modification_time')
    new_vm.spec.qemu.ClearField('porto_layer')
    pod_updates['/annotations/qyp_vm_spec'] = helpers.yson_dumps_vm(new_vm)

    new_vm.spec.labels[UPDATE_VMAGENT_VM_MARK_LABEL] = 'in_progress'
    new_labels = yputil.make_qyp_labels_dict(new_vm.spec)
    for key, value in new_labels.iteritems():
        pod_updates['/labels/{}'.format(key)] = yson.dumps(value)
        pod_set_updates['/labels/{}'.format(key)] = yson.dumps(value)

    # commit
    t_id, ts = ctx.pod_ctl.start_transaction()
    ctx.pod_ctl.update_pod(meta.id, meta.version.pod, pod_updates, t_id, ts)
    ctx.pod_ctl.update_pod_set(meta.id, meta.version.pod_set, pod_set_updates, t_id, ts)
    ctx.pod_ctl.commit_transaction(t_id)

    return new_vm.spec.vmagent_version


def run_update_labels(ctx, meta, spec):
    """

    :type ctx: infra.qyp.vmproxy.src.web.app.Ctx
    :type meta: infra.qyp.proto_lib.vmset_pb2.VMMeta
    :type spec: infra.qyp.proto_lib.vmset_pb2.VMSpec
    """
    pod_updates = {}
    pod_set_updates = {}
    new_labels = yputil.make_qyp_labels_dict(spec)

    for key, value in new_labels.iteritems():
        pod_updates['/labels/{}'.format(key)] = yson.dumps(value)
        pod_set_updates['/labels/{}'.format(key)] = yson.dumps(value)

    # commit
    t_id, ts = ctx.pod_ctl.start_transaction()
    ctx.pod_ctl.update_pod(meta.id, meta.version.pod, pod_updates, t_id, ts)
    ctx.pod_ctl.update_pod_set(meta.id, meta.version.pod_set, pod_set_updates, t_id, ts)
    ctx.pod_ctl.commit_transaction(t_id)


def run(meta, spec, ctx, login, update_vmagent, update_labels=False, config=None, force_update_vmagent=False):
    """
    :type meta: vmset_pb2.VMMeta
    :type spec: vmset_pb2.VMSpec
    :type ctx: vmproxy.web.app.Ctx
    :type login: str
    :type config: vmagent_pb2.VMConfig | None
    :type update_vmagent: bool
    :type update_labels: bool
    :type force_update_vmagent: bool
    """
    if not meta.id:
        raise ValueError('"meta.id" field not set')
    pod_pb = ctx.pod_ctl.get_pod(meta.id)
    old_vm = yputil.cast_pod_to_vm(
        pod_pb=pod_pb,
        pod_set_pb=ctx.pod_ctl.get_pod_set(meta.id)
    )
    vmagent_version = semantic_version.Version.coerce(old_vm.spec.vmagent_version)
    vmagent_0_28 = semantic_version.Version.coerce('0.28')
    vmagent_0_26 = semantic_version.Version.coerce('0.26')

    update_labels = update_labels and ctx.sec_policy.is_root(login)
    if not update_labels:
        # copy current labels from
        spec.ClearField('labels')
        spec.labels.update(old_vm.spec.labels)

    if update_labels:
        run_update_labels(ctx, meta, spec)
    elif update_vmagent:
        force_update_vmagent = force_update_vmagent or login in (
            'mocksoul',  # qdm server
            'robot-vmagent-rtc'  # sandbox eviction task
        )
        run_update_vmagent(
            ctx=ctx,
            pod_pb=pod_pb,
            old_vm=old_vm,
            spec=spec,
            meta=meta,
            login=login,
            force=force_update_vmagent
        )
    elif vmagent_version >= vmagent_0_28:
        fill_vm_spec_from_config_if_exists(spec, config)
        validation.validate_allocate_request(ctx, meta, spec, login)
        validation.validate_update_personal_resource_fit(ctx, spec, old_vm.spec, login)
        run_for_vmagent_ver_gte_0_28(
            ctx=ctx,
            meta=meta,
            spec=spec,
            pod_pb=pod_pb,
            old_vm=old_vm,
            login=login,
        )
    elif vmagent_0_26 <= vmagent_version < vmagent_0_28:
        validation.validate_allocate_request(ctx, meta, spec, login)
        validation.validate_update_personal_resource_fit(ctx, spec, old_vm.spec, login)
        run_for_vmagent_ver_gte_0_26_and_lt_0_28(
            ctx=ctx,
            meta=meta,
            spec=spec,
            pod_pb=pod_pb,
            old_vm=old_vm,
            login=login,
            config=config,
            vmagent_version=vmagent_version,
        )
    else:
        validation.validate_allocate_request(ctx, meta, spec, login)
        validation.validate_update_personal_resource_fit(ctx, spec, old_vm.spec, login)
        run_for_vmagent_ver_lt_0_26(
            meta=meta,
            spec=spec,
            ctx=ctx,
            login=login,
            old_vm=old_vm
        )
