import logging
import re
from collections import defaultdict

import semantic_version
from sepelib.core import config

from infra.qyp.proto_lib import vmset_pb2
from infra.qyp.vmproxy.src import errors
from infra.qyp.vmproxy.src.action import config as config_action
from infra.qyp.vmproxy.src.lib.yp import yputil

DEV_SEMGENT_NAME = 'dev'
DEFAULT_SEGMENT_NAME = 'default'
GPU_SEGMENT_NAME = 'gpu-dev'
ARM64_SEGMENT_NAME = 'dev-arm64'
MAX_CPU_GUARANTEE = 8000
MAX_LOGINS_COUNT = 100
MAX_ID_LENGTH = 30
TMP_ACCOUNT = 'tmp'
DEFAULT_SEGMENT_MAX_VOLUMES_SIZE = 1024 ** 4

VALID_RESOURCE_URL_PREFIXES_DEFAULT = ['http://', 'https://', 'rbtorrent:', 'qdm:']
log = logging.getLogger(__name__)


def validate_volumes_before_enabled_multi_storage(volumes):
    """

    :param volumes: list[infra.qyp.proto_lib.vmset_pb2.Volume]
    :return:
    """
    if len(volumes) > 1:
        raise ValueError('Can only set one volume')
    if volumes[0].name != yputil.MAIN_VOLUME_NAME:
        raise ValueError('Can only set one volume with name {}'.format(yputil.MAIN_VOLUME_NAME))
    storage_class = volumes[0].storage_class
    if storage_class and storage_class not in ('hdd', 'ssd'):
        msg = 'Volume storage_class should be one of (hdd, ssd), got {}'
        raise ValueError(msg.format(volumes[0].storage_class))


def validate_volume(volume_spec, volume_index):
    """

    :type volume_index: int
    :type volume_spec: infra.qyp.proto_lib.vmset_pb2.Volume
    :rtype: bool
    :raises: [ValueError]
    """
    is_main_volume = volume_spec.name == yputil.MAIN_VOLUME_NAME
    volume_spec.req_id = ''

    if volume_spec.storage_class not in ('hdd', 'ssd'):
        msg = 'Volume:{} storage_class should be one of (hdd, ssd), got {}'
        raise ValueError(msg.format(volume_spec.name, volume_spec.storage_class))

    if is_main_volume:
        if not volume_spec.resource_url:
            raise ValueError('Volume: {} resource_url for main volume required!'.format(volume_spec.name))
        volume_min_capacity = float(config.get_value('vmproxy.volumes.main_volume_min_capacity_gb')) * 1024.**3

        volume_spec.pod_mount_path = config.get_value('vmproxy.volumes.main_volume_pod_mount_path')
        volume_spec.vm_mount_path = config.get_value('vmproxy.volumes.main_volume_vm_mount_path')
    else:
        extra_volume_name_re = re.compile(config.get_value('vmproxy.volumes.extra_volume_name_re'))
        if not extra_volume_name_re.match(volume_spec.name):
            raise ValueError('Volume with index {} should match pattern "{}", got "{}"'.format(
                volume_index, extra_volume_name_re.pattern, volume_spec.name))
        extra_volume_name_black_list = config.get_value('vmproxy.volumes.extra_volume_name_black_list', [''])
        if volume_spec.name in extra_volume_name_black_list:
            raise ValueError('Volume with index:{} has name from black list: [{}]'.format(
                volume_index, ",".join(extra_volume_name_black_list)
            ))

        if not volume_spec.resource_url and volume_spec.image_type != vmset_pb2.Volume.RAW:
            raise ValueError('Volume:{} is empty and should have image type RAW'.format(volume_spec.name))

        volume_min_capacity = float(config.get_value('vmproxy.volumes.extra_volume_min_capacity_gb')) * 1024.**3
        volume_spec.pod_mount_path = "{}{}".format(
            config.get_value('vmproxy.volumes.extra_volume_pod_mount_path_prefix'), volume_spec.name)
        volume_spec.vm_mount_path = "{}{}".format(
            config.get_value('vmproxy.volumes.extra_volume_vm_mount_path_prefix'), volume_spec.name)
    valid_resource_url_prefixes = config.get_value('vmproxy.volumes.valid_resource_url_prefixes',
                                                   VALID_RESOURCE_URL_PREFIXES_DEFAULT)
    resource_url_valid = any([volume_spec.resource_url.startswith(p) for p in valid_resource_url_prefixes])

    if volume_spec.resource_url and not resource_url_valid:
        msg = "Volume:{} resource_url should start with [{}]".format(
            volume_spec.name, ",".join(valid_resource_url_prefixes))
        raise ValueError(msg)

    if volume_spec.capacity < volume_min_capacity:
        raise ValueError('Volume:{} capacity too small: ({}) < ({})'.format(
            volume_spec.name, volume_spec.capacity, int(volume_min_capacity)))


def validate_volumes(ctx, meta, spec):
    """
    :type ctx: vmproxy.web.app.Ctx
    :type meta: vmset_pb2.VMMeta
    :type spec: vmset_pb2.VMSpec
    """
    qemu_volumes = spec.qemu.volumes
    vmagent_version_raw = spec.vmagent_version or config.get_value('vmproxy.default_vmagent.version')

    if not qemu_volumes:
        raise ValueError('"spec.qemu.volumes" field not set')

    vmagent_version = semantic_version.Version.coerce(str(vmagent_version_raw))
    if vmagent_version < config_action.MULTI_STORAGE_VMAGENT_VERSION:
        validate_volumes_before_enabled_multi_storage(qemu_volumes)
    else:
        if qemu_volumes[0].name != yputil.MAIN_VOLUME_NAME:
            raise ValueError('Can only set one volume with name {}'.format(yputil.MAIN_VOLUME_NAME))

        volume_names = []
        max_extra_volumes_count = config.get_value('vmproxy.volumes.extra_volumes_max_count', 20)
        if len(qemu_volumes) - 1 > max_extra_volumes_count:
            raise ValueError('Extra volumes count should be less then: {}, current: {}'.format(
                max_extra_volumes_count, len(qemu_volumes) - 1))

        empty_guarantee_volumes = []
        total_volumes_size = 0
        for i, qemu_volume in enumerate(qemu_volumes):
            validate_volume(qemu_volume, i)
            volume_names.append(qemu_volume.name)
            storage_class_guarantee = spec.qemu.io_guarantees_per_storage.get(qemu_volume.storage_class, 0)
            if not storage_class_guarantee:
                empty_guarantee_volumes.append(qemu_volume.name)
            total_volumes_size += qemu_volume.capacity

        if len(list(set(volume_names))) < len(qemu_volumes):
            raise ValueError('Volume Names should be unique, got: {}'.format(volume_names))
        if not spec.qemu.forced_node_id and len(empty_guarantee_volumes):
            raise ValueError('IO guarantee not set for volumes {}'.format(empty_guarantee_volumes))
        if not spec.qemu.forced_node_id and spec.qemu.node_segment == DEFAULT_SEGMENT_NAME \
            and total_volumes_size > DEFAULT_SEGMENT_MAX_VOLUMES_SIZE:
            # https://st.yandex-team.ru/QEMUKVM-1479
            clusters_list = config.get_value('vmproxy.clusters_with_limited_volumes_size', None) or []
            if ctx.pod_ctl.yp_cluster in clusters_list:
                msg = 'Maximum total volumes size allowed in default segment is 1Tb, requested {} bytes'
                raise ValueError(msg.format(total_volumes_size))


def validate_allocate_request(ctx, meta, spec, login):
    """
    :type ctx: vmproxy.web.app.Ctx
    :type meta: vmset_pb2.VMMeta
    :type spec: vmset_pb2.VMSpec
    :type login: str
    :rtype: None
    """

    if not meta.id:
        raise ValueError('"meta.id" field not set')
    if len(meta.id) > MAX_ID_LENGTH:
        raise ValueError('id length should be less than {}, got {}'.format(MAX_ID_LENGTH, len(meta.id)))
    if not meta.auth.owners.logins and not meta.auth.owners.group_ids:
        raise ValueError('Authorization fields not set')
    logins_count = len(meta.auth.owners.logins)
    if logins_count > MAX_LOGINS_COUNT:
        raise ValueError('Too much literal owners, maximum {}, given {}'.format(MAX_LOGINS_COUNT, logins_count))
    root_users = set(config.get_value('vmproxy.root_users'))
    if len(spec.scheduling.hints) and login not in root_users:
        raise ValueError('Only root user can set scheduling hints')
    # uncomment after https://st.yandex-team.ru/QEMUKVM-1571
    # if spec.scheduling.node_filter and login not in root_users:
    #    raise ValueError('Only root user can set node_filter')

    # check correct of logins
    logins = meta.auth.owners.logins
    list_of_incorrect = []
    for name in logins:
        if re.match(r'^[\w-]+$', name) is None:
            list_of_incorrect.append(name)

    if len(list_of_incorrect) != 0:
        raise ValueError("incorrect meta.auth.owners.logins: " + ' '.join(list_of_incorrect))

    if spec.type == vmset_pb2.VMSpec.QEMU_VM:
        if not spec.qemu.network_id:
            raise ValueError('"spec.qemu.network_id" field not set')

        if not spec.qemu.node_segment:
            raise ValueError('"spec.qemu.node_segment" field not set')
        allowed_node_segments = config.get_value('vmproxy.node_segment')
        if spec.qemu.node_segment not in allowed_node_segments:
            msg = '"spec.qemu.node_segment" should be one of {}, got {}'
            raise ValueError(msg.format(allowed_node_segments, spec.qemu.node_segment))
        if not spec.qemu.HasField('resource_requests'):
            raise ValueError('"spec.qemu.resource_requests" field not set')
        if spec.qemu.resource_requests.vcpu_limit <= 0:
            raise ValueError('"spec.qemu.resource_requests.vcpu_limit" should be greater than zero')
        if (spec.qemu.node_segment in (DEFAULT_SEGMENT_NAME, ARM64_SEGMENT_NAME)
                and spec.qemu.resource_requests.vcpu_guarantee != spec.qemu.resource_requests.vcpu_limit):
            raise ValueError('Cpu guarantee ({}) should be equal cpu limit ({}) in {} segment'.format(
                spec.qemu.resource_requests.vcpu_guarantee,
                spec.qemu.resource_requests.vcpu_limit,
                spec.qemu.node_segment
            ))
        if spec.qemu.resource_requests.vcpu_guarantee > spec.qemu.resource_requests.vcpu_limit:
            raise ValueError('Cpu guarantee should be less than cpu limit')
        if spec.qemu.resource_requests.memory_guarantee != spec.qemu.resource_requests.memory_limit:
            raise ValueError('Memory guarantee should be equal memory limit')
        if ((not spec.account_id or spec.account_id == TMP_ACCOUNT)
                and spec.qemu.resource_requests.vcpu_guarantee > MAX_CPU_GUARANTEE):
            msg = 'Maximum for "spec.qemu.resource_requests.vcpu_guarantee" in temporary account is {}, given {}'
            raise ValueError(msg.format(MAX_CPU_GUARANTEE, spec.qemu.resource_requests.vcpu_guarantee))

        if spec.qemu.ip4_address_pool_id and (spec.qemu.enable_internet or spec.qemu.use_nat64):
            raise ValueError('"ip4_address_pool_id" mutually excludes fields "enable_internet" and "use_nat64"')
        if spec.qemu.enable_internet and spec.qemu.use_nat64:
            raise ValueError('"enable_internet" and "use_nat64" fields are mutually exclusive')
        if spec.qemu.node_segment == GPU_SEGMENT_NAME and not spec.qemu.HasField('gpu_request'):
            raise ValueError('At least 1 GPU should be requested in {} segment'.format(GPU_SEGMENT_NAME))
        vcpu_max = config.get_value('vmproxy.allocation_request_constraints.vcpu_limit', None)
        if vcpu_max and spec.qemu.resource_requests.vcpu_limit > vcpu_max:
            msg = 'Cpu limit {} exceeds maximum allowed value {}'.format(spec.qemu.resource_requests.vcpu_limit,
                                                                         vcpu_max)
            raise ValueError(msg)
        validate_volumes(ctx, meta, spec)
    else:
        raise ValueError('Unknown VM type {}'.format(spec.type))


def validate_macro(ctx, network_id, login):
    if not ctx.pod_ctl.check_use_macro_permission(network_id, login):
        raise errors.AuthorizationError('User {} must have access to {} macro'.format(login, network_id))


def validate_qdm_backup_spec(ctx, backup_spec, login):
    have_owners = bool(backup_spec.vmspec.meta.id)
    if not ctx.sec_policy.is_root(login) and have_owners \
            and (not ctx.sec_policy.is_allowed(login,
                                               backup_spec.vmspec.meta.auth.owners.logins,
                                               backup_spec.vmspec.meta.auth.owners.group_ids)):
        raise errors.AuthorizationError('Attempt to restore VM backup by person-non-owner')


def validate_personal_resource_fit(ctx, spec, login, ignore_disk_tax):
    """
    :type ctx: infra.qyp.vmproxy.src.web.app.Ctx
    :type spec: infra.qyp.proto_lib.vmset_pb2.VMSpec
    :type login: str
    :type ignore_disk_tax: bool
    :rtype: None
    """
    if spec.account_id not in ctx.personal_accounts:
        return
    accounts_resp = ctx.account_manager_client.request_account_data(login, spec.account_id)
    segment = spec.qemu.node_segment
    limits = get_personal_account_limits(accounts_resp, segment, spec.account_id)
    usage = get_personal_account_usage(accounts_resp, segment, spec.account_id)

    cpu_available = limits.cpu - usage.cpu
    if spec.qemu.resource_requests.vcpu_guarantee > cpu_available:
        raise errors.ValidationError(
            'CPU available for personal usage: {}; '
            'requested: {}'.format(cpu_available, spec.qemu.resource_requests.vcpu_guarantee)
        )

    mem_available = limits.mem - usage.mem
    if spec.qemu.resource_requests.memory_guarantee > mem_available:
        raise errors.ValidationError(
            'Memory available for personal usage: {}; '
            'requested: {}'.format(mem_available, spec.qemu.resource_requests.memory_guarantee)
        )

    storage_requests = defaultdict(int)
    for v in spec.qemu.volumes:
        storage_requests[v.storage_class] += v.capacity
        if v.name == yputil.MAIN_VOLUME_NAME and not ignore_disk_tax:
            root_quota = config.get_value('vmproxy.default_porto_layer.root_quota', default=0)
            workdir_quota = config.get_value('vmproxy.default_porto_layer.workdir_quota', default=0)
            storage_requests[v.storage_class] += root_quota + workdir_quota

    for storage, capacity in storage_requests.iteritems():
        available_space = limits.disk_per_storage[storage] - usage.disk_per_storage[storage]
        if capacity > available_space:
            raise errors.ValidationError(
                'Disk space available for {} for personal usage: {}; '
                'requested: {}'.format(storage, available_space, capacity)
            )

    if spec.qemu.gpu_request.capacity:
        capacity, model = spec.qemu.gpu_request.capacity, spec.qemu.gpu_request.model
        available_gpu = limits.gpu_per_model[model] - usage.gpu_per_model[model]
        if capacity > available_gpu:
            raise errors.ValidationError(
                'GPU available for model {} for personal usage: {}. requested: {}'.format(model, available_gpu, capacity)
            )

    enable_inernet = spec.qemu.enable_internet or bool(spec.qemu.ip4_address_pool_id)
    if enable_inernet and not (limits.internet_address - usage.internet_address) > 0:
        raise errors.ValidationError(
            'No internet address is available for personal usage in '
            'service: {} to enable internet option'.format(spec.account_id)
        )
    for st in spec.qemu.io_guarantees_per_storage:
        available_guarantee = limits.io_guarantees_per_storage[st] - usage.io_guarantees_per_storage[st]
        requested_quarantee = spec.qemu.io_guarantees_per_storage[st]
        if limits.io_guarantees_per_storage[st]:  # 0 - unlimited
            if requested_quarantee > available_guarantee:
                raise errors.ValidationError(
                    'Disk bandwidth available for {} for personal usage: {}. requested: {}'.format(
                        st, available_guarantee, requested_quarantee)
                )


def validate_update_personal_resource_fit(ctx, spec, old_spec, login):
    """
    :type ctx: infra.qyp.vmproxy.src.web.app.Ctx
    :type spec: infra.qyp.proto_lib.vmset_pb2.VMSpec
    :type old_spec: infra.qyp.proto_lib.vmset_pb2.VMSpec
    :type login: str
    :rtype: None
    """
    if spec.account_id not in ctx.personal_accounts:
        return
    if spec.qemu.node_segment != old_spec.qemu.node_segment:
        raise errors.ValidationError('Moving VM between segments is not allowed')
    if spec.account_id != old_spec.account_id:
        validate_personal_resource_fit(ctx, spec, login, ignore_disk_tax=False)
    else:
        diff = vmset_pb2.VMSpec()
        diff.CopyFrom(spec)
        if spec.qemu.resource_requests.vcpu_guarantee > old_spec.qemu.resource_requests.vcpu_guarantee:
            diff.qemu.resource_requests.vcpu_guarantee -= old_spec.qemu.resource_requests.vcpu_guarantee
        else:
            diff.qemu.resource_requests.vcpu_guarantee = 0

        if spec.qemu.resource_requests.memory_guarantee > old_spec.qemu.resource_requests.memory_guarantee:
            diff.qemu.resource_requests.memory_guarantee -= old_spec.qemu.resource_requests.memory_guarantee
        else:
            diff.qemu.resource_requests.memory_guarantee = 0

        diff.qemu.ClearField('volumes')
        volumes_by_storage, old_volumes_by_storage = defaultdict(int), defaultdict(int)
        main_volume_storage_class, old_main_volume_storage_class = '', ''
        for volume in spec.qemu.volumes:
            volumes_by_storage[volume.storage_class] += volume.capacity
            if volume.name == yputil.MAIN_VOLUME_NAME:
                main_volume_storage_class = volume.storage_class
        for volume in old_spec.qemu.volumes:
            old_volumes_by_storage[volume.storage_class] += volume.capacity
            if volume.name == yputil.MAIN_VOLUME_NAME:
                old_main_volume_storage_class = volume.storage_class

        for storage, capacity in volumes_by_storage.iteritems():
            v = diff.qemu.volumes.add()
            if storage == main_volume_storage_class:
                v.name = yputil.MAIN_VOLUME_NAME
            else:
                v.name = 'volume-for-{}'.format(storage)
            v.storage_class = storage
            v.capacity = max(capacity - old_volumes_by_storage[storage], 0)

        if diff.qemu.gpu_request.model == old_spec.qemu.gpu_request.model:
            diff.qemu.gpu_request.capacity = max(0, spec.qemu.gpu_request.capacity - old_spec.qemu.gpu_request.capacity)

        old_spec_enable_inernet = old_spec.qemu.enable_internet or bool(old_spec.qemu.ip4_address_pool_id)
        if old_spec_enable_inernet:
            diff.qemu.enable_internet = False
            diff.qemu.ip4_address_pool_id = ''

        for st in old_spec.qemu.io_guarantees_per_storage:
            diff.qemu.io_guarantees_per_storage[st] = max(
                0, spec.qemu.io_guarantees_per_storage[st] - old_spec.qemu.io_guarantees_per_storage[st]
            )

        ignore_disk_tax = main_volume_storage_class == old_main_volume_storage_class
        validate_personal_resource_fit(ctx, diff, login, ignore_disk_tax)


def get_personal_account_limits(accounts_resp, segment, account_id):
    """
    :type accounts_resp: infra.qyp.proto_lib.accounts_api_pb2.ListUserAccountsResponse
    :type segment: str
    :type account_id: str
    :rtype: infra.qyp.proto_lib.vmset_pb2.ResourceInfo
    """
    personal_limits = vmset_pb2.ResourceInfo()
    for accounts_by_cluster in accounts_resp.accounts_by_cluster:
        for acc in accounts_by_cluster.accounts:
            if acc.id == account_id:
                personal_limits.CopyFrom(acc.personal.limits.per_segment[segment])
                return personal_limits

    return personal_limits


def get_personal_account_usage(accounts_resp, segment, account_id):
    """
    :type accounts_resp: infra.qyp.proto_lib.accounts_api_pb2.ListUserAccountsResponse
    :type segment: str
    :type account_id: str
    :rtype: infra.qyp.proto_lib.vmset_pb2.ResourceInfo
    """
    personal_usage = vmset_pb2.ResourceInfo()
    for pers_summary in accounts_resp.personal_summary:
        if pers_summary.account_id == account_id:
            personal_usage.CopyFrom(pers_summary.total_usage.per_segment[segment])
            break
    return personal_usage
