from __future__ import unicode_literals

import warnings
import logging
import socket

import inject
from sepelib.core import config

from infra.swatlib import webserver, sandbox
from infra.swatlib.auth import abc
from infra.swatlib.auth import passport
from infra.swatlib.auth import staff
from infra.swatlib.auth import oauth
from infra.swatlib.auth import tvm
from infra.swatlib.rpc import authentication as rpc_auth
from infra.swatlib.simulate_behavior import SimulateBehaviorExt
from infra.qyp.vmproxy.src import pod_controller, security_policy
from infra.qyp.vmproxy.src.lib import nanny_client, vmagent_client, rt, qdm_client
from infra.qyp.vmproxy.src.web import app as web_app


VERSION = 'development_version'


class Application(object):
    name = 'api_server'

    def init_web_application(self, debug=False):
        default_yp_cluster = config.get_value('yp.default_cluster')
        sec_policy = security_policy.SecurityPolicy().from_sepelib_config()
        personal_quotas = {item['account_id']: item for item in config.get_value('personal_quotas', default=[])}
        personal_accounts = set(config.get_value('personal_accounts', default=[]))
        ctx = web_app.Ctx(
            nanny_client=nanny_client.NannyClient(**config.get_value('nanny')),
            vmagent_client=vmagent_client.VmagentClient(self.tvm_context),
            sec_policy=sec_policy,
            tvm_context=self.tvm_context,
            pod_ctl_factory=pod_controller.PodControllerFactory(default_yp_cluster, sec_policy),
            personal_quotas_dict=personal_quotas,
            personal_accounts=personal_accounts,
            vmproxy_config=self.get_vmproxy_config(),
            qdm_client=qdm_client.QDMClient(default_yp_cluster, self.tvm_context, **config.get_value('qdm', {}))
        )
        app = web_app.create_app(ctx, socket.gethostname(), version=VERSION, debug=debug)
        return app

    def get_vmproxy_config(self):
        def fill_config_data(path, src, dst):
            key = path[0]
            if key not in src:
                return
            value = src[key]
            if len(path) == 1:
                dst[key] = value
                return
            if isinstance(value, dict):
                if key not in dst:
                    dst[key] = {}
                fill_config_data(path[1:], value, dst[key])
                return
            elif isinstance(value, list):
                if key not in dst:
                    dst[key] = []
                    for i in value:
                        if isinstance(i, dict):
                            x = {}
                            dst[key].append(x)
                            fill_config_data(path[1:], i, x)
                            continue
                        dst[key].append(i)
                else:
                    for a, b in zip(value, dst[key]):
                        fill_config_data(path[1:], a, b)
            else:
                raise Exception('Error while parsing config file: expected list or dict, got {}'.format(type(value)))

        white_list = (
            'yp.default_cluster',
            'yp.clusters.cluster',
            'yp.clusters.url',
            'yp.clusters.enable_https',
            'images',
            'vmproxy.default_vmagent.version',
            'vmproxy.default_vmagent.url',
            'vmproxy.default_porto_layer.root_quota',
            'vmproxy.default_porto_layer.workdir_quota',
            'vmproxy.minimal_vmagent_for_update.version',
            'vmproxy.allocation_request_constraints.vcpu_limit',
            'personal_quotas',
            'vmproxy.volumes',
            'gpu_models',
            'gpu_settings',
        )
        result = {}
        for request in white_list:
            fill_config_data(request.strip().split('.'), config.get(), result)
        return result

    def _get_tvm_context(self):
        secret_file = config.get_value('tvm_context.secret_file')
        with open(secret_file) as f:
            secret = f.read()
        return tvm.TvmClient(
            client_id=config.get_value('tvm_context.client_id'),
            secret=secret,
            api_url=config.get_value('tvm_context.api_url')
        )

    def __init__(self, instance_id):
        self.instance_id = instance_id
        self.log = logging.getLogger(self.name)
        self.tvm_context = self._get_tvm_context()
        webapp = self.init_web_application()

        self.web = webserver.WebServer(config.get(), webapp, version=VERSION)
        self.simulate_behavior_ext = SimulateBehaviorExt(self.web, config.get_value('simulate_behavior', {}), webapp)
        inject.clear_and_configure(self.configure_injector)

    def configure_injector(self, binder):
        passport_client = passport.TvmPassportClient.from_config(
            config.get_value('passport'),
            tvm_client=self.tvm_context,
        )
        oauth_client = oauth.OAuth.from_config(config.get_value('oauth'))

        binder.bind(oauth.IOAuth, oauth_client)
        binder.bind(passport.IPassportClient, passport_client)
        binder.bind(rpc_auth.IRpcAuthenticator,
                    rpc_auth.CachingPassportAuthenticator(oauth_client=oauth_client,
                                                          passport_client=passport_client,
                                                          is_auth_disabled=not config.get_value('run.auth'),
                                                          force_return_user=config.get_value('run.force_return_user'))
                    )

        binder.bind(staff.IStaffClient,
                    staff.StaffClient.from_config(config.get_value('staff')))
        binder.bind(abc.IAbcClient,
                    abc.AbcClient.from_config(config.get_value('abc')))
        binder.bind(rt.IRtClient, rt.RtClient())
        binder.bind(
            sandbox.ISandboxClient,
            sandbox.SandboxClient.from_config(config.get_value('sandbox'))
        )

    def setup_environment(self):
        import urllib3
        from urllib3.connectionpool import ConnectionPool
        from gevent.queue import LifoQueue
        from yt.packages.requests.packages import urllib3 as yt_urllib3

        ConnectionPool.QueueCls = LifoQueue
        logging.getLogger('urllib3.connectionpool').setLevel(logging.ERROR)
        logging.getLogger('yt.packages.urllib3.connectionpool').setLevel(logging.ERROR)
        # Disable warnings about not validating SSL certificates
        warnings.simplefilter('ignore', urllib3.exceptions.SecurityWarning)
        # TODO: check this
        # warnings.simplefilter('ignore', yt_urllib3.exceptions.SecurityWarning)

    def run(self):
        self.log.info('=' * 30)
        self.log.info('config -> run.auth: {}'.format(config.get_value('run.auth')))
        self.log.info('config -> run.debug: {}'.format(config.get_value('run.debug')))
        self.log.info('config -> run.force_return_user: {}'.format(config.get_value('run.force_return_user')))
        self.log.info('config -> log.filepath: {}'.format(config.get_value('log.filepath')))
        self.log.info('config -> web.access_log.filepath: {}'.format(config.get_value('web.access_log.filepath')))
        self.log.info('config -> yp.default_cluster: {}'.format(config.get_value('yp.default_cluster')))
        self.log.info('config -> passport.blackbox_url: {}'.format(config.get_value('passport.blackbox_url')))
        self.log.info('config -> passport.blackbox_auth_url: {}'.format(config.get_value('passport.blackbox_auth_url')))
        self.log.info('ext -> simulate behavior: enabled: {}, url: {}'.format(self.simulate_behavior_ext.enabled,
                                                                              self.simulate_behavior_ext.url))
        self.log.info('starting {}...'.format(self.name))
        self.setup_environment()
        # Block here
        self.web.run()

    def stop(self):
        self.log.info('stopping {}...'.format(self.name))
        self.web.stop()
        self.log.info('=' * 30)
