import collections
import itertools

import cachetools
import gevent
import inject
from infra.swatlib.auth import abc

service_roles_cache = cachetools.TTLCache(maxsize=1000, ttl=60 * 60)
QYP_ROLE_CODE = 'qyp_user'
ADMIN_ROLE_SCOPE_SLUG = 'administration'
MANAGEMENT_ROLE_SCOPE_SLUG = 'services_management'
TMP_ACCOUNT = 'tmp'
CORRECT_ACCOUNT_PREFIX = 'abc:service:'


def filter_accounts_by_roles(account_ids, login, use_cache):
    """
    1. Find persons from account_ids who has QYP role
    2. Filter only accounts which either has person with qyp role or has no qyp role persons

    See:
    * QEMUKVM-545
    * ABC-6587
    * https://wiki.yandex-team.ru/qyp/personal_quota/#workflow

    :type account_ids: list[str]
    :type login: str
    :type use_cache: bool
    :rtype: list[str]
    """
    service_set = set()
    for account_id in account_ids:
        if account_id.startswith(CORRECT_ACCOUNT_PREFIX):
            service_id = account_id.split('abc:service:')[-1]
            service_set.add(service_id)

    if use_cache:
        services_to_fetch = service_set.difference(set(service_roles_cache.keys()))
    else:
        services_to_fetch = service_set

    if services_to_fetch:
        service_roles_dict = fetch_qyp_role_persons(services_to_fetch)
        for service_id in services_to_fetch:
            service_roles_cache[service_id] = service_roles_dict.get(service_id) or set()

    result = []
    for account_id in account_ids:
        service_id = account_id.split('abc:service:')[-1]
        logins = service_roles_cache.get(service_id, [])
        if not logins or login in logins:
            result.append(account_id)
    return result


def get_service_qyp_persons(account_id, use_cache):
    """
    :type account_id: str
    :type use_cache: bool
    :rtype: set[str]
    """
    if account_id == TMP_ACCOUNT:
        return []
    service_id = account_id.split('abc:service:')[-1]
    if not use_cache or service_id not in service_roles_cache:
        service_roles_cache[service_id] = fetch_qyp_role_persons([service_id]).get(service_id) or set()
    return service_roles_cache[service_id]


def grouper(iterable, n, fillvalue=None):
    """
    Collect data into fixed-length chunks or blocks
    # grouper('ABCDEFG', 3, 'x') --> ABC DEF Gxx"
    """
    args = [iter(iterable)] * n
    return itertools.izip_longest(*args, fillvalue=fillvalue)


def fetch_qyp_role_persons(service_ids):
    """
    :type service_ids: collections.Iterable[str]
    :rtype: dict[str, set[str]]
    """
    abc_client = inject.instance(abc.IAbcClient)
    service_roles_dict = collections.defaultdict(set)
    for service_batch in grouper(service_ids, 100):
        spec = {'service': service_batch, 'role__code': QYP_ROLE_CODE, 'fields': 'service.id,person.login'}
        for resp in abc_client.list_members_iter(spec=spec):
            for item in resp['results']:
                service_roles_dict[str(item['service']['id'])].add(str(item['person']['login']))
        gevent.sleep(0.1)
    return service_roles_dict


def list_user_services_from_abc(login):
    """
    :type login: str
    :rtype: list[str]
    """
    abc_client = inject.instance(abc.IAbcClient)
    services_dict = abc_client.list_user_services(login)
    return services_dict.keys()
