import json
from dateutil import parser
from infra.swatlib.metrics import InstrumentedSession
from infra.qyp.proto_lib import vmset_api_pb2
from infra.swatlib.rpc import exceptions
from sepelib.core import config


class HeartBeatClient:
    DEFAULT_HEARTBEAT_UI_URL = "https://heartbeat.yandex-team.ru"

    def __init__(self, url=None):
        self._url = url or self.DEFAULT_HEARTBEAT_UI_URL
        self._session = InstrumentedSession('heartbeat_ui')
        self.system_users = set(config.get_value('vmproxy.vm_system_users', []))
        self.entries_limit = 10

    def request_usage(self, vm_fqdn):
        """
        :type vm_fqdn: str
        :rtype: dict
        """
        url = self._url + "/vm_usage_stats_json/{}".format(vm_fqdn)
        resp = self._session.request('GET', url)
        if resp.status_code == 404:
            raise exceptions.NotFoundError('VM with fqdn: {} was not found'.format(vm_fqdn))
        resp.raise_for_status()
        return resp.json()

    def get_vm_usage_stats(self, vm_fqdn):
        """
        :type vm_fqdn: str
        :rtype: vmset_api_pb2.GetVmUsageResponse
        """
        data = self.request_usage(vm_fqdn)
        vm_usage = self.heartbeat_resp_to_proto(data)
        return vm_usage

    def heartbeat_resp_to_proto(self, data):
        """
        :type data: dict
        :rtype: infra.qyp.proto_lib.vmset_api.GetVmUsageResponse
        """
        vm_usage = vmset_api_pb2.GetVmUsageResponse()
        self._fill_dutop(vm_usage, data)
        self._fill_last(vm_usage, data)
        self._fill_who(vm_usage, data)

        return vm_usage

    def _fill_dutop(self, vm_usage, data):
        """
        :type vm_usage: infra.qyp.proto_lib.vmset_api.GetVmUsageResponse
        :type data: dict
        :rtype: None
        """
        dutop = data.get('dutop') or {}
        du_top_data = json.loads(dutop.get('du_top_data', "{}"))
        last_update = dutop.get('last_update')
        if last_update:
            t = parser.parse(last_update).replace(tzinfo=None)
            vm_usage.dutop.last_update_time.FromDatetime(t)
        for user, d in sorted(du_top_data.items(), key=lambda s: s[1].get('total_size', 0), reverse=True):
            if user in self.system_users:
                continue
            dutop_entry = vm_usage.dutop.entries.add()
            dutop_entry.user_name = user
            dutop_entry.size = d.get('total_size', 0)
            top_dir_sizes = d.get('top_dir_sizes', {})
            for dir_name, size in sorted(top_dir_sizes.items(), key=lambda s: s[1], reverse=True):
                top_dir = dutop_entry.top_dirs.add()
                top_dir.size = size
                top_dir.dir_name = dir_name

    def _fill_last(self, vm_usage, data):
        """
        :type vm_usage: infra.qyp.proto_lib.vmset_api.GetVmUsageResponse
        :type data: dict
        :rtype: None
        """
        last = data.get('last', {})
        sessions_seconds_length = last.get('user_sessions_seconds_length', {})
        user_sessions = last.get('user_sessions', {})
        last_update = last.get('last_update')
        from_time = last.get('wtmp_begin_dt')
        if last_update:
            t = parser.parse(last_update).replace(tzinfo=None)
            vm_usage.last.last_update_time.FromDatetime(t)
        if from_time:
            t = parser.parse(from_time).replace(tzinfo=None)
            vm_usage.last.from_time.FromDatetime(t)
        for u, d in user_sessions.items():
            t = parser.parse(d).replace(tzinfo=None)
            user_sessions[u] = t

        counter = 0
        for user, d in sorted(user_sessions.items(), key=lambda s: s[1], reverse=True):
            if user in self.system_users:
                continue
            last_entry = vm_usage.last.entries.add()
            last_entry.user_name = user
            last_entry.sessions_seconds_length = sessions_seconds_length.get(user, '')
            last_entry.session_end.FromDatetime(d)
            counter += 1
            if counter == self.entries_limit:
                break

    def _fill_who(self, vm_usage, data):
        """
        :type vm_usage: infra.qyp.proto_lib.vmset_api.GetVmUsageResponse
        :type data: dict
        :rtype: None
        """
        who = data.get('who', {})
        user_actions = who.get('user_actions', {})
        user_sessions = who.get('user_sessions', {})
        last_update = who.get('last_update')
        from_time = who.get('initialised')
        if last_update:
            t = parser.parse(last_update).replace(tzinfo=None)
            vm_usage.who.last_update_time.FromDatetime(t)
        if from_time:
            t = parser.parse(from_time).replace(tzinfo=None)
            vm_usage.who.from_time.FromDatetime(t)

        for u, d in user_sessions.items():
            user_sessions[u] = parser.parse(d).replace(tzinfo=None)

        counter = 0
        for user, d in sorted(user_sessions.items(), key=lambda s: s[1], reverse=True):
            if user in self.system_users:
                continue
            who_entry = vm_usage.who.entries.add()
            who_entry.user_name = user
            who_entry.session_end.FromDatetime(d)
            who_entry.actions.extend(user_actions.get(user, []))
            counter += 1
            if counter == self.entries_limit:
                break
