import datetime
import json
import logging

from infra.qyp.proto_lib import vmset_pb2
from infra.qyp.proto_lib.qdm_pb2 import QDMBackupSpec

import cachetools
import inject
from google.protobuf.json_format import ParseDict
from infra.swatlib.auth import staff
from infra.swatlib.metrics import InstrumentedSession

staff_groups_cache = cachetools.TTLCache(maxsize=300, ttl=30 * 60)
log = logging.getLogger(__name__)


def get_staff_groups(login):
    groups = staff_groups_cache.get(login)
    if groups is None:
        staff_client = inject.instance(staff.IStaffClient)
        try:
            groups = staff.get_group_ids(staff_client, login)
        except Exception as e:
            # Ignore staff errors.
            log.warning('Staff error while fetching user groups: %s', str(e))
            groups = []
        else:
            staff_groups_cache[login] = groups
    return groups


def normalize_url(url):
    """

    :type url: str
    :rtype: str
    """
    if not url.startswith('qdm:'):
        raise ValueError('"res_id" should stats with qdm:')
    ind = url.find('/')
    if ind != -1:
        url = url[:ind]
    return url.replace('qdm:', '')


class QDMClient(object):
    DEFAULT_URL = 'https://qdm.yandex-team.ru'
    QDM_TVM_ID = '2015617'
    QDM_STATE_MAP = {
        'new': vmset_pb2.BackupStatus.PLANNED,
        'draft': vmset_pb2.BackupStatus.IN_PROGRESS,
        'active': vmset_pb2.BackupStatus.COMPLETED,
        'archive': vmset_pb2.BackupStatus.REMOVED,
    }
    QDM_ORIGIN_MAP = {
        'evoq': vmset_pb2.BackupSpec.EVOQ,
        'qdm': vmset_pb2.BackupSpec.USER,
        'hot': vmset_pb2.BackupSpec.HOT,
    }

    def __init__(self, yp_cluster, tvm_context, url=None):
        self.yp_cluster = yp_cluster.lower()
        self._tvm_context = tvm_context
        self._base_url = url.rstrip('/') if url else self.DEFAULT_URL

    @property
    def session(self):
        session = InstrumentedSession('qdm')
        session.headers['Content-Type'] = 'application/json'
        session.headers['Accept-Encoding'] = ''
        session.headers['X-Ya-Service-Ticket'] = self._tvm_context.ticket_to(dst=self.QDM_TVM_ID)
        return session

    def get_revision_info(self, res_id):
        """

        :type res_id: str
        :rtype: infra.qyp.proto_lib.qdm_pb2.QDMBackupSpec
        """
        res_id = normalize_url(res_id)
        resp = self.session.get(self._base_url + '/api/v1/revision_info/{res_id}?fmt=json'.format(res_id=res_id))
        if resp.status_code == 404:
            return None
        resp.raise_for_status()
        qdm_spec = QDMBackupSpec()
        backup_spec_data = resp.json()
        dt = datetime.datetime.fromtimestamp(backup_spec_data['create_ts'])
        backup_spec_data['create_ts'] = dt.isoformat("T") + "Z"
        ParseDict(backup_spec_data, qdm_spec, ignore_unknown_fields=True)
        return qdm_spec

    def backup_create(self, vm_id):
        """
        :type vm_id: str
        """
        data = {
            'vmid': vm_id,
            'dc': self.yp_cluster,
        }
        resp = self.session.post(self._base_url + '/api/v1/backup_create', data=json.dumps(data))
        resp.raise_for_status()

    def backup_list(self, vm_id):
        """
        :type vm_id: str
        :rtype: list[vmset_pb2.Backup]
        """
        resp = self.session.get(self._base_url + '/api/v1/backup_list?vmid={vm_id}&dc={dc}'.format(
            vm_id=vm_id,
            dc=self.yp_cluster,
        ))
        resp.raise_for_status()
        result = []
        for item in resp.json():
            backup = vmset_pb2.Backup()
            backup.meta.id = str(item['key'])
            backup.meta.creation_time.FromSeconds(item['create_ts'])
            backup.spec.type = vmset_pb2.BackupSpec.QDM
            backup.spec.vm_id = vm_id
            backup.spec.generation = item['rev_id']
            backup.status.state = self.QDM_STATE_MAP.get(item['state'])
            backup.status.url = 'qdm:' + item['key']
            result.append(backup)
        return result

    def backup_status(self, vm_id):
        """
        :type vm_id: str
        :rtype: vmset_pb2.Backup
        """
        resp = self.session.get(self._base_url + '/api/v1/backup_status?vmid={vm_id}&dc={dc}'.format(
            vm_id=vm_id,
            dc=self.yp_cluster,
        ))
        if resp.status_code == 404:
            return None
        resp.raise_for_status()
        resp = resp.json()
        backup = vmset_pb2.Backup()
        backup.status.state = vmset_pb2.BackupStatus.IN_PROGRESS
        backup.spec.origin = self.QDM_ORIGIN_MAP.get(resp['origin'])
        return backup

    def user_backup_list(self, query):
        """
        :type query: vmset_pb2.BackupFindQuery
        :rtype: list[vmset_pb2.Backup]
        """
        params = {
            'user': query.login,
            'groups': get_staff_groups(query.login),
        }
        resp = self.session.get(self._base_url + '/api/v1/user_backup_list', params=params)
        resp.raise_for_status()
        result = []
        for item in resp.json():
            backup = vmset_pb2.Backup()
            backup.meta.id = str(item['key'])
            backup.meta.creation_time.FromSeconds(item['create_ts'])
            backup.spec.type = vmset_pb2.BackupSpec.QDM
            backup.spec.vm_id = item['vm_id']
            backup.spec.generation = item['rev_id']
            backup.spec.origin = self.QDM_ORIGIN_MAP.get(item['origin'])
            backup.status.state = self.QDM_STATE_MAP.get(item['state'])
            backup.status.url = 'qdm:' + item['key']
            vm = vmset_pb2.VM()
            ParseDict(item['vmspec'], vm, ignore_unknown_fields=True)
            backup.spec.vm_meta.CopyFrom(vm.meta)
            backup.spec.vm_spec.CopyFrom(vm.spec)
            if backup.status.state == vmset_pb2.BackupStatus.COMPLETED:
                backup.spec.total_size = sum(file_info['s'] for file_info in item['filemap']['f'])
            result.append(backup)
        return result
