import yt_yson_bindings
import yp.data_model as data_model
from sepelib.core import config as sepelib_config
from yp_proto.yp.client.api.proto import object_service_pb2
from infra.qyp.vmproxy.src.lib.yp import yputil
from infra.swatlib.metrics import YpInstrumentedSession
from yp.client import YpClient as BaseYpClient
from yt import yson


class YpClient(object):
    def __init__(self, stub):
        """
        :type stub: yp.client.object_service_pb2_grpc.ObjectServiceStub
        """
        self.stub = stub

    def select_objects(self, query, object_type, selectors, timestamp=None, offset=0, limit=0):
        """
        :type query: str | NoneType
        :type object_type: int
        :type selectors: list[str]
        :type timestamp: int | NoneType
        :type limit: int
        :type offset: int
        :rtype: object_service_pb2.TRspSelectObjects
        """
        req = object_service_pb2.TReqSelectObjects()
        req.object_type = object_type
        if limit:
            req.limit.value = limit
        if offset:
            req.offset.value = offset
        if query:
            req.filter.query = query
        if timestamp:
            req.timestamp = timestamp
        req.selector.paths.extend(selectors)
        return self.stub.SelectObjects(req)

    def list_pods(self, query, timestamp=None, selectors=None, offset=0, limit=0):
        """
        :type query: str
        :type timestamp: int | NoneType
        :type selectors: list[str] | NoneType
        :type limit: int
        :type offset: int
        :rtype: object_service_pb2.TRspSelectObjects
        """
        selectors = selectors or ['']
        return self.select_objects(query, data_model.OT_POD, selectors, timestamp, offset=offset, limit=limit)

    def list_pod_sets(self, query, timestamp=None, selectors=None, offset=0, limit=0):
        """
        :type query: str | NoneType
        :type timestamp: int | NoneType
        :type selectors: list[str] | NoneType
        :type limit: int
        :type offset: int
        :rtype: object_service_pb2.TRspSelectObjects
        """
        selectors = selectors or ['']
        return self.select_objects(query, data_model.OT_POD_SET, selectors, timestamp, offset=offset, limit=limit)

    def list_accounts(self, query, timestamp=None, selectors=None):
        """
        :type query: str | NoneType
        :type timestamp: int | NoneType
        :type selectors: list[str] | NoneType
        :rtype: object_service_pb2.TRspSelectObjects
        """
        selectors = selectors or ['']
        return self.select_objects(query, data_model.OT_ACCOUNT, selectors, timestamp)

    def get_accounts(self, ids, selectors=None, timestamp=None):
        """
        :type ids: list[str]
        :type selectors: list[str] | NoneType
        :rtype: object_service_pb2.TRspGetObjects
        """
        selectors = selectors or ['']
        return self.get_objects(ids, data_model.OT_ACCOUNT, selectors=selectors, timestamp=timestamp)

    def list_groups(self, query, timestamp=None, selectors=None):
        """
        :type query: str | NoneType
        :type timestamp: int | NoneType
        :type selectors: list[str] | NoneType
        :rtype: object_service_pb2.TRspSelectObjects
        """
        selectors = selectors or ['']
        return self.select_objects(query, data_model.OT_GROUP, selectors, timestamp)

    def get_groups(self, ids, selectors=None, timestamp=None):
        """
        :type ids: list[str]
        :type selectors: list[str] | NoneType
        :type timestamp: int | NoneType
        :rtype: object_service_pb2.TRspGetObjects
        """
        selectors = selectors or ['']
        return self.get_objects(ids, data_model.OT_GROUP, selectors=selectors, timestamp=timestamp)

    def list_nodes(self, query, timestamp=None, selectors=None):
        """
        :type query: str | NoneType
        :type timestamp: int | NoneType
        :type selectors: list[str] | NoneType
        :type timestamp: int | NoneType
        :rtype: object_service_pb2.TRspSelectObjects
        """
        selectors = selectors or ['']
        return self.select_objects(query, data_model.OT_NODE, selectors, timestamp)

    def list_resources(self, query, timestamp=None, selectors=None):
        """
        :type query: str | NoneType
        :type timestamp: int | NoneType
        :type selectors: list[str] | NoneType
        :rtype: object_service_pb2.TRspSelectObjects
        """
        selectors = selectors or ['']
        return self.select_objects(query, data_model.OT_RESOURCE, selectors, timestamp)

    def get_object(self, object_id, object_type, timestamp=None):
        """
        :type object_id: str
        :type object_type: int
        :type timestamp: int | NoneType
        :rtype: object_service_pb2.TRspGetObject
        """
        req = object_service_pb2.TReqGetObject()
        req.object_type = object_type
        req.object_id = object_id
        req.selector.paths.append('')
        if timestamp:
            req.timestamp = timestamp
        return self.stub.GetObject(req)

    def get_objects(self, object_ids, object_type, timestamp=None, selectors=None):
        """
        :type object_ids: list[str]
        :type object_type: int
        :type timestamp: int | NoneType
        :type selectors: list[str]
        :rtype: object_service_pb2.TRspGetObjects
        """
        req = object_service_pb2.TReqGetObjects()
        req.object_type = object_type
        if selectors:
            req.selector.paths.extend(selectors)
        else:
            req.selector.paths.append('')
        if timestamp:
            req.timestamp = timestamp
        for tmp in object_ids:
            sub = req.subrequests.add()
            sub.object_id = tmp

        req.options.ignore_nonexistent = True

        return self.stub.GetObjects(req)

    def get_pod(self, object_id, timestamp=None):
        """
        :type object_id: str
        :type timestamp: int | NoneType
        :rtype: object_service_pb2.TRspGetObject
        """
        return self.get_object(object_id, data_model.OT_POD, timestamp)

    def get_pod_set(self, object_id, timestamp=None):
        """
        :type object_id: str
        :type timestamp: int | NoneType
        :rtype: object_service_pb2.TRspGetObject
        """
        return self.get_object(object_id, data_model.OT_POD_SET, timestamp)

    def get_resource(self, object_id, timestamp=None):
        """
        :type object_id: str
        :type timestamp: int | NoneType
        :rtype: object_service_pb2.TRspGetObject
        """
        return self.get_object(object_id, data_model.OT_RESOURCE, timestamp)

    def get_group(self, object_id, timestamp=None):
        """
        :type object_id: str
        :type timestamp: int | NoneType
        :rtype: object_service_pb2.TRspGetObject
        """
        return self.get_object(object_id, data_model.OT_GROUP, timestamp)

    def remove_object(self, object_id, object_type, transaction_id=None):
        """
        :type object_type: int
        :type object_id: str
        :type transaction_id: str
        """
        req = object_service_pb2.TReqRemoveObject()
        if transaction_id is not None:
            req.transaction_id = transaction_id
        req.object_type = object_type
        req.object_id = object_id
        self.stub.RemoveObject(req)

    def remove_pod_set(self, object_id):
        """
        :type object_id: str
        """
        self.remove_object(object_type=data_model.OT_POD_SET, object_id=object_id)

    def create_pod(self, pod, transaction_id=None):
        """
        :type pod: data_model.TPod
        :type transaction_id: str
        """
        req = object_service_pb2.TReqCreateObject()
        if transaction_id is not None:
            req.transaction_id = transaction_id
        req.object_type = data_model.OT_POD
        req.attributes = yputil.dumps_proto(pod)
        self.stub.CreateObject(req)

    def create_pod_with_pod_set(self, pod_set, pod):
        """
        :type pod_set: data_model.TPodSet
        :type pod: data_model.TPod
        """
        req = object_service_pb2.TReqCreateObjects()
        sub = req.subrequests.add()
        sub.object_type = data_model.OT_POD_SET
        sub.attributes = yputil.dumps_proto(pod_set)
        sub = req.subrequests.add()
        sub.object_type = data_model.OT_POD
        sub.attributes = yputil.dumps_proto(pod)
        self.stub.CreateObjects(req)

    def update_object(self, object_id, object_type, set_updates, transaction_id=None):
        """
        :type object_id: str
        :type object_type: int
        :type set_updates:  dict[str, str]
        :type transaction_id: str | NoneType
        """
        req = object_service_pb2.TReqUpdateObject()
        if transaction_id is not None:
            req.transaction_id = transaction_id
        req.object_id = object_id
        req.object_type = object_type
        for path, value in set_updates.iteritems():
            update = req.set_updates.add()
            update.path = path
            update.value = value
        self.stub.UpdateObject(req)

    def update_pod(self, transaction_id, object_id, set_updates):
        """
        :type transaction_id: str
        :type object_id: str
        :type set_updates:  dict[str, str]
        """
        self.update_object(object_id, data_model.OT_POD, set_updates, transaction_id)

    def update_pod_set(self, transaction_id, object_id, set_updates):
        """
        :type transaction_id: str
        :type object_id: str
        :type set_updates:  dict[str, str]
        """
        self.update_object(object_id, data_model.OT_POD_SET, set_updates, transaction_id)

    def start_transaction(self):
        """
        :rtype: (str, int)
        """
        req = object_service_pb2.TReqStartTransaction()
        rsp = self.stub.StartTransaction(req)
        return rsp.transaction_id, rsp.start_timestamp

    def commit_transaction(self, transaction_id):
        """
        :type transaction_id: str
        """
        req = object_service_pb2.TReqCommitTransaction()
        req.transaction_id = transaction_id
        self.stub.CommitTransaction(req)

    def generate_timestamp(self):
        """
        :rtype: int
        """
        req = object_service_pb2.TReqGenerateTimestamp()
        rsp = self.stub.GenerateTimestamp(req)
        return rsp.timestamp

    def check_object_permissions(self, object_id, object_type, subject_id, permission):
        """
        :type object_id: str
        :type object_type: int
        :type subject_id: str
        :type permission: list[int]
        :rtype: bool
        """
        req = object_service_pb2.TReqCheckObjectPermissions()
        for p in permission:
            sub = req.subrequests.add()
            sub.object_id = object_id
            sub.object_type = object_type
            sub.subject_id = subject_id
            sub.permission = p
        rsp = self.stub.CheckObjectPermissions(req)

        for subresponse in rsp.subresponses:
            if subresponse.action == data_model.ACA_ALLOW:
                return True

        return False

    def check_write_permission(self, pod_id, subject_id):
        """
        :type pod_id: str
        :type subject_id: str
        :rtype: bool
        """
        return self.check_object_permissions(
            object_id=pod_id,
            object_type=data_model.OT_POD,
            subject_id=subject_id,
            permission=[data_model.ACA_GET_QYP_VM_STATUS, data_model.ACA_WRITE]
        )

    def check_read_permission(self, pod_id, subject_id):
        """
        :type pod_id: str
        :type subject_id: str
        :rtype: bool
        """
        return self.check_object_permissions(
            object_id=pod_id,
            object_type=data_model.OT_POD,
            subject_id=subject_id,
            permission=[data_model.ACA_GET_QYP_VM_STATUS, data_model.ACA_WRITE]
        )

    def get_user_access_allowed_to(self, user, object_type, permission):
        """
        :type user: str
        :type object_type: int
        :type permission: int
        :rtype: list[str]
        """
        req = object_service_pb2.TReqGetUserAccessAllowedTo()
        sub = req.subrequests.add()
        sub.user_id = user
        sub.object_type = object_type
        sub.permission = permission
        rsp = self.stub.GetUserAccessAllowedTo(req)
        return rsp.subresponses[0].object_ids

    def _get_user_access_allowed_to(self, user, object_type, permission):
        """
        Use get_user_access_allowed_to
        ? unused method

        :type user: str
        :type object_type: int
        :type permission: int
        :rtype: list[str]
        """
        req = object_service_pb2.TReqSelectObjects()
        req.object_type = object_type
        req.selector.paths.append('/meta/id')
        rsp = self.stub.SelectObjects(req)
        # TODO: check this
        req = object_service_pb2.TReqCheckObjectPermissions()
        for result in rsp.results:
            sub = req.subrequests.add()
            sub.object_type = object_type
            sub.object_id = yson.loads(result.values[0])
            sub.subject_id = user
            sub.permission = permission
        rsp = self.stub.CheckObjectPermissions(req)
        return [sub_rsp.object_id for sub_rsp in rsp.subresponses if sub_rsp.action == data_model.ACA_ALLOW]


def make_yp_client(cluster):
    """
    :type cluster: str
    :rtype: YpClient
    """
    for cluster_dict in sepelib_config.get_value('yp.clusters'):
        if cluster_dict['cluster'] == cluster:
            cluster_dict = cluster_dict.copy()
            cluster_dict['token'] = sepelib_config.get_value('yp.robot_token')

            session = YpInstrumentedSession()
            client = BaseYpClient(
                address=cluster_dict['url'],
                transport='http',
                config=cluster_dict,
                _http_session=session
            )
            stub = client.create_grpc_object_stub()
            return YpClient(stub)
