import hashlib
import time
import uuid
import base64

import yt_yson_bindings
from yt import yson
from sepelib.core import config
import yp.data_model as data_model
from yp_proto.yp.client.api.proto import cluster_api_pb2

from google.protobuf import json_format

from infra.qyp.proto_lib import vmset_pb2, vmagent_pb2

VMAGENT_RESOURCE_NAME = 'vmagent'
QEMU_BIN_RESOURCE_PATH = '$HOME/qemu_package/qemu/bin/qemu-system-x86_64'
QEMU_IMG_RESOURCE_PATH = '$HOME/qemu_package/qemu/bin/qemu-img'
CONF_GROUP_ID = 'qyp'
DEFAULT_STORAGE = 'hdd'
STORAGE_CLASS_MAP = {
    'hdd': '/place',
    'ssd': '/ssd',
}
STORAGE_CLASS_MAP_REV = {value: key for key, value in STORAGE_CLASS_MAP.iteritems()}
ROBOT_LOGIN = 'robot-vmagent-rtc'
SKYNET_MOUNT_POINT = '/Berkanavt/supervisor'
SKYNET_QUOTA = 107373108658176  # 99999G
# YASM SETTINGS
VMAGENT_ITYPE = 'qemuvm'
VMAGENT_CTYPE = 'prod'
VMAGENT_YP_TIER = 'yp'

MAIN_VOLUME_NAME = '/qemu-persistent'
MAIN_VOLUME_POD_MOUNT_PATH = '/qemu-persistent'
MAIN_VOLUME_VM_MOUNT_PATH = '/'

ENCODED_RESOURCE_URL_PREFIX = 'data:text/plain;charset=utf-8;base64'
MIN_ROOT_FS_LIMIT = 10 * 1024 ** 2
IO_BANDWIDTH_LIMIT_HDD = 75 * 1024 ** 2
IO_BANDWIDTH_LIMIT_SSD = 250 * 1024 ** 2


def make_encoded_resource_url(content):
    """
    :type content: str
    :rtype: str
    """
    encoded_content = base64.b64encode(content)
    return ",".join([ENCODED_RESOURCE_URL_PREFIX, encoded_content])


def decode_resource_url(url):
    """

    :type url: str
    :rtype: str
    """
    if url.startswith(ENCODED_RESOURCE_URL_PREFIX):
        return base64.b64decode(url.split(',')[1])


def can_user_change_pod_resources(login):
    """

    :type login: str
    :rtype: bool
    """
    auth_enabled = config.get_value('run.auth')
    user_is_admin = login in config.get_value('vmproxy.root_users')
    return not auth_enabled or user_is_admin


def cast_attr_dict_to_dict(obj):
    """
    :type obj: yt_proto.yt.core.ytree.proto.attributes_pb2.TAttributeDictionary
    :rtype: dict
    """
    return {item.key: yson.loads(item.value) for item in obj.attributes}


def cast_dict_to_attr_dict(d, obj):
    """
    :type d: dict
    :type obj: yt_proto.yt.core.ytree.proto.attributes_pb2.TAttributeDictionary
    """
    del obj.attributes[:]
    for key, value in d.iteritems():
        obj.attributes.add(key=key, value=yson.dumps(value))


def get_default_pod_resources():  # type: () -> dict[str, vmset_pb2.PodResource]
    default_pod_resources = config.get_value('vmproxy.default_pod_resources')
    res = {}
    for resource_key, resource_info in default_pod_resources.items():
        res[resource_key] = vmset_pb2.PodResource(**resource_info)
    return res


def cast_qemu_volumes_to_pod_disk_volume_requests(pod_id, vm_spec, disk_volume_requests, root_fs_req=None):
    """
    :type pod_id: str
    :type vm_spec: vmset_pb2.VMSpec
    :type disk_volume_requests: RepeatedCompositeFieldContainer[data_model.TPodSpec.TDiskVolumeRequest]
    :type root_fs_req: data_model.TPodSpec.TDiskVolumeRequest
    """
    qemu_volumes = vm_spec.qemu.volumes
    # Root fs
    if not root_fs_req:
        root_quota = config.get_value('vmproxy.default_porto_layer.root_quota')
        workdir_quota = config.get_value('vmproxy.default_porto_layer.workdir_quota')
        root_storage = qemu_volumes[0].storage_class or DEFAULT_STORAGE
        root_fs_volume = disk_volume_requests.add()
        root_fs_volume.id = '{}-{}'.format(pod_id, uuid.uuid4())
        root_fs_volume.storage_class = root_storage
        root_fs_volume.quota_policy.capacity = root_quota + workdir_quota
        root_fs_volume.labels.attributes.add(key='mount_path', value=yson.dumps('/'))
        root_fs_volume.labels.attributes.add(key='volume_type', value=yson.dumps('root_fs'))
        root_fs_volume.labels.attributes.add(key='root_fs_snapshot_quota', value=yson.dumps(root_quota))
        root_fs_volume.labels.attributes.add(key='work_dir_snapshot_quota', value=yson.dumps(workdir_quota))
    else:
        root_fs_volume = disk_volume_requests.add()
        root_fs_volume.CopyFrom(root_fs_req)

    io_bandwidth_dict = calculate_io_bandwidth(vm_spec)
    if root_fs_volume.storage_class in io_bandwidth_dict:
        # TODO remove condition after QEMUKVM-1505
        root_fs_guarantee = min(io_bandwidth_dict[root_fs_volume.storage_class]['guarantee'] / 3, MIN_ROOT_FS_LIMIT)
        root_fs_limit = min(io_bandwidth_dict[root_fs_volume.storage_class]['limit'] / 3, MIN_ROOT_FS_LIMIT)
        root_fs_volume.quota_policy.bandwidth_guarantee = root_fs_guarantee
        root_fs_volume.quota_policy.bandwidth_limit = root_fs_limit
        io_bandwidth_dict[root_fs_volume.storage_class]['guarantee'] -= root_fs_guarantee
        io_bandwidth_dict[root_fs_volume.storage_class]['limit'] -= root_fs_limit

    # Other volumes
    disk_numbers = {storage: 0 for storage in STORAGE_CLASS_MAP.keys()}
    for v in qemu_volumes:
        disk_numbers[v.storage_class or DEFAULT_STORAGE] += 1
    for storage_class, number in disk_numbers.items():
        if number and storage_class in io_bandwidth_dict:
            io_bandwidth_dict[storage_class]['guarantee'] /= number
            io_bandwidth_dict[storage_class]['limit'] /= number

    for i, v in enumerate(qemu_volumes):  # type: vmset_pb2.Volume
        req = disk_volume_requests.add()
        req.id = v.req_id or '{}-{}'.format(pod_id, uuid.uuid4())
        req.storage_class = v.storage_class or DEFAULT_STORAGE
        req.quota_policy.capacity = v.capacity
        if req.storage_class in io_bandwidth_dict:
            # TODO remove condition after QEMUKVM-1505
            req.quota_policy.bandwidth_guarantee = io_bandwidth_dict[req.storage_class]['guarantee']
            req.quota_policy.bandwidth_limit = io_bandwidth_dict[req.storage_class]['limit']
        disk_volume_requests_labels = {
            'mount_path': v.pod_mount_path,
            'volume_type': 'persistent',
            'qyp_resource_url': v.resource_url,
            'qyp_image_type': v.image_type,
            'qyp_vm_mount_path': v.vm_mount_path,
            'qyp_volume_name': v.name
        }
        for key, value in disk_volume_requests_labels.items():
            req.labels.attributes.add(key=key, value=yson.dumps(value))


def calculate_io_bandwidth(vm_spec):
    """
    :type vm_spec: vmset_pb2.VMSpec
    :rtype: dict[str, dict]
    """
    io_guarantees_per_storage = vm_spec.qemu.io_guarantees_per_storage
    result = {}
    for storage in STORAGE_CLASS_MAP.iterkeys():
        if storage not in io_guarantees_per_storage:
            continue
        io_guarantee = io_guarantees_per_storage[storage]
        io_limit = io_guarantee
        if storage == 'hdd':
            # QEMUKVM-1412
            io_limit *= 2
        if vm_spec.qemu.node_segment == 'dev' and storage == 'ssd':
            # QEMUKVM-1504
            io_limit = max(io_limit, IO_BANDWIDTH_LIMIT_SSD)
        if vm_spec.qemu.node_segment == 'dev' and storage == 'hdd':
            # QEMUKVM-1504
            io_limit = max(io_limit, IO_BANDWIDTH_LIMIT_HDD)
        result[storage] = {
            'guarantee': io_guarantee,
            'limit': io_limit,
        }
    return result


def cast_vmspec_to_pod_spec(pod_id, yp_cluster, spec, keys_url=None, login=None):
    """
    :type pod_id: str | unicode
    :type yp_cluster: str | unicode
    :type spec: vmset_pb2.VMSpec
    :type keys_url: str | NoneType
    :type login: str | NoneType
    :rtype: data_model.TPodSpec
    """
    network_id = spec.qemu.network_id
    vmagent_network_id = config.get_value('vmagent.network_id')
    windows_vm = spec.qemu.vm_type == vmset_pb2.VMType.WINDOWS
    # Set rootfs storage same as qemu_persistent volume
    pod_spec = data_model.TPodSpec()
    pod_spec.enable_scheduling = True
    pod_spec.host_name_kind = data_model.PHNK_PERSISTENT
    # Resource requests
    pod_spec.resource_requests.dirty_memory_limit = spec.qemu.resource_requests.dirty_memory_limit
    pod_spec.resource_requests.memory_limit = spec.qemu.resource_requests.memory_limit
    pod_spec.resource_requests.anonymous_memory_limit = spec.qemu.resource_requests.anonymous_memory_limit
    pod_spec.resource_requests.vcpu_guarantee = spec.qemu.resource_requests.vcpu_guarantee
    pod_spec.resource_requests.vcpu_limit = spec.qemu.resource_requests.vcpu_limit
    pod_spec.resource_requests.memory_guarantee = spec.qemu.resource_requests.memory_guarantee
    pod_spec.resource_requests.network_bandwidth_guarantee = spec.qemu.resource_requests.network_bandwidth_guarantee
    # commented due to QEMUKVM-1679
    # pod_spec.resource_requests.network_bandwidth_limit = spec.qemu.resource_requests.network_bandwidth_guarantee

    # Set node_id if exist
    forced_node_id = spec.qemu.forced_node_id
    if forced_node_id:
        hint = pod_spec.scheduling.hints.add()
        hint.node_id = forced_node_id
        hint.strong = True
    elif spec.scheduling.node_filter:
        pod_spec.node_filter = spec.scheduling.node_filter
    elif spec.scheduling.hints:
        # exclusive opportunity for root users to setting node for vm
        # mutually excludes forced node option
        for hint in spec.scheduling.hints:
            pod_hint = pod_spec.scheduling.hints.add()
            pod_hint.node_id = hint.node_id
            pod_hint.strong = hint.strong
    # Ip6 address requests
    # 1: vm backbone
    req = pod_spec.ip6_address_requests.add()
    req.network_id = network_id
    req.vlan_id = 'backbone'
    req.enable_dns = True
    req.labels.attributes.add(key='owner', value=yson.dumps('vm'))
    # 2: vm fastbone
    req = pod_spec.ip6_address_requests.add()
    req.network_id = network_id
    req.vlan_id = 'fastbone'
    if windows_vm:
        # QEMUKVM-902
        req.enable_dns = False
    else:
        req.enable_dns = True
    req.labels.attributes.add(key='owner', value=yson.dumps('vm'))
    # 3: container backbone
    req = pod_spec.ip6_address_requests.add()
    req.network_id = vmagent_network_id
    req.vlan_id = 'backbone'
    req.enable_dns = False
    req.labels.attributes.add(key='owner', value=yson.dumps('container'))
    req.labels.attributes.add(key='unistat', value=yson.dumps('enabled'))
    if spec.qemu.ip4_address_pool_id:
        req.ip4_address_pool_id = spec.qemu.ip4_address_pool_id
    elif spec.qemu.enable_internet:
        req.enable_internet = True

    # 4: container fastbone
    req = pod_spec.ip6_address_requests.add()
    req.network_id = vmagent_network_id
    req.vlan_id = 'fastbone'
    req.enable_dns = False
    req.labels.attributes.add(key='owner', value=yson.dumps('container'))

    # Host devices
    pod_spec.host_devices.add(path='/dev/kvm', mode='rw')
    pod_spec.host_devices.add(path='/dev/net/tun', mode='rw')

    # Sysctl properties
    pod_spec.sysctl_properties.add(name='net.ipv6.conf.all.proxy_ndp', value=yson.dumps(1))
    pod_spec.sysctl_properties.add(name='net.ipv6.conf.all.forwarding', value=yson.dumps(1))
    pod_spec.sysctl_properties.add(name='net.ipv4.conf.all.forwarding', value=yson.dumps(1))

    cast_qemu_volumes_to_pod_disk_volume_requests(pod_id, spec, pod_spec.disk_volume_requests, root_fs_req=None)

    vmagent_version = config.get_value('vmproxy.default_vmagent.version')

    pod_resources = get_default_pod_resources()
    if can_user_change_pod_resources(login) and spec.qemu.pod_resources:
        pod_resources.update(spec.qemu.pod_resources)
        if VMAGENT_RESOURCE_NAME in spec.qemu.pod_resources:
            vmagent_version = spec.vmagent_version

    if keys_url:
        pod_resources['authorized_keys'] = vmset_pb2.PodResource(url=keys_url)

    spec.vmagent_version = vmagent_version

    # Gpu
    for gpu_index in range(spec.qemu.gpu_request.capacity):
        gpu_req = pod_spec.gpu_requests.add()
        gpu_req.id = str(uuid.uuid4())
        gpu_req.model = spec.qemu.gpu_request.model

    # Iss payload
    iss_pb = make_iss_payload(
        pod_id=pod_id,
        vm_spec=spec,
        yp_cluster=yp_cluster,
        root_storage=spec.qemu.volumes[0].storage_class or DEFAULT_STORAGE,
        volumes=pod_spec.disk_volume_requests,
        pod_resources=pod_resources
    )
    pod_spec.iss.CopyFrom(iss_pb)
    return pod_spec


def make_resource_uuid(resource_url):
    """
    :type resource_url: str
    :rtype: str
    """
    return hashlib.sha1(resource_url).hexdigest()


def make_iss_payload(pod_id, vm_spec, yp_cluster, root_storage, volumes, pod_resources, porto_layer_url=None):
    """
    :type pod_id: str | unicode
    :type vm_spec: infra.qyp.proto_lib.vmset_pb2.VMSpec
    :type yp_cluster: str
    :type root_storage: str
    :type volumes: collections.Iterable[data_model.TDiskVolumeRequest]
    :type pod_resources: dict[str, vmset_pb2.PodResource]
    :type porto_layer_url: str | NoneType
    :rtype: cluster_api_pb2.HostConfiguration
    """
    now = int(time.time())
    conf = '{}-{}'.format(pod_id, now)
    layer_url = porto_layer_url or config.get_value('vmproxy.default_porto_layer.url')

    iss_proto = cluster_api_pb2.HostConfiguration()
    i = iss_proto.instances.add()
    i.id.slot.service = pod_id
    i.id.configuration.groupId = CONF_GROUP_ID
    i.id.configuration.groupStateFingerprint = conf
    i.dynamicProperties.update({
        'HBF_NAT': 'disabled',
        'SKYNET_SSH': 'enabled',
        "nanny_container_access_url": config.get_value('vmproxy.check_access_url', ''),
    })
    if vm_spec.qemu.node_segment == 'gpu-dev':
        # Move to all properties when finish testing
        i.dynamicProperties.update({
            'QEMU_SYSTEM_CMD_BIN_PATH': QEMU_BIN_RESOURCE_PATH,
            'QEMU_IMG_CMD_BIN_PATH': QEMU_IMG_RESOURCE_PATH,
        })

    yp_cluster = yp_cluster.lower()
    tags_template = 'a_geo_{geo} a_dc_{dc} a_itype_{itype} a_ctype_{ctype} a_prj_{prj} a_metaprj_unknown a_tier_{tier}'
    vmagent_port = str(config.get_value('vmproxy.default_agent_port'))
    i.properties.update({
        'INSTANCE_TAG_CTYPE': VMAGENT_CTYPE,
        'INSTANCE_TAG_ITYPE': VMAGENT_ITYPE,
        'INSTANCE_TAG_PRJ': pod_id,
        'INSTANCE_TAG_TIER': VMAGENT_YP_TIER,
        'POD_ID': pod_id,
        'STORAGE_PATH': MAIN_VOLUME_POD_MOUNT_PATH,
        'HOSTNAME': '{}.{}.yp-c.yandex.net'.format(pod_id, yp_cluster),
        'PORT': vmagent_port,
        'VMAGENT_PATH': './{}/vmagent'.format(VMAGENT_RESOURCE_NAME),
        'tags': tags_template.format(geo=yp_cluster,
                                     dc=yp_cluster,
                                     itype=VMAGENT_ITYPE,
                                     ctype=VMAGENT_CTYPE,
                                     prj=pod_id,
                                     tier=VMAGENT_YP_TIER),
        'yasmUnistatFallbackPort': vmagent_port,
        'NANNY_SERVICE_ID': pod_id,
        'VMCTL_FORCE_DIRECT_CONNECTION': 'True'
    })
    if vm_spec.qemu.gpu_request.capacity:
        gpu_req = vm_spec.qemu.gpu_request
        i.properties['QYP_GPU'] = '{}-{}-{}-{}'.format(
            gpu_req.model, gpu_req.capacity, gpu_req.min_memory, gpu_req.max_memory
        )
    if vm_spec.qemu.use_nat64:
        i.properties['USE_NAT64'] = 'true'
    i.targetState = 'ACTIVE'
    i.entity.instance.storage = STORAGE_CLASS_MAP[root_storage]

    i.entity.instance.container.constraints.update({
        'meta.enable_porto': 'isolate',
        'iss_hook_start.enable_porto': 'isolate',
        "iss_hook_start.capabilities_ambient": "NET_BIND_SERVICE",
        "iss_hook_start.net": "inherited",
    })

    # Hook timelimits
    iss_hook_status_tl = i.entity.instance.timeLimits['iss_hook_status']
    iss_hook_status_tl.restartPeriodScaleMs = config.get_value('hooks.limits.default.restart_period_scale') * 1000
    iss_hook_status_tl.restartPeriodBackOff = config.get_value('hooks.limits.default.restart_period_backoff')
    iss_hook_status_tl.maxRestartPeriodMs = config.get_value('hooks.limits.default.max_restart_period') * 1000
    iss_hook_status_tl.minRestartPeriodMs = config.get_value('hooks.limits.default.min_restart_period') * 1000
    iss_hook_status_tl.maxExecutionTimeMs = config.get_value('hooks.limits.default.max_execution_time') * 1000

    iss_hook_stop_tl = i.entity.instance.timeLimits['iss_hook_stop']
    iss_hook_stop_tl.maxExecutionTimeMs = config.get_value(
        'hooks.limits.iss_hook_stop.max_execution_time', 5 * 60) * 1000

    # Pod Resources
    for resource_key, resource_info in pod_resources.items():
        set_iss_payload_resource(
            payload_pb=iss_proto,
            name=resource_key,
            url=resource_info.url,
            content=resource_info.content,
            dynamic=resource_info.dynamic,
            check_period=resource_info.check_period,
            checksum=resource_info.checksum
        )
    # Volumes
    for volume_req in volumes:
        labels_dict = cast_attr_dict_to_dict(volume_req.labels)
        volume_type = labels_dict['volume_type']
        v = i.entity.instance.volumes.add()
        v.mountPoint = labels_dict['mount_path']
        v.storage = STORAGE_CLASS_MAP[volume_req.storage_class]
        if volume_type == 'root_fs':
            v.quotaBytes = labels_dict['root_fs_snapshot_quota']
            v.quotaCwdBytes = labels_dict['work_dir_snapshot_quota']
            l = v.layers.add()
            l.uuid = make_resource_uuid(layer_url)
            l.storage = STORAGE_CLASS_MAP[volume_req.storage_class]
            l.verification.checkPeriod = '0d0h0m'
            l.verification.checksum = 'EMPTY:'
            l.urls.append(layer_url)
        elif volume_type == 'persistent':
            v.uuid = volume_req.id
            v.quotaBytes = volume_req.quota_policy.capacity
            v.quotaCwdBytes = volume_req.quota_policy.capacity

    # Skynet bind volume
    v = i.entity.instance.volumes.add()
    v.mountPoint = SKYNET_MOUNT_POINT
    v.quotaBytes = SKYNET_QUOTA
    v.quotaCwdBytes = SKYNET_QUOTA
    v.properties['read_only'] = 'true'
    v.properties['storage'] = SKYNET_MOUNT_POINT
    v.properties['backend'] = 'bind'
    return iss_proto


def update_iss_payload_config_fingerprint(iss_proto, pod_id):
    """

    :type iss_proto:
    :type pod_id:
    """
    now = int(time.time())
    conf = '{}-{}'.format(pod_id, now)

    i = iss_proto.instances[0]
    i.id.configuration.groupStateFingerprint = conf


def get_iss_payload_root_storage_class(iss_proto):
    i = iss_proto.instances[0]
    return STORAGE_CLASS_MAP_REV[i.entity.instance.storage]


def update_iss_payload_use_nat64(iss_proto, vm_spec):
    """

    :type iss_proto:
    :type vm_spec:
    """
    i = iss_proto.instances[0]
    if vm_spec.qemu.use_nat64:
        i.properties['USE_NAT64'] = 'true'
    elif 'USE_NAT64' in i.properties:
        del i.properties['USE_NAT64']


def set_iss_payload_porto_layer(iss_proto, vm_spec):
    """

    :type iss_proto:
    :type vm_spec:
    """
    i = iss_proto.instances[0]
    for v in i.entity.instance.volumes:
        if v.mountPoint == '/':
            v.layers[0].urls[0] = vm_spec.qemu.porto_layer.url


def get_iss_payload_pod_resources(iss_proto):
    """

    :type iss_proto: cluster_api_pb2.HostConfiguration
    :type: dict[str, vmset_pb2.PodResource]
    """
    if not iss_proto.instances:
        return {}
    i = iss_proto.instances[0]
    res = {}
    for resource_name, iss_resource in i.entity.instance.resources.iteritems():
        dynamic = bool(iss_resource.dynamicResource.uuid)
        iss_resource_info = iss_resource.dynamicResource if dynamic else iss_resource.resource
        res[resource_name] = vmset_pb2.PodResource()
        res[resource_name].dynamic = dynamic
        res[resource_name].url = iss_resource_info.urls[0]
        res[resource_name].checksum = iss_resource_info.verification.checksum
        res[resource_name].check_period = iss_resource_info.verification.checkPeriod
    return res


def make_qyp_labels_dict(vm_spec):
    """
    :type vm_spec: vmset_pb2.VMSpec
    :rtype: dict[str, str]
    """
    labels = {
        'deploy_engine': 'QYP',
        'version': str(uuid.uuid4()),
        'vmagent_version': vm_spec.vmagent_version,
        'qyp_vm_mark': vm_spec.labels,
        'qyp_vm_autorun': vm_spec.qemu.autorun,
        'qyp_vm_type': vm_spec.qemu.vm_type,
    }
    if vm_spec.qemu.forced_node_id:
        labels['qyp_vm_forced_node_id'] = vm_spec.qemu.forced_node_id
    shelf_time_accounts = config.get_value('accounts_with_vm_shelf_time', {})
    if vm_spec.account_id in shelf_time_accounts:
        labels['qyp_vm_shelf_time_seconds'] = shelf_time_accounts[vm_spec.account_id]['seconds']
    return labels


def set_iss_payload_resource(payload_pb, name, url=None, content=None, dynamic=False, check_period=None,
                             checksum=None):
    """
    :type payload_pb: cluster_api_pb2.HostConfigurationInstance
    :type name: str
    :type url: basestring
    :type content: basestring
    :type dynamic: bool
    :type check_period: basestring
    :type checksum: str | NoneType
    """
    if (not content and not url) or (content and url):
        raise ValueError('Only content or url mast be passed')
    if content:
        url = make_encoded_resource_url(content)

    if dynamic:
        new_resource = payload_pb.instances[0].entity.instance.resources[name].dynamicResource
    else:
        new_resource = payload_pb.instances[0].entity.instance.resources[name].resource
    new_resource.uuid = make_resource_uuid(url)
    del new_resource.urls[:]
    new_resource.urls.append(url)
    new_resource.verification.checkPeriod = check_period or '0d0h0m'
    new_resource.verification.checksum = checksum or 'EMPTY:'


def get_config_from_iss_resource(iss):
    pod_resources = get_iss_payload_pod_resources(iss)
    config_resource = pod_resources.get('vm.config')
    if config_resource and config_resource.url:
        _config = vmagent_pb2.VMConfig()
        config_content = decode_resource_url(config_resource.url)
        json_format.Parse(config_content, _config)
        return _config


def cast_pod_to_vm(pod_pb, pod_set_pb):
    """
    :type pod_pb: data_model.TPod
    :type pod_set_pb: data_model.TPodSet
    :rtype: vmagent.vmset_pb2.VM
    """
    pod_labels = cast_attr_dict_to_dict(pod_pb.labels)
    pod_set_labels = cast_attr_dict_to_dict(pod_set_pb.labels)
    pod_annotations = cast_attr_dict_to_dict(pod_pb.annotations)
    vm = vmset_pb2.VM()
    annotations_vm = None
    if 'qyp_vm_spec' in pod_annotations:
        annotations_vm = vmset_pb2.VM.FromString(pod_annotations['qyp_vm_spec'])

    # Meta
    vm.meta.id = pod_pb.meta.id
    owners = pod_annotations['owners']
    vm.meta.auth.owners.logins.extend(owners['logins'])
    vm.meta.auth.owners.group_ids.extend(map(str, owners['groups']))
    vm.meta.version.pod = pod_labels['version']
    vm.meta.version.pod_set = pod_set_labels['version']
    vm.meta.creation_time.FromMicroseconds(pod_pb.meta.creation_time)
    vm.meta.last_modification_time.FromSeconds(pod_pb.status.master_spec_timestamp / 2 ** 30)

    if 'author' in owners:
        vm.meta.author = owners['author']

    # Spec
    vm.spec.account_id = pod_set_pb.spec.account_id
    for ip_req in pod_pb.spec.ip6_address_requests:
        if ip_req.enable_dns:
            vm.spec.qemu.network_id = ip_req.network_id
        if ip_req.ip4_address_pool_id:
            vm.spec.qemu.ip4_address_pool_id = ip_req.ip4_address_pool_id
        if ip_req.enable_internet:
            vm.spec.qemu.enable_internet = True
    vm.spec.qemu.node_segment = pod_set_pb.spec.node_segment_id
    vm.spec.qemu.resource_requests.dirty_memory_limit = pod_pb.spec.resource_requests.dirty_memory_limit
    vm.spec.qemu.resource_requests.memory_limit = pod_pb.spec.resource_requests.memory_limit
    vm.spec.qemu.resource_requests.anonymous_memory_limit = pod_pb.spec.resource_requests.anonymous_memory_limit
    vm.spec.qemu.resource_requests.vcpu_guarantee = pod_pb.spec.resource_requests.vcpu_guarantee
    vm.spec.qemu.resource_requests.vcpu_limit = pod_pb.spec.resource_requests.vcpu_limit
    vm.spec.qemu.resource_requests.memory_guarantee = pod_pb.spec.resource_requests.memory_guarantee
    vm.spec.qemu.resource_requests.network_bandwidth_guarantee = pod_pb.spec.resource_requests.network_bandwidth_guarantee
    iss_payload = pod_pb.spec.iss

    if iss_payload.instances:
        vm_config = get_config_from_iss_resource(iss_payload)
        i = iss_payload.instances[0]
        for iss_vol in i.entity.instance.volumes:
            if iss_vol.mountPoint == '/':
                vm.spec.qemu.porto_layer.url = iss_vol.layers[0].urls[0]

        if i.properties.get('USE_NAT64') == 'true':
            vm.spec.qemu.use_nat64 = True
    else:
        vm_config = None

    guarantees_dict = vm.spec.qemu.io_guarantees_per_storage
    guarantees_dict.update({storage: 0 for storage in STORAGE_CLASS_MAP.keys()})

    for volume_request in pod_pb.spec.disk_volume_requests:
        volume_request_labels = cast_attr_dict_to_dict(volume_request.labels)
        pod_mount_path = volume_request_labels['mount_path']
        guarantees_dict[volume_request.storage_class] += volume_request.quota_policy.bandwidth_guarantee
        if pod_mount_path == '/':
            continue

        qemu_volume = vm.spec.qemu.volumes.add()  # type: vmset_pb2.Volume
        qemu_volume.capacity = volume_request.quota_policy.capacity
        qemu_volume.storage_class = volume_request.storage_class
        qemu_volume.req_id = volume_request.id

        if pod_mount_path == MAIN_VOLUME_POD_MOUNT_PATH:
            qemu_volume.name = MAIN_VOLUME_NAME
            qemu_volume.vm_mount_path = '/'
            qemu_volume.pod_mount_path = pod_mount_path

            resource_url = volume_request_labels.get('qyp_resource_url')
            if resource_url is not None:
                qemu_volume.resource_url = resource_url
            elif vm_config:
                qemu_volume.resource_url = vm_config.disk.resource.rb_torrent

            image_type = volume_request_labels.get('qyp_image_type')
            if image_type is not None:
                qemu_volume.image_type = image_type
            elif vm_config:
                qemu_volume.image_type = vm_config.disk.type
        else:
            qemu_volume.name = volume_request_labels.get('qyp_volume_name')
            qemu_volume.resource_url = volume_request_labels['qyp_resource_url']
            qemu_volume.image_type = volume_request_labels['qyp_image_type']
            qemu_volume.pod_mount_path = pod_mount_path
            qemu_volume.vm_mount_path = volume_request_labels['qyp_vm_mount_path']

    if pod_pb.spec.gpu_requests:
        vm.spec.qemu.gpu_request.capacity = len(pod_pb.spec.gpu_requests)
        vm.spec.qemu.gpu_request.model = pod_pb.spec.gpu_requests[0].model

    if 'vmagent_version' in pod_labels:
        vm.spec.vmagent_version = pod_labels['vmagent_version']
    else:
        vm.spec.vmagent_version = 'N/A'

    qyp_vm_mark_labels = pod_labels.get('qyp_vm_mark', {})
    for label_name, label_value in qyp_vm_mark_labels.items():
        vm.spec.labels[label_name] = label_value

    autorun = pod_labels.get('qyp_vm_autorun')
    if autorun is not None:
        vm.spec.qemu.autorun = autorun
    elif vm_config:
        vm.spec.qemu.autorun = vm_config.autorun

    if annotations_vm is not None and annotations_vm.spec.qemu.HasField('qemu_options'):
        vm.spec.qemu.qemu_options.CopyFrom(annotations_vm.spec.qemu.qemu_options)

    vm_type = pod_labels.get('qyp_vm_type')
    if vm_type is not None:
        vm.spec.qemu.vm_type = vm_type
    elif vm_config:
        vm.spec.qemu.vm_type = vm_config.type

    if 'qyp_vm_forced_node_id' in pod_labels:
        vm.spec.qemu.forced_node_id = pod_labels['qyp_vm_forced_node_id']
    elif pod_pb.spec.node_filter:
        vm.spec.scheduling.node_filter = pod_pb.spec.node_filter
    elif pod_pb.spec.scheduling.hints:
        for pod_hint in pod_pb.spec.scheduling.hints:
            hint = vm.spec.scheduling.hints.add()
            hint.node_id = pod_hint.node_id
            hint.strong = pod_hint.strong

    if 'qyp_vm_shelf_time_seconds' in pod_labels:
        vm.spec.shelf_time = pod_labels['qyp_vm_shelf_time_seconds']

    # Status
    if pod_pb.status.scheduling.node_id:
        vm.status.scheduling.node_id = pod_pb.status.scheduling.node_id
    for ip in pod_pb.status.ip6_address_allocations:
        ip_allocation = vm.status.ip_allocations.add()
        ip_allocation.address = ip.address
        ip_allocation.vlan_id = ip.vlan_id
        ip_labels = cast_attr_dict_to_dict(ip.labels)
        if 'owner' in ip_labels:
            ip_allocation.owner = ip_labels['owner']
    return vm


def loads_proto(data, proto_class, ignore_unknown_fields=True):
    return yt_yson_bindings.loads_proto(data, proto_class=proto_class,
                                        skip_unknown_fields=ignore_unknown_fields)


def dumps_proto(proto_obj):
    return yt_yson_bindings.dumps_proto(proto_obj)
