import cachetools
import inject
from sepelib.core import config

from infra.swatlib.auth import staff


class SecurityPolicy(object):
    staff_client = inject.attr(staff.IStaffClient)

    def __init__(self, is_enabled=True, root_users=None):
        """
        :type is_enabled: bool
        :type root_users: list[str]
        """
        self.is_enabled = is_enabled
        self._root_users = frozenset(root_users) if root_users else frozenset()
        self.staff_groups_cache = cachetools.TTLCache(maxsize=300, ttl=30 * 60)

    @classmethod
    def from_sepelib_config(cls):
        """
        Factory method to construct policy object from sepelib config.
        :rtype: ServicesSecurityPolicy
        """
        return cls(
            is_enabled=config.get_value('run.auth', True),
            root_users=config.get_value('vmproxy.root_users', []),
        )

    def is_root(self, login):
        return login in self._root_users

    def is_allowed(self, login, logins, group_ids):
        """
        :type login: str
        :type logins: list[str]
        :type group_ids: list[str]
        :rtype: bool
        """
        if not self.is_enabled or self.is_root(login):
            return True

        return self.is_owner(login, logins, group_ids)

    def is_owner(self, login, logins, group_ids):
        """
        :type login: str
        :type logins: list[str]
        :type group_ids: list[str]
        :rtype: bool
        """
        if login in logins:
            return True
        if group_ids:
            allowed_groups = {unicode(g_id) for g_id in group_ids}
            groups = self.staff_groups_cache.get(login)
            if groups is None:
                groups = staff.get_group_ids(self.staff_client, login)
                self.staff_groups_cache[login] = groups
            if set(groups) & allowed_groups:
                return True
        return False

    def check_write_permission(self, pod_id, login, yp_client):
        if not self.is_enabled:
            return True
        return yp_client.check_write_permission(pod_id, login)

    def check_read_permission(self, pod_id, login, yp_client):
        if not self.is_enabled:
            return True
        return yp_client.check_read_permission(pod_id, login)
