from sepelib.core import config

from infra.qyp.vmproxy.src import errors


def get_instance(args, ctx):
    """
    :type args: dict
    :type ctx: web.app.Ctx
    :rtype: VMInstance
    """
    port = args.get('port')
    host = args.get('host')
    service = args.get('service')
    pod_id = args.get('pod_id')
    if pod_id:
        return PodVMInstance.from_pod_id(pod_id, ctx)
    elif host and port and service:
        return HostVMInstance.from_args(host, port, service, ctx)
    else:
        raise errors.ValidationError('Required host/port/service or pod_id/cluster params missing')


class VMInstance(object):
    VM_TYPE = None

    def check_access(self, login, ctx):
        """
        :type login: str
        :type ctx: web.app.Ctx
        """
        raise NotImplementedError

    def check_write_access(self, login, ctx):
        """
        :type login: str
        :type ctx: web.app.Ctx
        """
        raise NotImplementedError

    def get_agent_url(self, suffix=None):
        """
        :rtype: str
        """
        raise NotImplementedError

    def get_vnc_host_port(self):
        """
        :rtype: (str, int)
        """
        raise NotImplementedError


class HostVMInstance(VMInstance):
    VM_TYPE = 'gencfg'

    def __init__(self, host, port, service):
        """
        :type host: str
        :type port: str
        :type service: str
        """
        self.host = host
        self.port = int(port)
        self.service = service

    def __str__(self):
        return '{}:{}:{}'.format(self.host, self.port, self.service)

    @classmethod
    def from_args(cls, host, port, service, ctx):
        cls._validate(host, port, service, ctx)
        return cls(host, port, service)

    @staticmethod
    def _validate(host, port, service, ctx):
        try:
            port = int(port)
        except ValueError as e:
            raise errors.ValidationError("Invalid port value: {}".format(e.message))
        instances = ctx.nanny_client.get_service_instances(service)
        if not any(i['hostname'] == host and i['port'] == port for i in instances):
            raise errors.ValidationError('Specified instance does not belong to service')

    def _check_access(self, login, ctx):
        owners = ctx.nanny_client.get_service_owners(self.service)
        if not ctx.sec_policy.is_allowed(login, owners['logins'], owners['groups']):
            raise errors.AuthorizationError('User {} is not a service {} owner'.format(login, self.service))

    def check_access(self, login, ctx):
        self._check_access(login, ctx)

    def check_write_access(self, login, ctx):
        self._check_access(login, ctx)

    def get_agent_url(self, suffix=None):
        base_url = 'http://{}:{}'.format(self.host, self.port)
        if suffix:
            return '{}/{}'.format(base_url, suffix.lstrip('/'))
        return base_url

    def get_vnc_host_port(self):
        return self.host, self.port + 3


class PodVMInstance(VMInstance):
    VM_TYPE = 'yp'

    def __init__(self, pod_id, pod, container_ip, port):
        """
        :type pod_id: str
        :type pod: data_model_pb2.TPod
        :type container_ip: str
        :type port: str
        """
        self.pod_id = pod_id
        self.pod = pod
        self.container_ip = container_ip
        self.port = int(port)

    def __str__(self):
        return self.pod_id

    @classmethod
    def from_pod_id(cls, pod_id, ctx):
        pod = ctx.pod_ctl.get_active_pod(pod_id)
        container_ip = ctx.pod_ctl.get_pod_container_ip(pod)
        port = config.get_value('vmproxy.default_agent_port')
        return cls(pod_id, pod, container_ip, port)

    def check_access(self, login, ctx):
        if not ctx.pod_ctl.check_read_permission(self.pod_id, login):
            raise errors.AuthorizationError('User {} has no access to pod {}'.format(login, self.pod_id))

    def check_write_access(self, login, ctx):
        if not ctx.pod_ctl.check_write_permission(self.pod_id, login):
            raise errors.AuthorizationError('User {} has no write access to pod {}'.format(login, self.pod_id))

    def get_agent_url(self, suffix=None):
        base_url = 'http://[{}]:{}'.format(self.container_ip, self.port)
        if suffix:
            return '{}/{}'.format(base_url, suffix.lstrip('/'))
        return base_url

    def get_vnc_host_port(self):
        return self.container_ip, self.port + 3
