import copy

import flask
import json
import re

from infra.swatlib.auth.util import login_exempt
from sepelib.core import config
from infra.qyp.vmproxy.src.web.vmproxy_views import vmproxy_bp
from infra.qyp.vmproxy.src.web.vmset_service import vmset_service_bp
from infra.qyp.vmproxy.src.lib.heartbeat_client import HeartBeatClient
from infra.qyp.vmproxy.src.lib.account_manager_client import AccountManagerClient


class Ctx(object):
    """
    Web application context passed via flask.g attribute.
    """

    def __init__(self, nanny_client, vmagent_client, sec_policy, tvm_context, pod_ctl_factory,
                 personal_quotas_dict, personal_accounts, vmproxy_config, qdm_client=None):
        """

        :type nanny_client: infra.qyp.vmproxy.src.lib.nanny_client.NannyClient
        :type vmagent_client: infra.qyp.vmproxy.src.lib.vmagent_client.VmagentClient
        :type sec_policy:
        :type tvm_context:
        :type pod_ctl_factory: infra.qyp.vmproxy.src.pod_controller.PodControllerFactory
        :type personal_quotas_dict:
        :type personal_accounts: set
        :type vmproxy_config:
        :type qdm_client: infra.qyp.vmproxy.src.lib.qdm_client.QDMClient
        """
        self.nanny_client = nanny_client
        self.vmagent_client = vmagent_client
        self.sec_policy = sec_policy
        self.tvm_context = tvm_context
        self.pod_ctl_factory = pod_ctl_factory
        self.pod_ctl = self.pod_ctl_factory.get_object()
        self.personal_quotas_dict = personal_quotas_dict
        self.personal_accounts = personal_accounts
        self.vmproxy_config = vmproxy_config
        self.response_vmproxy_config = None
        self.qdm_client = qdm_client
        self.heartbeat_client = HeartBeatClient()
        self.account_manager_client = AccountManagerClient()

    def make_request_context(self, pod_ctl):
        new_ctx = copy.copy(self)
        new_ctx.pod_ctl = pod_ctl
        return new_ctx


def create_app(ctx, hostname, version, debug=False):
    # Create application
    app = flask.Flask('vmproxy')
    # By default Flask sorts keys in json to improve cache ability.
    # Our users don't cache responses based on content,
    # so let's try to improve performance.
    app.config['JSON_SORT_KEYS'] = False

    @app.before_request
    def add_ctx():
        flask.g.ctx = ctx

    @app.before_request
    def handle_options_request():
        if flask.request.method == 'OPTIONS':
            return ''  # skip authentication process

    allowed_origins = config.get_value('web.http.cors.allowed_origins', [])
    allowed_origins_regex = config.get_value('web.http.cors.allowed_origins_regex', [])

    @app.after_request
    def add_diagnostics_headers(response):
        response.headers['X-Backend-Version'] = version
        response.headers['X-Backend-Host'] = hostname
        response.headers['Access-Control-Allow-Methods'] = 'PUT, DELETE'
        response.headers['Access-Control-Expose-Headers'] = 'etag,x-total-items'
        response.headers['Access-Control-Allow-Headers'] = 'content-type,if-match'
        response.headers['Access-Control-Allow-Credentials'] = 'true'
        origin = flask.request.environ.get('HTTP_ORIGIN')
        if origin in allowed_origins:
            response.headers['Access-Control-Allow-Origin'] = origin
        elif origin and any(re.match(domain_pattern, origin) for domain_pattern in allowed_origins_regex):
            # QEMUKVM-1510
            response.headers['Access-Control-Allow-Origin'] = origin
        return response

    @app.route('/config_vmproxy', methods=['GET'])
    def render_config_white_list():
        if flask.g.ctx.response_vmproxy_config is None:
            flask.g.ctx.response_vmproxy_config = flask.Response(json.dumps(flask.g.ctx.vmproxy_config, indent=2),
                                                                 status=200, content_type='application/json')
        return flask.g.ctx.response_vmproxy_config

    @app.route('/alive', methods=['GET'])
    @login_exempt
    def alive():
        return flask.Response('', status=200)

    app.hostname = hostname
    app.version = version

    # Additional customization
    app.debug = debug
    app.register_blueprint(vmproxy_bp)
    app.register_blueprint(vmset_service_bp)
    return app
