import logging
import sys

import requests
import yp.common as yp_common

from infra.swatlib.rpc import exceptions
from infra.qyp.vmproxy.src import errors as vmproxy_errors
from infra.qyp.vmproxy.src.lib import vmagent_client


log = logging.getLogger(__name__)


def yp_error_dict_to_str(error):
    """
    :type error: dict
    :rtype: str
    """
    message = ""
    if "message" in error:
        message = error["message"]

    for inner_error in error.get("inner_errors", []):
        message += ': {}'.format(yp_error_dict_to_str(inner_error))
    return message


def yp_error_to_str(e):
    """
    :type e: yt.common.YtError
    :rtype: str
    """
    return yp_error_dict_to_str(e.simplify())


def translate_requests_http_error(e):
    message = 'Request "{}" failed with {} {}: {}'.format(e.request.url,
                                                          e.response.status_code,
                                                          e.response.reason,
                                                          e.response.content)
    return exceptions.InternalError, message


def translate_requests_error(e):
    message = ''
    if e.request is not None:
        message += 'Request to {} failed: '.format(e.request.url)
    message += unicode(e)
    return exceptions.InternalError, message


def translate_value_error(e):
    return exceptions.BadRequestError, e.message


YP_TO_RPC_MAPPING = {
    yp_common.YpDuplicateObjectIdError: exceptions.ConflictError,
    yp_common.YpNoSuchObjectError: exceptions.NotFoundError,
    yp_common.YpAuthorizationError: exceptions.ForbiddenError,
    yp_common.YpAuthenticationError: exceptions.UnauthenticatedError,
}

VMPROXY_TO_RPC_MAPPING = {
    vmproxy_errors.AuthorizationError: exceptions.ForbiddenError,
    vmproxy_errors.ValidationError: exceptions.BadRequestError,
    vmproxy_errors.WrongStateError: exceptions.BadRequestError,
    vmproxy_errors.ConcurrentModificationError: exceptions.ConflictError,
}


def translate_yp_response_error(e):
    rpc_exc_cls = YP_TO_RPC_MAPPING.get(e.__class__)
    if rpc_exc_cls is None:
        if yp_common.contains_code(e.error, 109):
            # YpAuthenticationError
            return exceptions.UnauthenticatedError, yp_error_to_str(e)
        if yp_common.contains_code(e.error, 100012):
            # Quota exceeded
            return exceptions.BadRequestError, yp_error_to_str(e)
        if yp_common.contains_code(e.error, 100013):
            # Cannot satisfy resource request on update
            return exceptions.BadRequestError, yp_error_to_str(e)
        return exceptions.InternalError, yp_error_to_str(e)
    return rpc_exc_cls, yp_error_to_str(e)


def translate_vmproxy_error(e):
    rpc_exc_cls = VMPROXY_TO_RPC_MAPPING.get(e.__class__)
    if rpc_exc_cls is None:
        return exceptions.InternalError, e.message
    return rpc_exc_cls, e.message


def translate_app_errors(f):
    """
    Handles application errors (like requests exception, ISS exception, etc) and translates them
    into RPC errors with proper error codes and nice description for user.

    :param f: function to wrap
    """

    def wrapper(*args, **kwargs):
        try:
            return f(*args, **kwargs)
        except requests.HTTPError as e:
            exc_cls, exc_args = translate_requests_http_error(e)
        except requests.RequestException as e:
            exc_cls, exc_args = translate_requests_error(e)
        except ValueError as e:
            exc_cls, exc_args = translate_value_error(e)
        except yp_common.YtResponseError as e:
            exc_cls, exc_args = translate_yp_response_error(e)
        except yp_common.YtError as e:
            exc_cls, exc_args = exceptions.InternalError, yp_error_to_str(e)
        except vmproxy_errors.OperationError as e:
            exc_cls, exc_args = translate_vmproxy_error(e)
        except vmagent_client.VmagentConnectError as e:
            exc_cls, exc_args = exceptions.BadRequestError, e.message
        # Grab traceback and raise new exception with it
        # So that we don't loose exception context
        traceback = sys.exc_info()[2]
        if not isinstance(exc_args, tuple):
            exc_args = (exc_args,)
        log.error('Request failed with error: %s', str(e))
        raise exc_cls, exc_args, traceback

    return wrapper
