import base64
import httplib
import logging

import inject
import flask
import requests.exceptions as request_exceptions
import yp.common as yp_common

from infra.qyp.proto_lib import vmagent_api_pb2
from infra.qyp.vmproxy.src import vm_instance
from infra.qyp.vmproxy.src import errors
from infra.qyp.vmproxy.src.action import config
from infra.qyp.vmproxy.src.action import backup as backup_action
from infra.qyp.vmproxy.src.action import helpers
from infra.qyp.vmproxy.src.lib import vmagent_client
from infra.qyp.vmproxy.src.web import error_utils
from infra.swatlib.rpc import authentication, exceptions as rpc_exceptions
from infra.swatlib.rpc.blueprint import make_authentication_request

vmproxy_bp = flask.Blueprint('api.vmproxy', __name__)
log = logging.getLogger(__name__)


@vmproxy_bp.before_request
def auth_request():
    authenticator = inject.instance(
        authentication.IRpcAuthenticator)  # type: authentication.CachingPassportAuthenticator
    if flask.request.endpoint:
        view = flask.current_app.view_functions.get(flask.request.endpoint)
        if not getattr(view, 'need_auth', True):
            auth_subject = authenticator.ANON_RESULT
        else:
            _auth_request = make_authentication_request(flask.request)
            auth_subject = authenticator.authenticate_request(_auth_request)
    else:
        auth_subject = authenticator.ANON_RESULT

    flask.g.identity = auth_subject


@vmproxy_bp.errorhandler(errors.ValidationError)
def handle_params_error(e):
    resp = flask.jsonify(error='Bad request', msg=e.message)
    resp.status_code = httplib.BAD_REQUEST
    return resp


@vmproxy_bp.errorhandler(ValueError)
def handle_params_error(e):
    resp = flask.jsonify(error='Bad request', msg=e.message)
    resp.status_code = httplib.BAD_REQUEST
    return resp


@vmproxy_bp.errorhandler(rpc_exceptions.UnauthenticatedError)
def handle_auth_error(e):
    resp = flask.jsonify(error='Authorization error', msg=e.message)
    resp.status_code = httplib.FORBIDDEN
    return resp


@vmproxy_bp.errorhandler(errors.WrongStateError)
def handle_wrong_state_error(e):
    resp = flask.jsonify(error='Wrong state', msg=e.message)
    resp.status_code = httplib.PRECONDITION_FAILED
    return resp


@vmproxy_bp.errorhandler(vmagent_client.VmagentConnectError)
def handle_vmagent_connect_error(e):
    resp = flask.jsonify(error='Wrong state', msg=e.message)
    resp.status_code = httplib.PRECONDITION_FAILED
    return resp


@vmproxy_bp.errorhandler(request_exceptions.RequestException)
def handle_http_error(e):
    err_msg = e.message.message  # requests puts exception in exception
    resp = flask.jsonify(error='Internal error', msg=err_msg)
    resp.status_code = e.response.status_code if e.response else httplib.INTERNAL_SERVER_ERROR
    return resp


@vmproxy_bp.errorhandler(yp_common.YpDuplicateObjectIdError)
def handle_duplicate_error(e):
    log.error('Request failed with error: %s', str(e))
    resp = flask.jsonify(error='Duplicate error', msg=error_utils.yp_error_to_str(e))
    resp.status_code = httplib.CONFLICT
    return resp


@vmproxy_bp.errorhandler(yp_common.YpNoSuchObjectError)
def handle_not_found_error(e):
    log.error('Request failed with error: %s', str(e))
    resp = flask.jsonify(error='Not found', msg=error_utils.yp_error_to_str(e))
    resp.status_code = httplib.NOT_FOUND
    return resp


@vmproxy_bp.errorhandler(yp_common.YpAuthorizationError)
def handle_not_found_error(e):
    log.error('Request failed with error: %s', str(e))
    resp = flask.jsonify(error='Forbidden', msg=error_utils.yp_error_to_str(e))
    resp.status_code = httplib.FORBIDDEN
    return resp


@vmproxy_bp.errorhandler(yp_common.YtResponseError)
def handle_not_found_error(e):
    log.error('Request failed with error: %s', str(e))
    if yp_common.contains_code(e.error, 109):
        resp = flask.jsonify(error='Unauthorized', msg=error_utils.yp_error_to_str(e))
        resp.status_code = httplib.UNAUTHORIZED
    else:
        resp = flask.jsonify(error='Internal error', msg=error_utils.yp_error_to_str(e))
        resp.status_code = httplib.INTERNAL_SERVER_ERROR
    return resp


@vmproxy_bp.errorhandler(yp_common.YtError)
def handle_not_found_error(e):
    log.error('Request failed with error: %s', str(e))
    resp = flask.jsonify(error='Internal error', msg=error_utils.yp_error_to_str(e))
    resp.status_code = httplib.INTERNAL_SERVER_ERROR
    return resp


def get_context():
    return flask.g.ctx


@vmproxy_bp.route('/status', methods=['GET'])
def handle_status():
    ctx = get_context()
    instance = vm_instance.get_instance(flask.request.args, ctx)
    instance.check_access(login=flask.g.identity.login, ctx=ctx)
    result = ctx.vmagent_client.status(url=instance.get_agent_url())
    return flask.Response(result, status=httplib.OK)


@vmproxy_bp.route('/action', methods=['POST'])
def handle_action():
    ctx = get_context()
    login = flask.g.identity.login
    instance = vm_instance.get_instance(flask.request.args, ctx)
    instance.check_write_access(login=login, ctx=ctx)
    data = flask.request.get_data()
    # Put config to pod spec if it is config action for yp
    data_pb = vmagent_api_pb2.VMActionRequest()
    data_pb.ParseFromString(base64.b64decode(data))
    log.info('Make action on %s by %s:\n%s', str(instance), login, data_pb)
    if data_pb.action == vmagent_api_pb2.VMActionRequest.BACKUP and instance.VM_TYPE == 'yp':
        pod = instance.pod
        vmagent_version = helpers.get_vmagent_version(pod)
        if vmagent_version is None or vmagent_version < backup_action.BACKUP_VERSION:
            raise ValueError('Minimal requested vmagent version {}, used {}'.format(
                backup_action.BACKUP_VERSION, vmagent_version
            ))
    if data_pb.action == vmagent_api_pb2.VMActionRequest.PUSH_CONFIG:
        config.run(instance, data_pb.config, ctx)
        result = None
    else:
        result = ctx.vmagent_client.action(url=instance.get_agent_url(), data=data)
    log.info('Action request on %s sent successfully', str(instance))
    return flask.Response(result, status=httplib.OK)


@vmproxy_bp.route('/debug', methods=['POST'])
def handle_debug():
    ctx = get_context()
    instance = vm_instance.get_instance(flask.request.args, ctx)
    instance.check_write_access(login=flask.g.identity.login, ctx=ctx)
    cmd = flask.request.args.get('cmd')
    if not cmd:
        raise errors.ValidationError('No debug cmd supplied')
    result = ctx.vmagent_client.debug(url=instance.get_agent_url(), cmd=cmd)
    return flask.Response(result, status=httplib.OK)
