import logging
import uuid

import base64
from google.protobuf import json_format
from infra.qyp.proto_lib import vmset_api_pb2, vmagent_api_pb2, vmset_pb2, vmagent_pb2, status_pb2
from infra.qyp.vmproxy.src.action import allocate as allocate_action, helpers
from infra.qyp.vmproxy.src.action import backup as backup_action
from infra.qyp.vmproxy.src.action import config as config_action
from infra.qyp.vmproxy.src.action import create as create_action
from infra.qyp.vmproxy.src.action import list_free_nodes as list_free_nodes_action
from infra.qyp.vmproxy.src.action import list_user_accounts as list_user_accounts_action
from infra.qyp.vmproxy.src.action import update as update_action
from infra.qyp.vmproxy.src.action import validation
from infra.qyp.vmproxy.src.lib.yp import yputil
from infra.qyp.vmproxy.src.web import error_utils

import cachetools
import flask
import yp.common as yp_common
import yp.data_model as data_model
from infra.qyp.vmproxy.src import vm_instance
from infra.swatlib.rpc import blueprint
from infra.swatlib.rpc import exceptions
from sepelib.core import config

DEFAULT_ALLOCATE_TIMEOUT = 5
LIST_VMS_LIMIT_MAXIMUM = 1000
LIST_VMS_LIMIT_DEFAULT = 1000


vmset_service_bp = blueprint.HttpRpcBlueprint(
    name='rpc.vmset_service',
    import_name=__name__,
    url_prefix='/api',
    status_msg=status_pb2.Status,
    validate_schema=True,
    infer_response_schema=True
)
access_response_cache = cachetools.TTLCache(maxsize=300, ttl=30 * 60)
vm_stats_cache = cachetools.TTLCache(maxsize=300, ttl=60 * 60)
log = logging.getLogger(__name__)


def get_context():
    """
    :rtype: infra.qyp.vmproxy.src.web.app.Ctx
    """
    return flask.g.ctx


@vmset_service_bp.method('AllocateVm',
                         request_type=vmset_api_pb2.AllocateVmRequest,
                         response_type=vmset_api_pb2.AllocateVmResponse)
@error_utils.translate_app_errors
def allocate_vm(req, auth_subject):
    ctx = get_context()
    logins = req.meta.auth.owners.logins
    group_ids = req.meta.auth.owners.group_ids
    if not ctx.sec_policy.is_allowed(auth_subject.login, logins, group_ids):
        raise exceptions.BadRequestError('User {} not set in auth field'.format(auth_subject.login))
    log.info('Allocate vm by %s:\n%s', auth_subject.login, req)
    result, error = allocate_action.set_vm_spec_defaults(req.spec)
    if not result:
        log.error(error)
        raise ValueError(error)

    create_action.run(req.meta, req.spec, ctx, auth_subject.login)
    log.info('Allocate request on vm %s sent successfully', req.meta.id)
    return vmset_api_pb2.AllocateVmResponse()


@vmset_service_bp.method('DeallocateVm',
                         request_type=vmset_api_pb2.DeallocateVmRequest,
                         response_type=vmset_api_pb2.DeallocateVmResponse)
@error_utils.translate_app_errors
def deallocate_vm(req, auth_subject):
    ctx = get_context()
    if not ctx.pod_ctl.check_write_permission(req.id, auth_subject.login):
        raise exceptions.ForbiddenError('User {} cannot delete pod {}'.format(auth_subject.login, req.id))
    log.info('Deallocate vm %s by %s', req.id, auth_subject.login)
    ctx.pod_ctl.delete_pod_with_pod_set(req.id)
    log.info('Deallocate request on vm %s sent successfully', req.id)
    return vmset_api_pb2.DeallocateVmResponse()


@vmset_service_bp.method('ListYpVm',
                         request_type=vmset_api_pb2.ListYpVmRequest,
                         response_type=vmset_api_pb2.ListYpVmResponse)
@error_utils.translate_app_errors
def list_vm(req, _):  # type: (vmset_api_pb2.ListYpVmRequest, ...) -> vmset_api_pb2.ListYpVmResponse

    if req.limit and req.limit > LIST_VMS_LIMIT_MAXIMUM:
        raise exceptions.BadRequestError("Can't use limit greater than {}".format(LIST_VMS_LIMIT_MAXIMUM))

    ctx = get_context()

    if req.query.login:
        result = ctx.pod_ctl.list_pods(
            query=req.query,
            sort=req.sort,
            skip=req.skip or 0,
            limit=req.limit or LIST_VMS_LIMIT_DEFAULT
        )
    else:
        result = ctx.pod_ctl.list_all_pods(
            query=req.query,
            skip=req.skip or 0,
            limit=req.limit or LIST_VMS_LIMIT_DEFAULT
        )

    return vmset_api_pb2.ListYpVmResponse(vms=result)


@vmset_service_bp.method('CreateVm',
                         request_type=vmset_api_pb2.CreateVmRequest,
                         response_type=vmset_api_pb2.CreateVmResponse)
@error_utils.translate_app_errors
def create_vm(req, auth_subject):
    ctx = get_context()
    logins = req.meta.auth.owners.logins
    group_ids = req.meta.auth.owners.group_ids
    if not ctx.sec_policy.is_allowed(auth_subject.login, logins, group_ids):
        raise exceptions.BadRequestError('User {} not set in auth field'.format(auth_subject.login))

    if req.forced_node_id:
        req.spec.qemu.forced_node_id = req.forced_node_id
    log.info('Create vm by %s:\n%s', auth_subject.login, req)
    req_config = req.config if req.HasField('config') else None
    create_action.run(req.meta, req.spec, ctx, auth_subject.login, config=req_config)
    log.info('Create request on %s sent successfully', req.meta.id)
    return vmset_api_pb2.CreateVmResponse()


@vmset_service_bp.method('CheckVmAccess',
                         request_type=vmset_api_pb2.CheckVmAccessRequest,
                         response_type=vmset_api_pb2.CheckVmAccessResponse,
                         need_authentication=False, max_in_flight=20)
@error_utils.translate_app_errors
def check_vm_access(req, _):
    if not req.login:
        raise exceptions.BadRequestError('No login specified')
    if not req.service_id:
        raise exceptions.BadRequestError('No service_id specified')

    key = (req.service_id, req.login)
    r = access_response_cache.get(key)
    if r is not None:
        return r

    r = vmset_api_pb2.CheckVmAccessResponse()

    ctx = get_context()
    if not ctx.pod_ctl.sec_policy.is_root(req.login):
        r.access_granted = False
        access_response_cache[key] = r
        return r

    r.access_granted = True
    request = ctx.pod_ctl.get_keys_by_logins([req.login])
    for user in request:
        for k in user:
            try:
                key_body = k['key']
                key_fingerprint = k['fingerprint']
            except KeyError:
                continue
            pb_key = r.keys.add()
            pb_key.key = key_body
            pb_key.fingerprint = key_fingerprint
    access_response_cache[key] = r
    return r


def get_vm_instance(vm_id, ctx):
    """
    :type vm_id: vmset_api_pb2.VmId
    :type ctx: vmproxy.web.app.Ctx
    :rtype: vm_instance.VMInstance
    """
    if vm_id.HasField('nanny_args'):
        return vm_instance.HostVMInstance.from_args(
            host=vm_id.nanny_args.host,
            port=vm_id.nanny_args.port,
            service=vm_id.nanny_args.service,
            ctx=ctx
        )
    elif vm_id.HasField('pod_id'):
        return vm_instance.PodVMInstance.from_pod_id(
            pod_id=vm_id.pod_id,
            ctx=ctx
        )
    else:
        raise exceptions.BadRequestError('Required host/port/service or pod_id params missing')


@vmset_service_bp.method('GetStatus',
                         request_type=vmset_api_pb2.GetStatusRequest,
                         response_type=vmset_api_pb2.GetStatusResponse)
@error_utils.translate_app_errors
def get_status(req, auth_subject):
    ctx = get_context()
    instance = get_vm_instance(req.vm_id, ctx)
    instance.check_access(login=auth_subject.login, ctx=ctx)
    status = ctx.vmagent_client.status(url=instance.get_agent_url())
    vmagent_resp = vmagent_api_pb2.VMStatusResponse()
    vmagent_resp.ParseFromString(base64.b64decode(status))

    # Backward compatibility for old UI
    # TODO: Remove this after update UI to multi volume version
    if not vmagent_resp.config.disk.resource.rb_torrent and vmagent_resp.config.volumes:
        vmagent_resp.config.disk.resource.rb_torrent = vmagent_resp.config.volumes[0].resource_url
        vmagent_resp.config.disk.type = vmagent_resp.config.volumes[0].image_type

    return vmset_api_pb2.GetStatusResponse(
        state=vmagent_resp.state,
        config=vmagent_resp.config,
        vmagent_version=vmagent_resp.vmagent_version,
        data_transfer_state=vmagent_resp.data_transfer_state,
    )


@vmset_service_bp.method('MakeAction',
                         request_type=vmset_api_pb2.MakeActionRequest,
                         response_type=vmset_api_pb2.MakeActionResponse)
@error_utils.translate_app_errors
def make_action(req, auth_subject):
    ctx = get_context()
    instance = get_vm_instance(req.vm_id, ctx)
    instance.check_write_access(login=auth_subject.login, ctx=ctx)
    log.info('Make action on %s by %s:\n%s', req.vm_id, auth_subject.login, req)
    vmagent_resp = vmagent_api_pb2.VMActionResponse()

    minimal_vmagent_version = None

    if instance.VM_TYPE == 'yp':
        if req.action == vmagent_api_pb2.VMActionRequest.BACKUP:
            minimal_vmagent_version = backup_action.BACKUP_VERSION
        elif req.action == vmagent_api_pb2.VMActionRequest.QDMUPLOAD:
            minimal_vmagent_version = backup_action.QDMUPLOAD_VERSION
        elif req.action == vmagent_api_pb2.VMActionRequest.QDMUPLOAD_BG:
            minimal_vmagent_version = backup_action.QDMUPLOAD_BG_VERSION

    if minimal_vmagent_version is not None:
        pod = instance.pod
        vmagent_version = helpers.get_vmagent_version(pod)
        if vmagent_version is None or vmagent_version < minimal_vmagent_version:
            raise ValueError('Minimal requested vmagent version {}, used {}'.format(
                minimal_vmagent_version, vmagent_version
            ))

    if req.action == vmagent_api_pb2.VMActionRequest.PUSH_CONFIG:
        pod = instance.pod
        vmagent_version = helpers.get_vmagent_version(pod)
        if vmagent_version is None:
            raise ValueError('Minimal requested vmagent version {}, used {}'.format(
                minimal_vmagent_version, vmagent_version
            ))
        if vmagent_version and vmagent_version >= config_action.MULTI_STORAGE_VMAGENT_VERSION:
            current_vm = yputil.cast_pod_to_vm(
                pod_pb=pod,
                pod_set_pb=ctx.pod_ctl.get_pod_set(req.vm_id.pod_id)
            )
            update_action.run(
                meta=current_vm.meta,
                spec=current_vm.spec,
                ctx=ctx,
                login=auth_subject.login,
                config=req.config,
                update_vmagent=False,
                update_labels=False,
            )
        else:
            config_action.run(instance, req.config, ctx)
    else:
        vmagent_req = helpers.cast_vmset_to_vmagent_action_req(req)
        encoded_data = base64.b64encode(vmagent_req.SerializeToString())
        result = ctx.vmagent_client.action(url=instance.get_agent_url(), data=encoded_data)
        vmagent_resp.ParseFromString(base64.b64decode(result))
    log.info('Action request on %s sent successfully', req.vm_id)
    return vmset_api_pb2.MakeActionResponse(state=vmagent_resp.state)


@vmset_service_bp.method('GetVm',
                         request_type=vmset_api_pb2.GetVmRequest,
                         response_type=vmset_api_pb2.GetVmResponse)
@error_utils.translate_app_errors
def get_vm(req, auth_subject):
    ctx = get_context()
    instance = vm_instance.PodVMInstance.from_pod_id(pod_id=req.vm_id, ctx=ctx)
    instance.check_access(auth_subject.login, ctx)
    vm = yputil.cast_pod_to_vm(
        pod_pb=ctx.pod_ctl.get_pod(req.vm_id),
        pod_set_pb=ctx.pod_ctl.get_pod_set(req.vm_id)
    )
    status = ctx.vmagent_client.status(url=instance.get_agent_url())
    # Put state and config from vmagent
    vmagent_resp = vmagent_api_pb2.VMStatusResponse()
    vmagent_resp.ParseFromString(base64.b64decode(status))
    vm.status.state.CopyFrom(vmagent_resp.state)
    vm.config.CopyFrom(vmagent_resp.config)
    return vmset_api_pb2.GetVmResponse(vm=vm)


@vmset_service_bp.method('UpdateVm',
                         request_type=vmset_api_pb2.UpdateVmRequest,
                         response_type=vmset_api_pb2.UpdateVmResponse)
@error_utils.translate_app_errors
def update_vm(req, auth_subject):
    ctx = get_context()
    if not ctx.pod_ctl.check_write_permission(pod_id=req.meta.id, subject_id=auth_subject.login):
        raise exceptions.ForbiddenError('User {} has no access to pod {}'.format(auth_subject.login, req.meta.id))

    log.info('Update vm %s by %s:\n%s', req.meta.id, auth_subject.login, req)
    req_config = req.config if req.HasField('config') else None
    update_action.run(
        meta=req.meta,
        spec=req.spec,
        ctx=ctx,
        login=auth_subject.login,
        update_vmagent=req.update_vmagent,
        force_update_vmagent=req.force_update_vmagent,
        update_labels=req.update_labels,
        config=req_config
    )
    log.info('Update request on vm %s sent successfully', req.meta.id)

    vm = yputil.cast_pod_to_vm(
        pod_pb=ctx.pod_ctl.get_pod(req.meta.id),
        pod_set_pb=ctx.pod_ctl.get_pod_set(req.meta.id)
    )
    return vmset_api_pb2.UpdateVmResponse(vm=vm)


@vmset_service_bp.method('ListBackup',
                         request_type=vmset_api_pb2.ListBackupRequest,
                         response_type=vmset_api_pb2.ListBackupResponse)
@error_utils.translate_app_errors
def list_backup(req, auth_subject):
    if not req.vm_id:
        raise exceptions.BadRequestError('No vm_id specified')
    skip = req.skip or 0
    limit = req.limit or 1000

    ctx = get_context()
    pod_id = req.vm_id
    pod = ctx.pod_ctl.get_pod(pod_id)

    yp_backup_list = []
    progress_backup = ctx.qdm_client.backup_status(pod_id)
    if progress_backup:
        yp_backup_list.append(progress_backup)
    for item in ctx.pod_ctl.get_backup_list(pod):
        item_pb = yputil.loads_proto(str(item), vmset_pb2.Backup)
        backup_url = item_pb.status.url
        if not backup_url or not backup_url.startswith('qdm:'):
            yp_backup_list.append(item_pb)
    backup_list = ctx.qdm_client.backup_list(vm_id=pod_id)
    backup_list.extend(yp_backup_list)
    return vmset_api_pb2.ListBackupResponse(backups=backup_list[skip:skip + limit])


@vmset_service_bp.method('ListUserBackups',
                         request_type=vmset_api_pb2.ListUserBackupsRequest,
                         response_type=vmset_api_pb2.ListUserBackupsResponse)
@error_utils.translate_app_errors
def list_user_backups(req, _):
    if not req.query.login:
        raise exceptions.BadRequestError('No query.login specified')
    skip = req.skip or 0
    limit = req.limit or 1000

    ctx = get_context()
    backup_list = ctx.qdm_client.user_backup_list(req.query)
    cluster_vms = set()
    for backup in backup_list:
        vm_id, cluster = backup.spec.vm_id.split('.')
        if cluster.upper() == ctx.pod_ctl.yp_cluster:
            cluster_vms.add(vm_id)
    alive_vms = ctx.pod_ctl.get_alive_pods(cluster_vms)

    result = []
    for backup in backup_list:
        vm_id, cluster = backup.spec.vm_id.split('.')
        if cluster.upper() != ctx.pod_ctl.yp_cluster:
            continue

        backup.status.vm_alive = vm_id in alive_vms
        if req.query.vm_name:
            if req.query.vm_name not in vm_id:
                continue

        if req.query.segment:
            if backup.spec.vm_spec.qemu.node_segment not in req.query.segment:
                continue

        if req.query.HasField('creation_time_gte'):
            if backup.meta.creation_time.ToSeconds() < req.query.creation_time_gte.ToSeconds():
                continue

        if req.query.HasField('creation_time_lte'):
            if backup.meta.creation_time.ToSeconds() > req.query.creation_time_lte.ToSeconds():
                continue

        if req.query.vm_alive:
            if req.query.vm_alive == vmset_pb2.BackupFindQuery.ALIVE_ONLY and not backup.status.vm_alive:
                continue
            if req.query.vm_alive == vmset_pb2.BackupFindQuery.REMOVED_ONLY and backup.status.vm_alive:
                continue

        result.append(backup)
    return vmset_api_pb2.ListUserBackupsResponse(backups=result[skip:skip + limit])


@vmset_service_bp.method('CreateBackup',
                         request_type=vmset_api_pb2.CreateBackupRequest,
                         response_type=vmset_api_pb2.CreateBackupResponse)
@error_utils.translate_app_errors
def create_backup(req, auth_subject):
    if not req.spec.vm_id:
        raise exceptions.BadRequestError('No vm_id specified')
    if not req.spec.sandbox_task_id:
        raise exceptions.BadRequestError('No sandbox task id specified')
    ctx = get_context()
    pod_id = req.spec.vm_id
    if not ctx.pod_ctl.check_write_permission(pod_id, auth_subject.login):
        raise exceptions.ForbiddenError('User {} cannot manage backups for pod {}'.format(auth_subject.login, pod_id))

    sb_task_id = req.spec.sandbox_task_id
    log.info('Create vm %s backup by sb-task %s:\n%s', req.spec.vm_id, sb_task_id, req)

    backup = vmset_pb2.Backup()
    backup.meta.id = str(uuid.uuid4())
    backup.meta.creation_time.GetCurrentTime()
    backup.spec.CopyFrom(req.spec)
    backup.status.state = vmset_pb2.BackupStatus.IN_PROGRESS

    pod = ctx.pod_ctl.get_pod(pod_id)
    backup_list = ctx.pod_ctl.get_backup_list(pod)
    for item in backup_list:
        backup_pb = yputil.loads_proto(str(item), vmset_pb2.Backup)
        if backup_pb.status.state in (vmset_pb2.BackupStatus.PLANNED, vmset_pb2.BackupStatus.IN_PROGRESS):
            raise exceptions.BadRequestError('Backup already in progress: {}, sb-task {}'.format(
                backup_pb.meta.id, backup_pb.spec.sandbox_task_id
            ))
        if backup_pb.spec.sandbox_task_id == sb_task_id:
            raise exceptions.ConflictError('Backup from sb-task {} already exists'.format(sb_task_id))
    backup_list.append(yputil.dumps_proto(backup))
    ctx.pod_ctl.update_backup_list(pod, backup_list)
    log.info('Create backup request on vm %s sent successfully', req.spec.vm_id)
    return vmset_api_pb2.CreateBackupResponse(backup=backup)


@vmset_service_bp.method('UpdateBackup',
                         request_type=vmset_api_pb2.UpdateBackupRequest,
                         response_type=vmset_api_pb2.UpdateBackupResponse)
@error_utils.translate_app_errors
def update_backup(req, auth_subject):
    if not req.id:
        raise exceptions.BadRequestError('No id specified')
    if not req.vm_id:
        raise exceptions.BadRequestError('No vm_id specified')
    ctx = get_context()
    pod_id = req.vm_id
    if not ctx.pod_ctl.check_write_permission(pod_id, auth_subject.login):
        raise exceptions.ForbiddenError('User {} cannot manage backups for  pod {}'.format(auth_subject.login, pod_id))

    log.info('Update vm %s backup %s:\n%s', pod_id, req.id, req)

    pod = ctx.pod_ctl.get_pod(pod_id)
    backup_list = ctx.pod_ctl.get_backup_list(pod)
    new_backup_list = []
    updated = None
    for item in backup_list:
        backup_pb = yputil.loads_proto(str(item), vmset_pb2.Backup)
        if backup_pb.meta.id == req.id:
            backup_pb.status.CopyFrom(req.status)
            if req.status == vmset_pb2.BackupStatus.IN_PROGRESS:
                backup_pb.meta.creation_time.GetCurrentTime()
            new_backup_list.append(yputil.dumps_proto(backup_pb))
            updated = backup_pb
        else:
            new_backup_list.append(item)
    if updated is None:
        raise exceptions.NotFoundError('Backup {} for vm {} does not exist'.format(req.id, pod_id))
    ctx.pod_ctl.update_backup_list(pod, new_backup_list)
    log.info('Update backup request on %s sent successfully', req.id)
    return vmset_api_pb2.UpdateBackupResponse(backup=updated)


@vmset_service_bp.method('RemoveBackup',
                         request_type=vmset_api_pb2.RemoveBackupRequest,
                         response_type=vmset_api_pb2.RemoveBackupResponse)
@error_utils.translate_app_errors
def remove_backup(req, auth_subject):
    if not req.vm_id:
        raise exceptions.BadRequestError('No vm_id specified')
    if not req.id:
        raise exceptions.BadRequestError('No id specified')
    ctx = get_context()
    pod_id = req.vm_id
    if not ctx.pod_ctl.check_write_permission(pod_id, auth_subject.login):
        raise exceptions.ForbiddenError('User {} cannot manage backups for  pod {}'.format(auth_subject.login, pod_id))

    log.info('Remove vm %s backup %s by %s:\n%s', req.vm_id, req.id, auth_subject.login, req)

    pod = ctx.pod_ctl.get_pod(pod_id)
    backup_list = ctx.pod_ctl.get_backup_list(pod)
    new_backup_list = []
    removed = None
    for item in backup_list:
        backup_pb = yputil.loads_proto(str(item), vmset_pb2.Backup)
        if backup_pb.meta.id == req.id:
            removed = True
        else:
            new_backup_list.append(item)
    if removed is None:
        raise exceptions.NotFoundError('Backup {} for vm {} does not exist'.format(req.id, pod_id))
    ctx.pod_ctl.update_backup_list(pod, new_backup_list)
    log.info('Remove backup request on %s sent successfully', req.id)
    return vmset_api_pb2.RemoveBackupResponse()


@vmset_service_bp.method('RestoreBackup',
                         request_type=vmset_api_pb2.RestoreBackupRequest,
                         response_type=vmset_api_pb2.RestoreBackupResponse)
@error_utils.translate_app_errors
def restore_backup(req, auth_subject):
    """
    :type req: infra.qyp.proto_lib.vmset_api_pb2.RestoreBackupRequest
    :type auth_subject:
    :rtype: infra.qyp.proto_lib.vmset_api_pb2.RestoreBackupResponse
    """
    if not req.vm_id:
        raise exceptions.BadRequestError('No vm_id specified')
    if not req.resource_url:
        raise exceptions.BadRequestError('No resource_url specified')

    ctx = get_context()

    pod_id = req.vm_id
    if not ctx.pod_ctl.check_write_permission(pod_id, auth_subject.login):
        raise exceptions.ForbiddenError('User {} cannot restore backup for pod {}'.format(auth_subject.login, pod_id))

    log.info('Restore vm {} backup {} by {}:\n{}'.format(req.vm_id, req.resource_url, auth_subject.login, req))
    pod_pb = ctx.pod_ctl.get_pod(pod_id)
    vm = yputil.cast_pod_to_vm(
        pod_pb=pod_pb,
        pod_set_pb=ctx.pod_ctl.get_pod_set(pod_id)
    )
    if req.resource_url.startswith('qdm:'):
        backup_spec = ctx.qdm_client.get_revision_info(req.resource_url)

        if backup_spec:
            validation.validate_qdm_backup_spec(ctx, backup_spec, auth_subject.login)
        else:
            raise ValueError('QDM not found resource with id: {}'.format(req.resource_url))

        backup_action.rewrite_qemu_volumes_from_backup_spec(vm.spec.qemu, backup_spec)

        if backup_spec.HasField('vmspec'):
            # old backups has no vmspec field
            if vm.meta.id != backup_spec.vmspec.meta.id:
                raise exceptions.BadRequestError('Backup from another vm: {}, current: {}'.format(
                    backup_spec.vmspec.meta.id, vm.meta.id))

            # Copy vm settings from backup (exclude auth)
            vm.spec.qemu.autorun = backup_spec.vmspec.spec.qemu.autorun
            vm.spec.qemu.network_id = backup_spec.vmspec.spec.qemu.network_id
            vm.spec.qemu.node_segment = backup_spec.vmspec.spec.qemu.node_segment
            vm.spec.qemu.vm_type = backup_spec.vmspec.spec.qemu.vm_type
            vm.spec.qemu.resource_requests.CopyFrom(backup_spec.vmspec.spec.qemu.resource_requests)
            vm.spec.qemu.enable_internet = backup_spec.vmspec.spec.qemu.enable_internet
            vm.spec.qemu.ip4_address_pool_id = backup_spec.vmspec.spec.qemu.ip4_address_pool_id
            vm.spec.qemu.use_nat64 = backup_spec.vmspec.spec.qemu.use_nat64

    elif req.resource_url.startswith('rbtorrent:'):
        del vm.spec.qemu.volumes[1:]
        vm.spec.qemu.volumes[0].resource_url = req.resource_url
        vm.spec.qemu.volumes[0].image_type = vmset_pb2.Volume.RAW
    else:
        raise exceptions.BadRequestError('resource_url has wrong format')

    # Clear custom info and update vmagent
    vm.spec.qemu.ClearField('porto_layer')
    vm.spec.qemu.ClearField('pod_resources')

    update_action.recreate_vm(
        ctx=ctx,
        meta=vm.meta,
        spec=vm.spec,
        login=auth_subject.login,
        pod_set_updates={},
    )

    return vmset_api_pb2.RestoreBackupResponse()


@vmset_service_bp.method('BackupVm',
                         request_type=vmset_api_pb2.BackupVmRequest,
                         response_type=vmset_api_pb2.BackupVmResponse)
@error_utils.translate_app_errors
def backup_vm(req, auth_subject):
    ctx = get_context()
    instance = get_vm_instance(req.vm_id, ctx)
    instance.check_write_access(login=auth_subject.login, ctx=ctx)
    if req.backup_storage == vmset_api_pb2.DEFAULT_BACKUP_STORAGE:
        backup_storage = config.get_value('vmproxy.default_backup_storage', 'SANDBOX_RESOURCE')
    else:
        backup_storage = vmset_api_pb2.BackupStorage.Name(req.backup_storage)
    log.info('Make backup on %s by %s:\n%s to backup_storage: %s', req.vm_id, auth_subject.login, req, backup_storage)
    backup_action.run(instance, ctx, backup_storage)
    log.info('Backup request on %s sent successfully', req.vm_id)
    return vmset_api_pb2.BackupVmResponse()


@vmset_service_bp.method('ListUserAccounts',
                         request_type=vmset_api_pb2.ListUserAccountsRequest,
                         response_type=vmset_api_pb2.ListUserAccountsResponse)
@error_utils.translate_app_errors
def list_user_accounts(req, _):
    if not req.login:
        raise exceptions.BadRequestError('No login specified')
    if req.login == 'root':
        raise exceptions.BadRequestError('Invalid login: {}'.format(req.login))
    ctx = get_context()
    use_cache = req.consistency == vmset_api_pb2.WEAK
    result = list_user_accounts_action.run(ctx, req.login, req.account_id, req.segment, use_cache)
    rsp = vmset_api_pb2.ListUserAccountsResponse()
    rsp.accounts.extend(result)
    return rsp


@vmset_service_bp.method('GetVmStats',
                         request_type=vmset_api_pb2.GetVmStatsRequest,
                         response_type=vmset_api_pb2.GetVmStatsResponse)
@error_utils.translate_app_errors
def get_vm_stats(req, _):  # type: (vmset_api_pb2.GetVmStatsRequest, ...) -> vmset_api_pb2.GetVmStatsResponse
    ctx = get_context()
    vm_stats_key = frozenset(req.account), frozenset(req.segment)
    if req.consistency != vmset_api_pb2.WEAK or not vm_stats_cache.get(vm_stats_key):
        vm_stats_cache[vm_stats_key] = ctx.pod_ctl.get_pod_stats(
            account=req.account,
            segment=req.segment
        )

    total, usage, vmagent_versions_stats = vm_stats_cache[vm_stats_key]
    resp = vmset_api_pb2.GetVmStatsResponse(
        total=total,
        usage=usage
    )
    for vmagent_version, count in vmagent_versions_stats.items():
        resp.vmagent_versions[vmagent_version] = count
    return resp


@vmset_service_bp.method('ListFreeNodes',
                         request_type=vmset_api_pb2.ListFreeNodesRequest,
                         response_type=vmset_api_pb2.ListFreeNodesResponse)
@error_utils.translate_app_errors
def list_free_nodes(req, _):  # type: (vmset_api_pb2.ListFreeNodesRequest, ...) -> vmset_api_pb2.ListFreeNodesResponse
    ctx = get_context()
    list_segments = config.get_value('vmproxy.node_segment', [])
    if req.segment not in list_segments:
        raise exceptions.BadRequestError('Incorrect segment: {}'.format(req.segment))
    resp = vmset_api_pb2.ListFreeNodesResponse()
    for node_id, res_info in list_free_nodes_action.run(ctx, req.segment).iteritems():
        resp.free_nodes[node_id].CopyFrom(res_info)
    return resp


@vmset_service_bp.method('GetVmFromBackup',
                         request_type=vmset_api_pb2.GetVmFromBackupRequest,
                         response_type=vmset_api_pb2.GetVmFromBackupResponse)
@error_utils.translate_app_errors
def get_vm_from_backup(req, _):
    """
    :type req: vmset_api_pb2.GetVmFromBackupRequest
    :rtype: vmset_api_pb2.RestoreBackupResponse
    """
    if not req.qdm_res_id:
        raise exceptions.BadRequestError('No qdm_res_id specified')
    if not req.qdm_res_id.startswith('qdm:'):
        raise exceptions.BadRequestError('qdm_res_id wrong: {}'.format(req.qdm_res_id))

    ctx = get_context()
    vm = vmset_pb2.VM()
    backup_spec = ctx.qdm_client.get_revision_info(req.qdm_res_id)

    if backup_spec is None:
        raise exceptions.NotFoundError('Backup with id: {} Not Found'.format(req.qdm_res_id))

    if backup_spec.HasField('vmspec'):
        vm.CopyFrom(backup_spec.vmspec)

    backup_action.rewrite_qemu_volumes_from_backup_spec(vm.spec.qemu, backup_spec)

    # Clear custom info
    vm.spec.qemu.ClearField('porto_layer')
    vm.spec.qemu.ClearField('pod_resources')
    vm.meta.ClearField('version')
    vm.meta.ClearField('creation_time')
    vm.meta.ClearField('last_modification_time')
    vm.spec.ClearField('labels')
    vm.spec.vmagent_version = ''
    return vmset_api_pb2.GetVmFromBackupResponse(vm=vmset_pb2.VM(meta=vm.meta, spec=vm.spec))


@vmset_service_bp.method('AcknowledgeEviction',
                         request_type=vmset_api_pb2.AcknowledgeEvictionRequest,
                         response_type=vmset_api_pb2.AcknowledgeEvictionResponse)
@error_utils.translate_app_errors
def acknowledge_eviction(req, auth_subject):
    """

    :type req: vmset_api_pb2.AcknowledgeEvictionRequest
    :param auth_subject:
    :rtype: vmset_api_pb2.AcknowledgeEvictionResponse
    """
    if not req.vm_id:
        raise exceptions.BadRequestError('No vm_id specified')
    if not req.qdm_res_id:
        raise exceptions.BadRequestError('No qdm_res_id specified')

    ctx = get_context()
    if not ctx.sec_policy.is_root(auth_subject.login):
        raise exceptions.ForbiddenError('user with login: {} has not permissions for AcknowledgeEviction'.format(
            auth_subject.login))

    pod_pb = ctx.pod_ctl.get_pod(req.vm_id)
    pod_set_pb = ctx.pod_ctl.get_pod_set(req.vm_id)
    vm = yputil.cast_pod_to_vm(
        pod_pb=pod_pb,
        pod_set_pb=pod_set_pb
    )

    current_vmagent_version = helpers.get_vmagent_version(pod_pb)
    pod_set_updates = {}
    pod_updates = {}

    if vm.spec.qemu.forced_node_id:
        nodes = list_free_nodes_action.run(ctx, vm.spec.qemu.node_segment)
        new_node_id = list_free_nodes_action.match_node(nodes, pod_pb.spec)

        if new_node_id is not None:
            vm.spec.qemu.forced_node_id = new_node_id
            log.info('Matched node: {} for pod: {}'.format(new_node_id, pod_pb.meta.id))
            hint = data_model.TPodSpec.TSchedulingHint()
            hint.node_id = new_node_id
            hint.strong = True
            pod_updates['/spec/scheduling/hints'] = helpers.yson_dumps_list_of_proto([hint])
            pod_updates['/labels/qyp_vm_forced_node_id'] = new_node_id
            pod_set_updates['/labels/qyp_vm_forced_node_id'] = new_node_id
        else:
            log.warning('Not able to find free node for pod: {}'.format(pod_pb.meta.id))

    if current_vmagent_version < config_action.MULTI_STORAGE_VMAGENT_VERSION:
        iss_proto = pod_pb.spec.iss
        pod_resources = yputil.get_iss_payload_pod_resources(iss_proto)
        vm_config_resource = pod_resources.get(config_action.VM_CONFIG_RESOURCE_NAME)
        if not vm_config_resource:
            raise exceptions.BadRequestError('VM has no {} resource: {}'.format(
                config_action.VM_CONFIG_RESOURCE_NAME, pod_resources.keys()))

        config_json_string = base64.b64decode(vm_config_resource.url.split('data:text/plain;charset=utf-8;base64,')[1])
        config_pb = vmagent_pb2.VMConfig()
        json_format.Parse(config_json_string, config_pb)

        config_pb.disk.type = vmagent_pb2.VMDisk.RAW
        config_pb.disk.delta_size = 0
        config_pb.disk.resource.rb_torrent = req.qdm_res_id
        config_pb.disk.path = '/'  # ?

        config_action.put_config_as_resource(iss_proto, config_pb)
        yputil.update_iss_payload_config_fingerprint(iss_proto, req.vm_id)
        pod_updates['/spec/iss'] = yputil.dumps_proto(iss_proto)
    else:
        backup_spec = ctx.qdm_client.get_revision_info(req.qdm_res_id)
        if not backup_spec:
            raise exceptions.BadRequestError('QDM not found resource with id: {}'.format(req.qdm_res_id))

        backup_action.rewrite_qemu_volumes_from_backup_spec(vm.spec.qemu, backup_spec)

        del pod_pb.spec.disk_volume_requests[:]
        yputil.cast_qemu_volumes_to_pod_disk_volume_requests(
            pod_id=vm.meta.id,
            vm_spec=vm.spec,
            disk_volume_requests=pod_pb.spec.disk_volume_requests)

        pod_resources = yputil.get_iss_payload_pod_resources(pod_pb.spec.iss)

        new_iss_proto = yputil.make_iss_payload(
            pod_id=vm.meta.id,
            vm_spec=vm.spec,
            yp_cluster=ctx.pod_ctl.yp_cluster,
            volumes=pod_pb.spec.disk_volume_requests,
            root_storage=yputil.get_iss_payload_root_storage_class(pod_pb.spec.iss),
            pod_resources=pod_resources
        )

        config_action.put_config_id_as_resource(new_iss_proto, config_action.gen_config_id())

        upds = {
            '/spec/disk_volume_requests': helpers.yson_dumps_list_of_proto(pod_pb.spec.disk_volume_requests),
            '/spec/iss': yputil.dumps_proto(new_iss_proto),
            '/annotations/qyp_vm_spec': helpers.yson_dumps_vm(vmset_pb2.VM(spec=vm.spec, meta=vm.meta))
        }
        pod_updates.update(upds)

    pod_cur_version = yputil.cast_attr_dict_to_dict(pod_pb.labels)['version']
    pod_set_cur_version = yputil.cast_attr_dict_to_dict(pod_set_pb.labels)['version']
    try:
        ctx.pod_ctl.update_pod_with_acknowledge_eviction(
            pod_id=pod_pb.meta.id, pod_version=pod_cur_version, pod_updates=pod_updates,
            pod_set_version=pod_set_cur_version, pod_set_updates=pod_set_updates
        )
    except yp_common.YtResponseError as e:
        _find_in_error_message = 'No eviction is currently requested for pod'
        if yp_common.validate_error_recursively(e.error, lambda error: _find_in_error_message in error.get('message')):
            return vmset_api_pb2.AcknowledgeEvictionResponse(result=vmset_api_pb2.NO_EVICTION)
        else:
            raise e

    return vmset_api_pb2.AcknowledgeEvictionResponse(result=vmset_api_pb2.DONE_EVICTION)


@vmset_service_bp.method('GetVmUsage',
                         request_type=vmset_api_pb2.GetVmUsageRequest,
                         response_type=vmset_api_pb2.GetVmUsageResponse)
@error_utils.translate_app_errors
def get_vm_usage(req, auth_subject):
    ctx = get_context()
    if not ctx.pod_ctl.check_write_permission(pod_id=req.vm_id, subject_id=auth_subject.login):
        raise exceptions.ForbiddenError('User {} has no access to vm {}'.format(auth_subject.login, req.vm_id))

    vm_fqdn = '{}.{}.yp-c.yandex.net'.format(req.vm_id, ctx.pod_ctl.yp_cluster.lower())
    vm_usage = ctx.heartbeat_client.get_vm_usage_stats(vm_fqdn)

    return vm_usage
