"""
ReConf - Hierarchical configurations building microframework.

https://st.yandex-team.ru/RUNTIMECLOUD-8180

"""
import logging

from infra.reconf.util import handled, symbols


class OptHandler(handled.KeysHandler):
    """
    Base class for config option handlers.

    Opt handler responsible for default values for declared keys and control
    inserting new values passed to conf object.

    """


class NodesHandler(OptHandler):
    """
    Nodes handler for ConfSet.

    By default accept any inserting items as is.

    """


class SubnodesHandler(OptHandler):
    """
    Subconfs handler, returned container type should be defined in conf's
    'subnodes' attribute.

    """
    @staticmethod
    def get_defaults():
        return {'children': None}

    def get_default_value(self, key):
        raise ConfOptAbsent

    def get_handled_value(self, key, val):
        return self._bound.subnodes(val, bound=self._bound)


class ConfOptAbsent(handled.HandledKeyAbsent):
    """ Raised by opt handlers when default opt should absent in conf. """


class ConfOptDenied(handled.HandledKeyDenied):
    """ Raised by opt handlers when opt denied for some reason. """


class BaseConf(object):
    """ Mixin providing essential things every conf should have. """
    opt_factory = None
    resolver = None

    def __init__(self, *args, bound=None, opt_factory=None, resolver=None,
                 shared=None):
        if opt_factory is not None:
            self.opt_factory = opt_factory
        elif bound is not None:
            self.opt_factory = bound.opt_factory

        if resolver is not None:
            self.resolver = resolver
        elif bound is not None:
            self.resolver = bound.resolver

        # dict shared among all confs, sort of global context
        if shared is not None:
            self.shared = shared
        elif bound is None:  # root conf
            self.shared = {}
        else:
            self.shared = bound.shared

        self._bound = bound

        super().__init__(*args)  # only args used for initial data

    def create_handler(self, handler_class):
        """ Alter opt handlers using factory (if provided), see OptFactory """
        if self.opt_factory is None:
            return super().create_handler(handler_class)

        return self.opt_factory.create_handler(handler_class, self)

    def get_upward(self, type_=None):
        """ Search upwards and return first conf with desired type """
        conf = self._bound

        if type_ is not None:
            while not isinstance(conf, type_):
                conf = conf._bound

        return conf


class DictConf(BaseConf, handled.HandledDict):
    """
    Conf is a dict with keys controlled by handlers. Each handler declare keys
    it responsible for. Any key-value pair written to conf processed by
    according handler. Default value is set when no initial value provided for
    key.

    """


class ListConf(BaseConf, handled.HandledList):
    """
    Conf is a list with items controlled by handler. Default value is set when
    no initial items provided.

    """


class ConfSet(DictConf):
    """
    Configurations set.

    Used as subnodes container for another conf or as standalone thing to
    represent bunch (forest) of confs.

    Turns into cortesian product out of passed initial key-value pairs and
    pre-defined branches (conf classes) when initialized.

    """
    branches = ()
    default_handler = NodesHandler

    def __init__(self, *args, **kwargs):
        super().__init__(**kwargs)

        self.__has_endpoints = False  # has non-confs among children
        self.__has_nodes = False  # has confs among children

        self.init_nodes(dict(*args))

    def add_endpoint(self, name, body):
        logging.debug('Adding endpoint ' + name)
        self[name] = body
        self.__has_endpoints = True

    def add_node(self, name, body):
        logging.debug('Adding node ' + name)
        self[name] = body
        self.__has_nodes = True

    def build(self):
        """
        Build options for contained confs.

        """
        for name, body in self.items():
            if isinstance(body, ConfNode):
                logging.debug('Build opts for ' + name)
                body.build()

        return self

    def create_node(self, origin, node_class, content):
        return node_class(
            content,
            bound=self,
            origin=origin,
            opt_factory=self.opt_factory,
            resolver=self.resolver,
        )

    def get_initial_branches(self):
        return self.branches

    def get_branches(self):
        if not self.branches and self._bound is not None:
            if self._bound.branches:
                return self._bound.branches
            else:
                return (self._bound.__class__,)

        return self.get_initial_branches()

    def get_endpoint_id(self, origin, conf_class):
        return str(origin) + ':' + conf_class.__name__

    get_node_id = get_endpoint_id

    def has_endpoints(self):
        """
        Contain non-confs (endpoints) or not.

        """
        return self.__has_endpoints

    def has_nodes(self):
        """
        Contain confs (non-endpoints) or not.

        """
        return self.__has_nodes

    def init_nodes(self, initial_dict):
        """ Produce branch nodes out of initial items """
        for origin, content in initial_dict.items():
            for conf_class in self.get_branches():
                if isinstance(content, dict):  # conf
                    node = self.create_node(origin, conf_class, content)
                    try:
                        self.add_node(
                            self.get_node_id(origin, conf_class),
                            node,
                        )
                    except ConfOptDenied:
                        logging.debug('Node discarded')
                        continue
                else:
                    try:
                        self.add_endpoint(
                            self.get_endpoint_id(origin, conf_class),
                            content,
                        )
                    except ConfOptDenied:
                        logging.debug('Endpoint discarded')
                        continue

    def iterate_nodes(self):
        """
        Yield tuples (name, body) for confs and all their subconfs in
        post-order (depth-first).

        """
        yield from iterate_depth_first(self)


class ConfNode(DictConf):
    """ Base class for conf nodes """
    doc_url = None  # docs url string

    branches = ()
    default_handler = None  # disabled - handlers added in two stages

    opt_handlers = ()
    subnodes = ConfSet
    subnodes_handler = SubnodesHandler

    # validate class essential attributes (cheaper than object's)
    validate_class = True
    validate_class_doc_url = False  # disabled by default (project specific)
    validate_class_docstring = True  # it's nice to have docstrings

    def __init__(self, *args, origin=None, **kwargs):
        super().__init__(**kwargs)

        self.initial_dict = dict(*args)
        self.name = None
        self.origin = origin

        if self._bound is not None:
            self.name = self._bound.get_node_id(self.origin, self.__class__)

        logging.debug('Init node ' + (self.name or 'root'))
        self.init_subnodes()

    def init_subnodes(self):
        self.add_handler(self.subnodes_handler, self.initial_dict)

    def __init_subclass__(subclass, **kwargs):
        super().__init_subclass__(**kwargs)
        if subclass.validate_class:  # may be entirely disabled
            # Use usual method for validation, descendants may (and should)
            # override and extent it.
            subclass.validate()

    def build(self):
        """
        Build conf options.

        """
        # TODO: set default handler here

        for name in self.opt_handlers:
            self.add_handler(self.__getattribute__(name), self.initial_dict)

        if 'children' in self:
            self['children'].build()

        return self

    def has_endpoints(self):
        """
        Exists non-subconfs (endpoints) among children or not.

        """
        if 'children' in self:
            return self['children'].has_endpoints()

        return False

    def has_subnodes(self):
        """
        Exists subconfs (non-endpoints) among children or not.

        """
        if 'children' in self:
            return self['children'].has_nodes()

        return False

    def iterate_subnodes(self):
        """
        Yield all subconfs, for details see ConfSet.iterate_nodes.

        """
        if 'children' in self:
            yield from self['children'].iterate_nodes()

    @classmethod
    def validate(cls):
        if cls.validate_class_doc_url:
            cls.validate_doc_url()

        if cls.validate_class_docstring:
            cls.validate_docstring()

    @classmethod
    def validate_doc_url(cls):
        # ensure attr present in definition of validated class (no inheritance)
        if 'doc_url' not in cls.__dict__:
            raise ValidationError('Doc url absent', cls=cls)

    @classmethod
    def validate_docstring(cls):
        if cls.__doc__ is None:
            raise ValidationError('Docstring absent', cls=cls)


class OptFactory(object):
    """
    Global object which follows conf building and allow to override conf opts
    (redefine inherited conf opt handlers).

    This comes from fact that sometimes whole confset should be inherited and
    each entry in this set is already have it's own inheritance hierarchy.

    """
    def create_handler(self, handler_class, conf):
        return handler_class(bound=conf)


class ValidationError(Exception):
    """
    Base class for all validation errors

    """
    def __init__(self, message, *args, cls=None, **kwargs):
        if cls is not None:
            message = symbols.symbol_full_name(cls) + ':: ' + message

        super().__init__(message, *args, **kwargs)


def iterate_depth_first(forest, subnodes_key='children'):
    """
    Yield tuples (name, body) for all confs in the forest in post-order.

    """
    # sort items just to have reproducible results order
    stack = [((k, v) for k, v in sorted(forest.items()))]
    stash = []

    while True:
        try:
            name, body = next(stack[-1])
        except StopIteration:
            if stash:
                yield stash.pop()  # emit previous level node
                stack.pop()  # throw away exhausted generator
                continue
            else:
                break

        if body is None:  # endpoint
            continue

        if not body and body.__class__ is dict:  # trimmed conf
            continue

        try:
            stack.append((k, v) for k, v in sorted(body[subnodes_key].items()))
            stash.append((name, body))
        except KeyError:
            yield name, body
