"""
Illustration for default (cortesian product) mode for ReConf conf builder.

"""
import json

from infra import reconf


# desired confs tree
TREE = {
    'top.example.com': {
        'children': {
            'mid-one.example.com': {
                'children': {
                    'back-one.example.com': None,
                    'back-two.example.com': None,
                },
            },
            'mid-two.example.com': {
                'children': {
                    'back-one.example.com': None,
                },
            },
        }
    }
}


class Logging(reconf.OptHandler):
    """
    Handler for logging opts.

    """
    def get_defaults(self):
        return {'log_level': 'INFO', 'log_file': '/var/log/proxy.log'}


class LoggingDebug(Logging):
    """
    Same as above, but logging level redefined.

    """
    def get_defaults(self):
        defaults = super().get_defaults()
        defaults['log_level'] = 'DEBUG'
        return defaults


class Timeouts(reconf.OptHandler):
    """
    Handler for timeout opts.

    """
    def get_defaults(self):
        return {'conn_timeout': 1, 'read_timeout': 4}


class http(reconf.ConfNode):
    """
    Http conf.

    `opt_handlers` is a tuple of attribute names, each such attrubute hold
    handler type which will be instantiated on build stage.

    """
    opt_handlers = ('logging', 'timeouts')
    logging = Logging
    timeouts = Timeouts


class https(http):
    """
    Https conf. All the same as for http, but logging opts handler replaced.

    """
    logging = LoggingDebug


class ProxyConfSet(reconf.ConfSet):
    """
    Set of conf types. As output on __init__ stage we will get cortesian
    product of children in the initial structure and branches defined here.

    """
    branches = (http, https)


def main():
    confs = ProxyConfSet(TREE)
    confs.build()

    print(json.dumps(confs, indent=3, sort_keys=True))
