"""
Resolvers provide external data to the conf builders.

"""
import json
import requests
import requests.adapters
import urllib3.util.retry


class AbstractResolver(object):
    """
    Base class for resolvers.

    Any resolver may contain hierarchy of subresolvers; Each resolver maintain
    cache for it's queries type. Resolver imitate `dict` to give access to it's
    subresolvers.

    """
    use_cache = True
    use_fetch = True

    def __init__(self):
        self.__cache = {}
        self.__sources = {}  # subresolvers

    def __iter__(self):
        return self.__sources.__iter__()

    def __delitem__(self, src):
        self.__sources.__delitem__(src)

    def __contains__(self, src):
        return self.__sources.__contains__(src)

    def __getitem__(self, src):
        return self.__sources.__getitem__(src)

    def __setitem__(self, src, val):
        self.__sources.__setitem__(src, val)

    def dump_cache(self):
        dump = {}

        if self.__cache:
            dump['cache'] = self.__cache

        if self.__sources:
            for src in self.__sources:
                cache = self.__sources[src].dump_cache()
                if cache:
                    dump.setdefault('sources', {})[src] = cache

        return dump

    def dump_cache_to_file(self, filename):
        with open(filename, 'w') as f:
            return json.dump(self.dump_cache(), f, indent=2, sort_keys=True)

    def load_cache(self, dump):
        self.__cache = dump.get('cache', {})

        for src in self.__sources:
            src_dump = dump.get('sources', {}).get(src, {})
            self.__sources[src].load_cache(src_dump)

    def load_cache_from_file(self, filename):
        with open(filename) as f:
            self.load_cache(json.load(f))

    def get_cache_key(self, query):
        return query

    def resolve(self, query):
        if self.use_cache:
            key = self.get_cache_key(query)

            if key not in self.__cache:
                if self.use_fetch:
                    value = self.resolve_query(query)

                    # some resolvers may return generators
                    if hasattr(value, '__next__'):
                        self.__cache[key] = tuple(value)
                    else:
                        self.__cache[key] = value
                else:
                    raise KeyError('No cached value for ' + key)

            return self.__cache[key]

        if self.use_fetch:
            return self.resolve_query(query)

        raise RuntimeError('Unable to resolve ' + self.get_cache_key(query))

    def resolve_query(self, query):
        raise NotImplementedError

    def set_mode(self, use_cache=None, use_fetch=None, propagate=False):
        if use_cache is not None:
            self.use_cache = use_cache

        if use_fetch is not None:
            self.use_fetch = use_fetch

        if propagate:
            for src in self.__sources:
                self.__sources[src].set_mode(
                    use_cache=self.use_cache,
                    use_fetch=self.use_fetch,
                    propagate=propagate,
                )


class HttpResolver(AbstractResolver):
    """ Base class for HTTP resolvers """
    api_url = "http://localhost:80"

    retry_attempts = 4
    retry_backoff_factor = 0.5
    retry_methods = frozenset(('GET', 'HEAD', 'POST'))
    retry_statuses = frozenset((500, 502, 503, 504))

    def __init__(self, *args, api_url=None, http_session=None, **kwargs):
        super().__init__(*args, **kwargs)

        if http_session is None:
            self.http_session = requests.Session()

            adapter = requests.adapters.HTTPAdapter(
                max_retries=urllib3.util.retry.Retry(
                    backoff_factor=self.retry_backoff_factor,
                    connect=self.retry_attempts,
                    read=self.retry_attempts,
                    total=self.retry_attempts,
                    allowed_methods=self.retry_methods,
                    status_forcelist=self.retry_statuses,
                ),
            )

            self.http_session.mount('http://', adapter)
            self.http_session.mount('https://', adapter)
        else:
            self.http_session = http_session

        if api_url is not None:
            self.api_url = api_url
