"""
Collection of containers with handled access.

"""


class HandledKeyAbsent(Exception):
    """ Raised by keys handlers when default key should not be inserted. """


class HandledKeyDenied(Exception):
    """ Raised by keys handlers when key denied for some reason. """


class HandlerAbsentError(Exception):
    """ Raised when no handler found for a key. """


class HandlerConflictError(Exception):
    """ Raised when more than one handler declared to handle same key. """


class Handler(object):
    """
    Provide defaults and implement logic for new values.

    Should have full access to bound container to be able to perform actions
    on values managed by another handlers.
    """
    def __init__(self, bound=None):
        self._bound = bound
        self._defaults = self.get_defaults()

    def __repr__(self):
        return self.__class__.__name__ + '()'

    @staticmethod
    def get_defaults():
        return ()

    def get_default_value(self, key):
        return self._defaults[key]

    def get_handled_value(self, key, val):
        return val


class KeysHandler(Handler):
    """
    Keys handler for HandledDict.

    Should be able to handle more than just one key.
    """
    @staticmethod
    def get_defaults():
        return {}


class DefaultKeysHandler(KeysHandler):
    """
    Default keys handler for HandledDict.

    Raise HandlerAbsentError for any access to bound HandledDict.

    """
    def get_default_value(self, key):
        raise HandlerAbsentError(key)

    def get_handled_value(self, key, val):
        raise HandlerAbsentError(key)


class HandledDict(dict):
    """
    Dict with handled write access.

    `handlers` is a tuple and should contain subclasses (not instances!) of
    KeysHandler class.

    `default_handler` is a subclass of KeysHandler to handle keys without
    dedicated handlers.

    """
    default_handler = DefaultKeysHandler
    handlers = ()

    def __init__(self, *args, **kwargs):
        dict.__init__(self)

        initial_data = dict(*args, **kwargs)
        self.__handled_keys = {}

        # process initial data by dedicated handlers, set default values
        for name in self.handlers:
            self.add_handler(self.__getattribute__(name), initial_data)

        if self.default_handler is None:
            # disabled, remaining data will be simply ignored
            return

        self.__default_handler = self.create_handler(self.default_handler)

        for key, val in initial_data.items():
            dict.__setitem__(
                self, key,
                self.__default_handler.get_handled_value(key, val)
            )

    def __setitem__(self, key, val):
        # Do not bypass HandledKeyAbsent here: deliberate insertion should not
        # be ignored. HandledKeyDenied or other suitable exception should be
        # raised in handler instead to give caller a chance to handle situation
        # properly
        val = self.get_key_handler(key).get_handled_value(key, val)
        dict.__setitem__(self, key, val)

    def __repr__(self):
        return self.__class__.__name__ + '(' + dict.__repr__(self) + ')'

    def add_handler(self, handler_class, initial_data):
        if handler_class is None:  # may be disabled
            return

        handler = self.create_handler(handler_class)

        for key in handler._defaults:
            if key in self.__handled_keys:
                raise HandlerConflictError(key)

            self.__handled_keys[key] = handler
            try:
                if key in initial_data:
                    val = handler.get_handled_value(key, initial_data.pop(key))
                else:
                    val = handler.get_default_value(key)
                dict.__setitem__(self, key, val)
            except HandledKeyAbsent:  # handled, but should absent
                pass

    def create_handler(self, handler_class):
        return handler_class(bound=self)

    def get_key_handler(self, key):
        try:
            return self.__handled_keys[key]
        except KeyError:
            return self.__default_handler

    def update(self, iterable=None, /, **kwargs):
        if iterable is not None:
            try:
                for key, val in iterable.items():
                    self[key] = val
            except AttributeError:
                for key, val in iterable:
                    self[key] = val

        for key, val in kwargs.items():
            self[key] = val


# TODO: implement rest modification methods
class HandledList(list):
    """
    List with handled write access.

    """
    handler = Handler

    def __init__(self, *args):
        list.__init__(self)
        self._handler = self.create_handler(self.handler)
        self._defaults = self._handler.get_defaults()

        if args:
            if len(args) > 1:
                raise TypeError(
                    self.__class__.__name__ + '() takes at most 1 argument ('
                    + str(len(args)) + ' given)'
                )
            self.extend(args[0])
        else:
            list.extend(self, self._defaults)

    def __repr__(self):
        return self.__class__.__name__ + '(' + list.__repr__(self) + ')'

    def __setitem__(self, key, val):
        val = self._handler.get_handled_value(key, val)
        list.__setitem__(self, key, val)

    def append(self, val):
        list.append(self, self._handler.get_handled_value(None, val))

    def create_handler(self, handler_class):
        return handler_class(bound=self)

    def extend(self, it):
        for val in it:
            list.append(self, self._handler.get_handled_value(None, val))

    def insert(self, key, val):
        list.insert(self, key, self._handler.get_handled_value(key, val))
