from infra.reconf import ConfNode, ConfSet, OptFactory
from infra.reconf_juggler import opts


class CheckSet(ConfSet):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.shared.setdefault('origin_idx', {})

    def add_node(self, name, body):
        super().add_node(name, body)

        # Not in reconf because abstract conf tree may contain nodes with same
        # name in different branches; but for Juggler it's impossible and
        # should be checked. Also, using this index we can keep relations
        # between checks with same origin (for unreach services at least)
        if name in self.shared['origin_idx'].setdefault(body.origin, {}):
            raise Exception('Check id duplicates: ' + name)

        self.shared['origin_idx'][body.origin][name] = body

    def get_endpoint_id(self, origin, check_class):
        service_name = check_class.provides(is_node=False)

        if service_name is None:  # EVENTS groups, for example
            return str(origin)

        return str(origin) + ':' + service_name

    def get_node_id(self, origin, check_class):
        return str(origin) + ':' + check_class.provides(is_node=True)


class Check(ConfNode):
    """
    Base class for Juggler checks

    """
    opt_handlers = (
        '_tags',  # may be used for other opts tuning
        '_meta',  # used for aggregators tuning
        '_active',
        '_aggregator',
        '_ctime',
        '_description',
        '_flap',
        '_mtime',
        '_namespace',
        '_notifications',
        '_options',
        '_refresh_time',
        '_ttl',
    )
    subnodes = CheckSet

    # option handlers classes; as attrs to be redefinable in derivatives
    _active = opts.actives.ActiveHandler
    _aggregator = opts.aggregators.AggregatorHandler
    _ctime = opts.ctimes.CtimeHandler
    _description = opts.descriptions.DescriptionHandler
    _flap = opts.flaps.FlapHandler
    _meta = opts.metas.MetaHandler
    _mtime = opts.mtimes.MtimeHandler
    _namespace = opts.namespaces.NamespaceHandler
    _notifications = opts.notifications.NotificationsHandler
    _options = opts.options.OptionsHandler
    _refresh_time = opts.refresh_times.RefreshTimeHandler
    _tags = opts.tags.TagsHandler
    _ttl = opts.ttls.TtlHandler

    # https://wiki.yandex-team.ru/sm/juggler/check-dependencies/#napraktike
    _unreach_service_classes = ()
    _unreach_service_hold = None

    category = None  # any string; will add 'category_{}' tag if not None
    level_autotags = True  # level_root and level_leaf tags will be added
    maintainers = ()  # typical use to define staff groups here

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.shared.setdefault('origin_idx', {}).setdefault(self.origin, {})

    def build(self):
        super().build()

        if 'tags' in self:
            # Tags container is simple list at this moment, to keep it sorted
            # we should set new value through handler (sort implemented in opt
            # handler). Pop/set will be removed in the future and tags will be
            # modified directly (when tags container become smarter).
            tags = self.pop('tags')
            tags.extend(map(lambda x: 'maintainer_' + x, self.maintainers))

            if self.category is not None:
                tags.append('category_' + self.category)

            if self.level_autotags:
                if not self.has_subnodes():
                    tags.append('level_leaf')
                if not self._bound or not self._bound._bound:
                    tags.append('level_root')

            self['tags'] = tags

        return self

    def get_aggr_unreach_mode(self):
        """
        Children's dependency mode.

        https://wiki.yandex-team.ru/sm/juggler/aggregation-modules/#zavisimostimezhduproverkami

        """
        return 'skip'  # as most sensible

    # It seems depends should be defined here, in check; this makes
    # aggregators more resusable and flexible. At least for now.
    def get_aggr_unreach_service(self):
        """
        Children's dependency services.

        https://wiki.yandex-team.ru/sm/juggler/aggregation-modules/#zavisimostimezhduproverkami

        """
        ret = []

        for check_class in self._unreach_service_classes:
            for _, sibling in self.shared['origin_idx'][self.origin].items():
                if self is sibling:
                    continue

                if isinstance(sibling, check_class):
                    depend = {'check': ':' + sibling.provides()}
                    if self._unreach_service_hold is not None:
                        depend['hold'] = self._unreach_service_hold

                    ret.append(depend)

        return ret

    def get_raw_events_total(self):
        if self.resolver is None:
            return None

        if 'children' not in self:
            # aggregate w/o children monitor event with self origin:provides
            return 1

        total = 0
        for name, body in self['children'].items():
            if body is None:  # endpoint
                total += self.resolver['juggler']['instances_count'].resolve(
                    name.rsplit(':', 1)[0])

        return total

    @classmethod
    def provides(cls, is_node=True):
        """
        Return check's 'service' name.

        """
        return cls.__name__


class CheckOptFactory(OptFactory):
    """
    Provide check opt handlers.

    Namespace and notifications historically are part of juggler check and
    can't be overrided when whole CheckSet subclassed. This class provides
    opt handlers for check builder, hence we have a place to put extra logic to
    redefine checks opts and tweak checks behavior.

    """
