import argparse
import json
import logging

from library.python.svn_version import svn_revision

from infra.reconf.util.symbols import symbol_full_name
from infra.reconf_juggler.declarative import DeclaredCheckSet
from infra.reconf_juggler.resolvers import RootResolver
from infra.reconf_juggler.util.jsdk import tree2jsdk


COMMON_BUILD_TARGETS = (
    'initial_tree',
    'checks_tree',
    'checks_full',
    'jsdk_dump',
)


class JugglerChecksBuilder(object):
    """
    Base class for ReConf-Juggler based checks builders.
    Nothing fancy, minimizing biolerplate code.

    IMPORTANT: build_* method names are reserved for target builders.

    """
    build_targets = ('initial_data',) + COMMON_BUILD_TARGETS
    default_target = 'checks_full'

    resolver = None

    loglevels = ('CRITICAL', 'ERROR', 'WARNING', 'INFO', 'DEBUG')
    default_loglevel = 'INFO'

    def __init__(self, resolver=None):
        if resolver is None:
            self.resolver = RootResolver()
        else:
            self.resolver = resolver

    def build(self, target=None):
        """
        Simple linear build targets dispatcher: resulting data passed from one
        target builder to another in sequence defined in `build_targets` attr.

        """
        if target is None:
            target = self.default_target

        if target not in self.build_targets:
            raise RuntimeError("Unsupported build target {}".format(target))

        data = None

        for tgt in self.build_targets:
            logging.info('Building {}'.format(tgt))
            data = self.__getattribute__('build_' + tgt)(data)

            if tgt == target:
                break

        return data

    def build_initial_data(self, unused):
        """
        Return initial data.
        Most suitable to control initial data (often fetched by resolvers from
        third-part services like wall-e)

        """
        return {}  # may be simply ignored when method doesn't overridden

    def build_initial_tree(self, initial_data):
        """
        Return initial structure.
        Most suitable for debug reference structure.

        """
        return initial_data

    def build_checks_tree(self, initial_tree):
        """
        Return checks tree.
        Most suitable for debug/diff full checks tree.

        """
        raise NotImplementedError

    def build_checks_full(self, checks_tree):
        """
        Return final checks tree with all opts built.
        Most suitable for debug/diff final checks.

        """
        return checks_tree.build()

    def build_jsdk_dump(self, checks_tree):
        """
        Return juggler-sdk formatted checks structure.
        Suitable for deploy:
            `./builder --target jsdk_dump | juggler-sdk load --mark <mark>`

        """
        return tree2jsdk(checks_tree)

    def dump(self, data):
        """
        Return serialized checks.

        """
        return json.dumps(data, indent=3, sort_keys=True)

    def init_cli(self, app_desc=None):
        """
        Init command line mode related stuff.

        """
        logging.basicConfig(
            format='[%(asctime)s] %(levelname)s %(message)s',
            level=logging.DEBUG
        )

        self.argparser = argparse.ArgumentParser(description=app_desc)
        self.argparser.add_argument(
            '--dump-resolved', default=None, metavar='FILE', type=str)
        self.argparser.add_argument(
            '--version', action='version', version=str(svn_revision()))
        self.argparser.add_argument(
            '--load-resolved', default=None, metavar='FILE', type=str)
        self.argparser.add_argument(
            '--loglevel', default=self.default_loglevel,
            type=self.set_loglevel, choices=self.loglevels)
        self.argparser.add_argument(
            '--target', type=str, default=None, choices=self.build_targets)

    def run(self, app_desc=None):
        """
        Entry point for commandline mode.

        """
        self.init_cli(app_desc=app_desc)
        self.args = self.argparser.parse_args()

        if self.args.load_resolved is not None:
            self.resolver.load_cache_from_file(self.args.load_resolved)

        print(self.dump(self.build(self.args.target)))

        if self.args.dump_resolved is not None:
            self.resolver.dump_cache_to_file(self.args.dump_resolved)

    def set_loglevel(self, level_name):
        logging.getLogger().setLevel(level_name)

        return level_name


class DeclaredChecksBuilder(JugglerChecksBuilder):
    """
    Build checks tree out of structure with declared class names for checks.

    """
    default_check_class = 'infra.reconf_juggler.Check'

    def __init__(self, names_map=None, **kwargs):
        if names_map is None:
            self._names_map = self.get_names_map()
        else:
            self._names_map = names_map

        super().__init__(**kwargs)

    def build_initial_tree(self, initial_data, names_map=None, **kwargs):
        """
        Return checks tree with resolved names.

        """
        if names_map is None:
            names_map = self._names_map

        self.resolve_names(initial_data, names_map)

        return initial_data

    # TODO: redefinable default_check_class by kwargs (see build_initial_tree)
    def build_checks_tree(self, initial_tree):
        """
        Return checks tree with initialized nodes.

        """
        self.resolve_classes(initial_tree, self.default_check_class)

        return DeclaredCheckSet(initial_tree)

    def build_checks_full(self, checks_tree):
        """
        Return final checks tree with all opts built.

        """
        return checks_tree.build()

    def get_names_map(self):
        """
        Return keywords dict for names formatter.

        """
        return {}

    def resolve_classes(self, tree, parent_class):
        """
        Propagate classes (declared in ['meta']['reconf']['class']) from
        parents to children. Default class used if no class declared in check
        hierarchy.

        """
        for name, body in tree.items():
            if body is None or not body:  # endpoint or trimmed check
                continue

            defined_class = body.setdefault(
                'meta', {}).setdefault(
                'reconf', {}).setdefault(
                'class', parent_class)

            logging.debug('Setting class for {} ({})'.format(
                name, defined_class))

            if 'children' in body:
                self.resolve_classes(body['children'], defined_class)

    def resolve_names(self, tree, names_map):
        """
        Evaluate check names. Standard pythons's format() used for template
        engine.

        For more info see `python3 -c "help('FORMATTING')"`

        """
        for name, body in tuple(tree.items()):
            new_name = name.format(**names_map)

            if new_name != name:
                logging.debug('Renaming {} => {}'.format(name, new_name))
                if new_name in tree:
                    raise RuntimeError('Check with such name already exist')

                tree[new_name] = body
                del tree[name]

            if body is not None and 'children' in body:
                self.resolve_names(body['children'], names_map)


class ProxyChecksBuilder(JugglerChecksBuilder):
    """
    Proxy build request to another builders and merge result.

    """
    build_targets = COMMON_BUILD_TARGETS
    builders = ()

    def build(self, target=None):
        # synchronize target to avoid mix of different merged itargets caused
        # by different default target in builders
        if target is None:
            target = self.default_target

        merged_dict = {}
        merged_list = []  # lists used in jsdk_dump target (at least)

        for builder_class in self.builders:
            logging.info('Build target {} using {}'.format(
                target,
                symbol_full_name(builder_class)
            ))

            data = builder_class(resolver=self.resolver).build(target)

            if isinstance(data, dict):
                merged_dict.update(data)
            elif isinstance(data, list):
                merged_list.extend(data)
            else:
                raise RuntimeError("Unsupported format for a build result")

        if merged_dict and merged_list:
            raise RuntimeError("Different result types recieved from builders")

        return merged_dict if merged_dict else merged_list
