"""
Workflow and usage examples for ReConf-Juggler framework
https://st.yandex-team.ru/RUNTIMECLOUD-8171

"""
from infra.reconf_juggler import CheckOptFactory, CheckSet, checks
from infra.reconf_juggler.builders import JugglerChecksBuilder
from infra.reconf_juggler.opts import actives, aggregators, namespaces
from infra.reconf_juggler.trees import NodeSet
from infra.reconf_juggler.trees.locations import Locations


RTC_GROUPS = {
    'WALLE%PROD@prj=rtc': None,
    'WALLE%PROD@prj=rtc-yabs': None,
    'WALLE%PROD@prj=rtc-mtn': None,
}

YP_GROUPS = {
    'man': {'children': {'WALLE%PROD@prj=yp-man': None}, 'tags': ['grp_man']},
    'vla': {'children': {'WALLE%PROD@prj=yp-vla': None}, 'tags': ['grp_vla']},
}


# Define custom checks options (any option may be subclassed)
class CustomOptActiveSSH(actives.ssh):
    # May be transformed on build stage by the methods of this class instance
    @staticmethod
    def get_defaults():
        defaults = actives.ssh.get_defaults()
        defaults['active_kwargs']['timeout'] = 42
        return defaults


# Checks also may (and should) be subclassed to create new checks types
class CustomCheckSSH(checks.ssh):
    """
    Define own SSH check

    """
    # Handful of opts for short illustration
    opt_handlers = ('_meta', '_active', '_aggregator', '_tags')
    _active = CustomOptActiveSSH
    _aggregator = aggregators.SelectiveAggregator
    # set all derivatives from AbstractUnreachCheck, as depends (per instance)
    _unreach_service_classes = (checks.AbstractUnreachCheck,)


# Check sets may be subclassed and reused too
class RTCChecksSet(CheckSet):
    branches = (
        checks.META,
        # checks.UNREACHABLE,  # commented for shortage, feel free to enable
        CustomCheckSSH,
    )


# Define optional conf factory
# Yet another way to customize checks: override opts using opts factories
class MyShinyNamespace(namespaces.NamespaceHandler):
    def get_default_value(self, key):
        return 'bar' if self._bound.has_subnodes() else 'foo'


class OptFactory(CheckOptFactory):
    def create_handler(self, handler_class, check):
        if issubclass(handler_class, namespaces.NamespaceHandler):
            return super().create_handler(MyShinyNamespace, check)

        return super().create_handler(handler_class, check)


# Builder as a class - just to minimize boilerplate code
class Builder(JugglerChecksBuilder):
    def build_initial_tree(self, initial_data):
        initial_tree = {}
        initial_tree.update(
            Locations(
                {'rtc': {'children': RTC_GROUPS, 'tags': ['prj_rtc']}},
                resolver=self.resolver,
            ),
        )
        initial_tree.update(
            NodeSet(
                {'yp': {'children': YP_GROUPS, 'tags': ['prj_yp']}},
                resolver=self.resolver,
            )
        )

        return initial_tree

    def build_checks_tree(self, initial_tree):
        return RTCChecksSet(
            initial_tree,
            opt_factory=OptFactory(),
            resolver=self.resolver,
        )


def main():
    Builder().run(app_desc=__doc__)
