"""
Collection of Juggler aggregators.

https://wiki.yandex-team.ru/sm/juggler/aggregation-modules

"""
from infra.reconf_juggler.opts import CheckOptHandler
from infra.reconf_juggler.util import quorum


class AggregatorHandler(CheckOptHandler):
    allow_empty_groups = False
    downtimes_mode = None  # force_ok|skip|ignore
    nodata_mode = None  # force_ok|force_crit|force_warn|skip

    @staticmethod
    def get_defaults():
        return {
            'aggregator': 'logic_or',
            'aggregator_kwargs': None,
        }

    def get_default_value(self, key):
        val = super().get_default_value(key)

        if key == 'aggregator_kwargs':
            if val is None:
                val = {}

            if not self._bound.has_subnodes():
                # unreach works over raw events (meaningless in metaggregators)
                unreach_service = self._bound.get_aggr_unreach_service()
                if unreach_service:
                    val['unreach_service'] = list(unreach_service)
                    val['unreach_mode'] = self._bound.get_aggr_unreach_mode()

                if self.allow_empty_groups and self._bound.has_endpoints():
                    val['allow_empty_groups'] = True

            if self.downtimes_mode is not None:
                val['downtimes_mode'] = self.downtimes_mode

            if self.nodata_mode is not None:
                val['nodata_mode'] = self.nodata_mode

        return val if val else None

    def get_handled_value(self, key, val):
        return val


class DescriptionMismatch(AggregatorHandler):
    """
    Crit when children's discriptions are different.

    """
    def get_default_value(self, key):
        if key == 'aggregator':
            return 'description_mismatch'

        return super().get_default_value(key)


class LogicAnd(AggregatorHandler):
    """
    Crit when all children in crit state.

    """
    def get_default_value(self, key):
        if key == 'aggregator':
            return 'logic_and'

        return super().get_default_value(key)


class LogicOr(AggregatorHandler):
    """
    Most basic aggregator. Used mostly for metachecks (aggregator over
    aggregators where usually no logic required)

    """
    def get_default_value(self, key):
        if key == 'aggregator':
            return 'logic_or'

        return super().get_default_value(key)


class TimedMoreThanLimitIsProblem(AggregatorHandler):
    """
    Most feature rich aggregator. Used mostly for leaf checks (over raw
    events).

    """
    # Percent thresholds will be tuned, set to None to disable
    quorum_calc = quorum.Calculator(init_builtin_tables=True)

    def get_default_value(self, key):
        if key == 'aggregator':
            return 'timed_more_than_limit_is_problem'

        val = super().get_default_value(key)

        if key == 'aggregator_kwargs':
            if val is None:
                val = {}

            val['limits'] = self.get_default_limits()

            if self.quorum_calc is not None:
                raw_events_total = self._bound.get_raw_events_total()
                if raw_events_total is not None:
                    for limit in val['limits']:
                        self.tune_limit_thresholds(limit, raw_events_total)

        return val

    def get_default_limits(self):
        return [
            {
                'crit': '5%',
                'day_end': 7,
                'day_start': 1,
                'time_end': 23,
                'time_start': 0,
                'warn': '3%',
            },
        ]

    def get_tuned_threshold(self, threshold, amount):
        if threshold.endswith('%'):  # absolute values ignored
            share = float(threshold[:-1])  # cut percent sign

            if share <= 50:  # meaningles on larger share
                quorum = self.quorum_calc.calculate(amount, 100 - share)
                threshold = str(round(quorum['QPRCI'] * 100, 4)) + '%'

        return threshold

    def tune_limit_thresholds(self, limit, amount):
        limit['crit'] = self.get_tuned_threshold(limit['crit'], amount)
        limit['warn'] = self.get_tuned_threshold(limit['warn'], amount)

        if limit['warn'] == limit['crit']:
            limit['warn'] = '0%'


class SelectiveAggregator(AggregatorHandler):
    """
    Return apropriate aggregator depending of tree level.

    """
    leaf_aggregator = TimedMoreThanLimitIsProblem
    meta_aggregator = LogicOr  # crit when any of children is crit

    def __new__(cls, *args, bound=None, **kwargs):
        if bound.has_subnodes():
            aggr_class = cls.meta_aggregator
        else:
            aggr_class = cls.leaf_aggregator

        return aggr_class(*args, bound=bound, **kwargs)
