import io
import itertools
import logging
import os.path
import requests
import walle_api
import yaml
import zipfile

from infra.reconf.resolvers import AbstractResolver, HttpResolver
from infra.ya_salt.lib import envutil


class AbstractHostmanResolver(HttpResolver):
    """
    Common stuff for Hostman resolvers.

    """
    # We keep MSK servers on top because it's final deploy region, so all other
    # regions (usually) already running unit and it may be safely monitored.
    # Probably unit spec should support section for such purposes (disabling
    # aggregates/alerts and so on): HOSTMAN-777.
    hm_servers_order = ('msk', 'man')
    hostman_servers = (f'{s}:8080' for s in itertools.chain(*(
        envutil.PROD_LOCATIONS_SETTINGS[l][envutil.REPO_HOSTS] for l in hm_servers_order)))

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.__api_url = None

    @property
    def api_url(self):
        if self.__api_url is None:
            # we don't randomize servers here to have reproducible results
            # we don't use existing session to avoid it's configured retries
            logging.info('Looking for a hostman API server')

            for server in self.hostman_servers:
                try:
                    if requests.get(f'http://{server}/unistat', timeout=4).ok:
                        self.__api_url = f'http://{server}/v1/'
                        break
                except requests.exceptions.ConnectTimeout as e:
                    logging.error('Failed to reach %s: %s', server, e)
                    continue

            if self.__api_url is None:
                raise RuntimeError('No alive hostman servers found')

        return self.__api_url


class AbstractJugglerResolver(HttpResolver):
    """
    Common stuff for juggler resolvers.

    """
    api_url = "http://juggler.search.yandex.net:8998/api-slb/"


class AbstractWalleResolver(AbstractResolver):
    """
    Common stuff for wall-e resolvers.

    """
    api_url = None  # default (prod)

    def __init__(self, *args, api_url=None, walle_client=None, **kwargs):
        super().__init__(*args, **kwargs)

        if api_url is not None:
            self.api_url = api_url

        if walle_client is None:
            self.walle_client = walle_api.WalleClient(url=self.api_url)
        else:
            self.walle_client = walle_client

    def get_cache_key(self, query):
        return query.__repr__()


class HostmanUnitsResolver(AbstractHostmanResolver):
    defaults_filename = '.reconf-juggler.defaults'

    def get_defaults(self, archive, dirname):
        """ Collect and merge defaults descending to the root dir. """
        if dirname not in self._opts_cache:
            if dirname:
                opts = self.get_defaults(archive, os.path.dirname(dirname))
            else:
                opts = {}

            filename = os.path.join(dirname, self.defaults_filename)

            try:
                opts.update(yaml.safe_load(
                    archive.read(filename).decode('utf8')))
            except KeyError:  # no such file
                pass
            except Exception as e:
                logging.error('Failed to load %s: %s', filename, e)

            self._opts_cache[dirname] = opts

        return self._opts_cache[dirname].copy()

    def resolve_query(self, query):
        """
        Yield (filename, spec, monitoring_opts, err) for each hosman unit file.

        """
        reply = self.http_session.get(self.api_url + 'salt/zip')
        reply.raise_for_status()

        archive = zipfile.ZipFile(io.BytesIO(reply.content))
        self._opts_cache = {}

        for filename in archive.namelist():
            if os.path.splitext(filename)[-1].lower() not in {'.yaml', '.yml'}:
                continue

            dirname = os.path.dirname(filename)
            if os.path.basename(dirname) not in {'units.d', 'porto-daemons.d'}:
                continue

            opts = self.get_defaults(archive, dirname)
            try:
                spec = archive.read(filename).decode('utf8')
                spec = tuple(yaml.safe_load_all(spec))[-1]

                # opts stored in annotations as nested YAML (HOSTMAN-777)
                try:
                    opts.update(yaml.safe_load(
                        spec['meta']['annotations']['reconf-juggler']))
                except KeyError:
                    pass
            except Exception as e:
                yield filename, None, None, str(e)
                continue

            yield filename, spec, opts, None


class HostmanResolver(AbstractHostmanResolver):
    """
    Hostman meta resolver.

    """
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self['units'] = HostmanUnitsResolver(http_session=self.http_session)


class JugglerAggregatesResolver(AbstractJugglerResolver):
    """
    Juggler aggregates resolver.

    Return Juggler aggregates hierarchy.

    We do not use juggler sdk here because:
    a) JugglerApi.get_checks method did not provide info about children types,
       this leads to enormous amount of JugglerApi.get_checks calls (for each
       child actually) to determine endpoint it or subaggregate. See
       https://st.yandex-team.ru/JUGGLER-1290 for details.

    b) JugglerApi (at least now) has no retries for network and server errors.

    """
    def _get_aggregates(self, object_names=None, service_names=None, tags=None,
                        namespace_name=None, recursive=True):
        reply = self.http_session.get(
            self.api_url + 'checks/checks?do=1&format=json',
            params={
                'namespace_name': namespace_name,
                'host_name': object_names,
                'service_name': service_names,
                'tag_name': tags,
                'include_children': False,
                'include_notifications': True,
                'include_meta': True,
                'recursive_namespaces': True,
            },
        )
        reply.raise_for_status()

        aggregates = {}
        for object_name, object_body in reply.json().items():
            for service_name, aggregate_body in object_body.items():
                aggregates[object_name + ':' + service_name] = aggregate_body
                children = self._get_children(object_name, service_name,
                                              recursive=recursive)
                if children:
                    aggregate_body['children'] = children

        return aggregates

    def _get_children(self, object_name, service_name, recursive=True):
        reply = self.http_session.get(
            self.api_url + 'checks/children_tree?do=1&format=json',
            params={
                'host_name': object_name,
                'service_name': service_name,
                'expand_flag': False,
            },
        )
        reply.raise_for_status()

        children = reply.json()
        for name, body in children.items():
            if body.__class__ is dict:  # aggregate, details in st/JUGGLER-1290
                body.clear()  # drop api provided subtree (useless)
                if recursive:
                    object_name, service_name = name.split(':', 1)
                    body.update(self._get_aggregates(
                        object_names=object_name,
                        service_names=service_name,
                        recursive=recursive,
                    ).popitem()[1])

        return children

    def resolve_query(self, query):
        return self._get_aggregates(**query)

    def get_cache_key(self, query):
        return query.__repr__()


class JugglerInstancesResolver(AbstractJugglerResolver):
    """
    Juggler instances resolver.

    """
    def resolve_query(self, query):
        """
        Return list of instances for provided filter.

        """
        group_type, uri = query.split('%')

        reply = self.http_session.get(
            self.api_url + "proxy/groups/expand?do=1&format=json",
            params={'group_type': group_type, 'uri': uri},
        )
        reply.raise_for_status()

        return list(reply.json().keys())


class JugglerInstancesCountResolver(JugglerInstancesResolver):
    """
    Juggler instances count resolver.

    """
    def resolve_query(self, query):
        """
        Return count of instances for provided filter.

        """
        return len(super().resolve_query(query))


class JugglerRawEventsResolver(AbstractJugglerResolver):
    """
    Juggler raw events resolver.

    """
    api_url = 'http://juggler-api.search.yandex.net/v2/'
    limit = 200  # max allowed at this moment

    def _get_raw_events(self, query):
        reply = self.http_session.post(
            self.api_url + 'events/get_raw_events',
            json=query,
        )
        reply.raise_for_status()

        return reply.json()

    def resolve_query(self, query):
        """
        Yield raw events for provided filter.

        All events returned when no limit provided in query.

        """
        if 'limit' in query:
            return self._get_raw_events(query)['items']

        query = query.copy()
        query['limit'] = self.limit
        query.setdefault('offset', 0)

        while True:
            reply = self._get_raw_events(query)
            for event in reply['items']:
                yield event

            query['offset'] += self.limit
            if query['offset'] > reply['total']:
                break

    def get_cache_key(self, query):
        return query.__repr__()


class JugglerResolver(AbstractJugglerResolver):
    """
    Juggler meta resolver.

    """
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        self['aggregates'] = JugglerAggregatesResolver(
            api_url=self.api_url,
            http_session=self.http_session,
        )
        self['instances'] = JugglerInstancesResolver(
            api_url=self.api_url,
            http_session=self.http_session
        )
        self['instances_count'] = JugglerInstancesCountResolver(
            api_url=self.api_url,
            http_session=self.http_session,
        )
        self['raw_events'] = JugglerRawEventsResolver(
            http_session=self.http_session,
        )


class WalleHostsResolver(AbstractWalleResolver):
    """
    Return hosts generator for specified request.

    `query` is a dict with opts, passed directly as kwargs to
    `walle_api.walle_client.iter_hosts()`

    """
    def resolve_query(self, query):
        return self.walle_client.iter_hosts(**query)


class WalleHostsCountResolver(WalleHostsResolver):
    """
    Return hosts count for specified request.

    `query` is a dict with opts, passed directly as kwargs to
    `walle_api.walle_client.iter_hosts()`

    """
    def resolve_query(self, query):
        return len(tuple(super().resolve_query(query)))


class WalleProjectsResolver(AbstractWalleResolver):
    """
    Return projects list for specified request.

    `query` is a dict with opts, passed directly as kwargs to
    `walle_api.walle_client.get_projects()`

    """
    def resolve_query(self, query):
        return self.walle_client.get_projects(**query)['result']


class WalleResolver(AbstractWalleResolver):
    """
    Wall-e meta resolver.

    """
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        self['projects'] = WalleProjectsResolver(
            walle_client=self.walle_client
        )
        self['hosts'] = WalleHostsResolver(
            walle_client=self.walle_client,
        )
        self['hosts_count'] = WalleHostsCountResolver(
            walle_client=self.walle_client,
        )


class RootResolver(AbstractResolver):
    """
    Main resolver.

    """
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self['hostman'] = HostmanResolver()
        self['juggler'] = JugglerResolver()
        self['walle'] = WalleResolver()
