"""
ReConf-Juggler command-line structures converter.

"""
import argparse
import json
import logging
import sys

import infra.reconf_juggler.util.d3js as d3js
from infra.reconf_juggler.util.jsdk import jsdk2tree, tree2jsdk

from . import html


FMT_MAP = {
    'jsdk': {
        'tree': jsdk2tree,
    },
    'tree': {
        'd3js': d3js.convert,
        'html': html.tree2html,
        'jsdk': tree2jsdk,
    },
}

IFMTS = sorted(FMT_MAP.keys())
OFMTS = []
for i in FMT_MAP.values():
    OFMTS.extend(i.keys())
OFMTS.sort()


def convert(data, ifmt, ofmt):
    if ifmt in FMT_MAP and ofmt in FMT_MAP[ifmt]:
        return FMT_MAP[ifmt][ofmt](data)

    raise RuntimeError("Unable to convert {} to {}".format(ifmt, ofmt))


def main():
    logging.basicConfig(
        format='[%(asctime)s] %(levelname)s %(message)s',
        level=logging.DEBUG
    )

    argparser = argparse.ArgumentParser(description=__doc__)
    argparser.add_argument('--ifmt', required=True, type=str, choices=IFMTS)
    argparser.add_argument('--ofmt', required=True, type=str, choices=OFMTS)
    argparser.add_argument(
        '--ofile',
        default=sys.stdout,
        type=argparse.FileType('w'),
        help='output file; STDOUT used by default',
    )
    argparser.add_argument(
        'file',
        default=sys.stdout,
        type=argparse.FileType('r'),
        help='file with data to convert',
    )
    args = argparser.parse_args()

    converted = convert(json.load(args.file), args.ifmt, args.ofmt)

    if isinstance(converted, str):
        print(converted, file=args.ofile)
    else:
        json.dump(converted, args.ofile, indent=3, sort_keys=True)
