"""
ReConf-Juggler aggregates diff tool.

"""
import argparse
import json
import logging
import os
import psutil
import sys

import nested_diff.diff_tool

import infra.reconf
import infra.reconf.util.diff as diffs
import infra.reconf_juggler

from infra.reconf_juggler.util.jsdk import jsdk2tree


class Preprocessor(object):
    def __init__(self, **kwargs):
        tasks_defaults = {
            'trim_default_opts': False,
            'trim_mark_tags': False,
        }

        self.__tasks = []
        for task in kwargs:
            enabled = kwargs.get(task, tasks_defaults[task])
            if enabled:
                self.__tasks.append(task)

        if 'trim_default_opts' in self.__tasks:
            self._default_opts = infra.reconf_juggler.Check()
            self._default_opts.build()

    def process(self, checks):
        """
        Run desired methods over each check in the tree.

        """
        if not self.__tasks:
            return

        for name, body in infra.reconf.iterate_depth_first(checks):
            for task in self.__tasks:
                self.__getattribute__(task)(name, body)

    def trim_default_opts(self, name, body):
        """
        juggler-sdk doesn't dump check opt if it's value is default.
        Drop such options to make dumps even.

        """
        for key in tuple(body):
            try:
                if body[key] == self._default_opts[key]:
                    logging.debug('Drop default value for {} from {}'.format(
                        key, name))
                    del body[key]
            except KeyError:
                pass

    def trim_mark_tags(self, name, body):
        """
        juggler-sdk pollutes checks adding 'a_mark_{}' tag to the tags list.
        Filter out such tags to make dumps even.

        """
        if 'tags' not in body or body['tags'] is None:
            return

        body['tags'] = list(
            filter(lambda x: not x.startswith('a_mark_'), body['tags']))


def set_loglevel(level_name):
    logging.getLogger().setLevel(level_name)

    return level_name


def svn_diff():
    """
    Human friendly diff tool for large JSON files controlled by SVN.

    Usage:
    `svn diff --diff-cmd=jdiff filename1 [filename2 [... filenameN]]`

    """
    return nested_diff.diff_tool.App(
        args=['--ifmt', 'json', sys.argv[-2], sys.argv[-1]]
    ).run()


def main():
    if psutil.Process(pid=os.getppid()).name() == 'svn':
        sys.exit(svn_diff())

    logging.basicConfig(
        format='[%(asctime)s] %(levelname)s %(message)s',
        level=logging.DEBUG
    )

    argparser = argparse.ArgumentParser(description=__doc__)
    argparser.add_argument(
        '--ifmt',
        choices=('auto', 'jsdk', 'tree'),
        default='auto',
        help='input format; "auto" used by default',
        type=str,
    )
    argparser.add_argument(
        '--ignore-default-opts',
        action='store_true'
    )
    argparser.add_argument(
        '--ignore-mark-tags',
        action='store_true'
    )
    argparser.add_argument(
        '--loglevel',
        choices=('CRITICAL', 'ERROR', 'WARNING', 'INFO', 'DEBUG'),
        default='INFO',
        type=set_loglevel,
    )
    argparser.add_argument(
        '--ofile',
        default=sys.stdout,
        type=argparse.FileType('w'),
        help='output file; STDOUT used by default',
    )
    argparser.add_argument(
        '--ofmt',
        choices=('unified', 'nested'),
        default='nested',
        help='output format; "nested" used by default',
        type=str,
    )

    argparser.add_argument('file1', type=argparse.FileType('r'))
    argparser.add_argument('file2', type=argparse.FileType('r'))

    args = argparser.parse_args()

    old = json.load(args.file1)
    new = json.load(args.file2)

    args.file1.close()
    args.file2.close()

    if args.ifmt == 'auto':
        if isinstance(old, list):
            old = jsdk2tree(old)
        if isinstance(new, list):
            new = jsdk2tree(new)
    elif args.ifmt == 'jsdk':
        old = jsdk2tree(old)
        new = jsdk2tree(new)

    pproc = Preprocessor(
        trim_default_opts=args.ignore_default_opts,
        trim_mark_tags=args.ignore_mark_tags,
    )
    pproc.process(old)
    pproc.process(new)

    if args.ofmt == 'unified':
        diff, equal = diffs.unified(old, new, context=20)
    else:
        diff, equal = diffs.nested(old, new, term_colors=args.ofile.isatty())

    args.ofile.write(diff)
    sys.exit(0 if equal else 8)
